package com.instabug.apm.cache.handler.networklog;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;

import com.instabug.apm.cache.handler.session.SessionMetaDataCacheHandler;
import com.instabug.apm.configuration.APMConfigurationProvider;
import com.instabug.apm.di.ServiceLocator;
import com.instabug.apm.logger.internal.Logger;
import com.instabug.apm.model.APMNetworkLog;
import com.instabug.library.model.common.Session;

import java.util.List;

public class NetworkLogMigrationHandlerImpl implements NetworkLogMigrationHandler {

    @NonNull
    private final NetworkLogCacheHandler cacheHandler;
    @NonNull
    private final DanglingNetworkLogCacheHandler danglingCacheHandler;
    @NonNull
    private final APMConfigurationProvider configurationProvider;
    @Nullable
    private final SessionMetaDataCacheHandler sessionMetaDataCacheHandler;
    private Logger apmLogger = ServiceLocator.getApmLogger();

    public NetworkLogMigrationHandlerImpl(@NonNull NetworkLogCacheHandler cacheHandler,
                                          @NonNull DanglingNetworkLogCacheHandler danglingCacheHandler,
                                          @NonNull APMConfigurationProvider configurationProvider,
                                          @Nullable SessionMetaDataCacheHandler sessionMetaDataCacheHandler) {
        this.cacheHandler = cacheHandler;
        this.danglingCacheHandler = danglingCacheHandler;
        this.configurationProvider = configurationProvider;
        this.sessionMetaDataCacheHandler = sessionMetaDataCacheHandler;
    }

    @Override
    public void migrate(@NonNull final Session runningSession, @NonNull final Session lastSession) {
        long limit = configurationProvider.getNetworkLogsRequestLimit();
        List<APMNetworkLog> batch;
        do {
            batch = getBatch(limit);
            if (batch != null) {
                for (APMNetworkLog trace : batch) {
                    if (shouldLinkToLastSession(trace)) migrate(trace, lastSession);
                    else migrate(trace, runningSession);
                }
                deleteBatch(batch);
            }
        }
        while (batch != null && batch.size() > 0);

    }

    @Nullable
    @VisibleForTesting
    List<APMNetworkLog> getBatch(long limit) {
        return danglingCacheHandler.getNetworkLogsByLimit(limit);
    }

    @VisibleForTesting
    void migrate(@NonNull final APMNetworkLog networkLog, @NonNull final Session session) {
        if (sessionMetaDataCacheHandler != null) {
            long id = cacheHandler.insertNetworkLog(session.getId(), networkLog);
            if (id != -1) {
                // migrate dangling attributes
                String traceName = "[" + networkLog.getMethod() + "] " + networkLog.getUrl();
                cacheHandler.addAttributes(id, traceName, danglingCacheHandler.getTraceAttributes(networkLog.getId()));
            }
            apmLogger.d("Migrated network request: " + networkLog.getUrl());
            if (id > 0) {
                sessionMetaDataCacheHandler.addToNetworkLogsTotalCount(session.getId(), 1);
                int droppedCount = cacheHandler.trimToLimit(session.getId(), configurationProvider.getNetworkLogsRequestLimit());
                if (droppedCount > 0) {
                    sessionMetaDataCacheHandler.addToNetworkLogsDroppedCount(session.getId(), droppedCount);
                }
                cacheHandler.trimToLimit(configurationProvider.getNetworkLogsCacheLimit());
            }
        }
    }

    @VisibleForTesting
    void deleteBatch(@NonNull List<APMNetworkLog> batch) {
        danglingCacheHandler.deleteNetworkLogsByLimit(batch.size());
    }

    @VisibleForTesting
    boolean shouldLinkToLastSession(@NonNull APMNetworkLog networkLog) {
        return !networkLog.getExecutedInBackground();
    }
}
