package com.instabug.apm.util.powermanagement;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.os.Build;
import android.os.PowerManager;

import androidx.annotation.NonNull;
import androidx.annotation.RequiresApi;

import com.instabug.apm.di.ServiceLocator;
import com.instabug.apm.logger.internal.Logger;
import com.instabug.library.core.InstabugCore;

import java.util.ArrayList;
import java.util.List;

/**
 * Receives and pass the changes of the power save mode changes
 * Send a {@link PowerManagementCallback} if the power save mode state was changed
 *
 * @see PowerManager#ACTION_POWER_SAVE_MODE_CHANGED
 */

public class PowerSaveModeBroadcast extends BroadcastReceiver {

    private final Logger apmLogger = ServiceLocator.getApmLogger();
    @NonNull
    private final List<PowerManagementCallback> callbacks;
    @NonNull
    private final Context appContext;

    public PowerSaveModeBroadcast(@NonNull Context appContext) {
        this.callbacks = new ArrayList<>();
        this.appContext = appContext;
    }

    @Override
    public synchronized void onReceive(Context context, Intent intent) {
        PowerManager pm = (PowerManager) context.getApplicationContext().getSystemService(Context.POWER_SERVICE);
        if (pm != null) {
            for (PowerManagementCallback callback : callbacks)
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                    callback.onPowerSaveModeChanged(pm.isPowerSaveMode());
                }
        }
    }

    @RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
    public synchronized void register(@NonNull PowerManagementCallback callback) {
            try {
                if (callbacks.size() > 0) {
                    apmLogger.logSDKProtected("PowerSaveModeBroadcast is already registered. Skipping re-registering");
                } else {
                    appContext.registerReceiver(this, new IntentFilter(PowerManager.ACTION_POWER_SAVE_MODE_CHANGED));
                    apmLogger.logSDKProtected("PowerSaveModeReceiver registered on " + callback.getClass().getSimpleName());
                }
                callbacks.add(callback);
            } catch (Exception e) {
                apmLogger.logSDKProtected("Error: " + e.getMessage() + "While registering power saver mode receiver");
            }

    }


    @RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
    public synchronized void unregister(@NonNull PowerManagementCallback callback) {
        try {
            if (callbacks.remove(callback) && callbacks.isEmpty()) {
                appContext.unregisterReceiver(this);
                apmLogger.logSDKProtected("PowerSaveModeReceiver unregistered from " + callback.getClass().getSimpleName());
            }
        } catch (Exception e) {
            InstabugCore.reportError(e, "unregister PowerSaveModeBroadcast got error");
        }
    }
}
