package com.instabug.apm.uitrace.uihangs

import android.os.Build
import android.view.Choreographer.FrameCallback
import androidx.annotation.RequiresApi
import com.instabug.apm.cache.model.UiHangModel
import com.instabug.apm.configuration.APMConfigurationProvider
import com.instabug.apm.uitrace.di.UiTracesServiceLocator
import com.instabug.library.factory.Factory

interface UiHangHandler {
    /**
     * @return the collected UiHangModel
     * THe collected UiHangModel is only reset when starting listening to UiHangs
     */

    val uiHangModel: UiHangModel?

    /**
     * Start listening to UI hangs, also reset the current UiHangModel
     */
    fun start()

    /**
     * Stop listening to UI hangs
     */
    fun stop()

    /**
     * Clear UI Hang model
     */
    fun clearUiHangModel()
}

@RequiresApi(Build.VERSION_CODES.JELLY_BEAN)
class UiHangHandlerImpl(
    private val configurationProvider: APMConfigurationProvider,
    private val choreographer: APMChoreographer,
    private val frameDropsCalculator: FrameDropsCalculator
) : UiHangHandler, FrameCallback {
    private var _uiHangModel: UiHangModel? = null
    private var isRegistered = false
    private var uiTracesLargeDropThreshold = Float.MAX_VALUE
    private var uiTracesSmallDropThreshold = Float.MAX_VALUE

    private val canStart: Boolean
        get() = !isRegistered && configurationProvider.isAutoUiHangsEnabled()
    override val uiHangModel: UiHangModel?
        get() = _uiHangModel

    override fun start() {
        if (canStart) {
            uiTracesLargeDropThreshold = configurationProvider.uiTraceLargeDropThreshold
            uiTracesSmallDropThreshold = configurationProvider.uiTraceSmallDropThreshold
            isRegistered = true
            frameDropsCalculator.reset()
            _uiHangModel = UiHangModel()
            choreographer.addCallback(this)
        }
    }

    override fun stop() {
        if (isRegistered) {
            isRegistered = false
            choreographer.removeCallback(this)
        }
    }

    override fun clearUiHangModel() {
        _uiHangModel = null
    }

    override fun doFrame(frameTimeNanos: Long) {
        frameDropsCalculator.calculateFrameDuration(frameTimeNanos, uiTracesSmallDropThreshold)?.also {
            _uiHangModel?.incrementSmallDropsDuration(it)
            incrementLargeDropDurationIfPossible(it)
        }
    }

    private fun incrementLargeDropDurationIfPossible(frameRenderingDurationMicros: Long) =
        _uiHangModel
            ?.takeIf { frameRenderingDurationMicros > uiTracesLargeDropThreshold }
            ?.apply { incrementLargeDropsDuration(frameRenderingDurationMicros) }
}

@RequiresApi(Build.VERSION_CODES.JELLY_BEAN)
class UiHangsHandlerFactory : Factory<UiHangHandler> {
    override fun create(): UiHangHandler = UiTracesServiceLocator.getUiHangHandler()
}
