package com.instabug.apm.uitrace.di

import android.annotation.SuppressLint
import android.os.Build
import android.os.Handler
import android.os.Looper
import androidx.annotation.RequiresApi
import com.instabug.apm.di.ContextProvider
import com.instabug.apm.di.ServiceLocator
import com.instabug.apm.di.getOrCreateSingleInstance
import com.instabug.apm.di.getOrCreateSingleton
import com.instabug.apm.uitrace.UiTracePowerManagementCallbackFactory
import com.instabug.apm.uitrace.UiTraceWrapper
import com.instabug.apm.uitrace.UiTraceWrapperFactory
import com.instabug.apm.uitrace.WebViewTraceListenerFactoryUiTraceDecorator
import com.instabug.apm.uitrace.activitycallbacks.APMUiTraceActivityCallbacks
import com.instabug.apm.uitrace.activitycallbacks.CompositeApmUiTraceActivityCallbacks
import com.instabug.apm.uitrace.activitycallbacks.EmptyAPMUiTraceActivityCallbacks
import com.instabug.apm.uitrace.handler.CPAutomaticUiTracesHandlerImpl
import com.instabug.apm.uitrace.handler.NativeAutomaticUiTraceHandler
import com.instabug.apm.uitrace.handler.UiTraceHandler
import com.instabug.apm.uitrace.handler.UiTraceHandlerImpl
import com.instabug.apm.uitrace.handler.UiTraceWrapperHandlerImpl
import com.instabug.apm.uitrace.manager.UiTracesManager
import com.instabug.apm.uitrace.manager.UiTracesManagerImpl
import com.instabug.apm.uitrace.repo.UiTracesRepo
import com.instabug.apm.uitrace.repo.UiTracesRepoImpl
import com.instabug.apm.uitrace.uihangs.APMChoreographer
import com.instabug.apm.uitrace.uihangs.APMChoreographerImpl
import com.instabug.apm.uitrace.uihangs.FrameDropsCalculatorImpl
import com.instabug.apm.uitrace.uihangs.UiHangHandler
import com.instabug.apm.uitrace.uihangs.UiHangHandlerImpl
import com.instabug.apm.uitrace.uihangs.UiHangsHandlerFactory
import com.instabug.apm.webview.webview_trace.handler.WebViewTraceEventListener
import com.instabug.library.BuildFieldsProvider
import com.instabug.library.Platform
import com.instabug.library.factory.ParameterizedFactory
import com.instabug.library.settings.SettingsManager
import com.instabug.library.tracking.InstabugInternalTrackingDelegate
import com.instabug.library.util.threading.PoolProvider
import java.lang.ref.WeakReference
import java.util.concurrent.Executor

object UiTracesServiceLocator {

    private const val UI_TRACES_ORDERED_EXECUTOR_ID = "ui-traces-ordered-executor"

    @Volatile
    private var _uiTracesRepo: UiTracesRepo? = null

    @Volatile
    private var _compositeApmUiTraceActivityCallbacks: CompositeApmUiTraceActivityCallbacks? = null

    @Volatile
    private var _uiTracesManager: UiTracesManager? = null

    @Volatile
    private var _APMChoreographer: APMChoreographer? = null

    private var handlerWeakReference: WeakReference<UiTraceHandler?>? = null
    private var wrapperFactoryWeakReference: WeakReference<ParameterizedFactory<UiTraceWrapper, Long>?>? =
        null

    private val isBelowJellybean
        get() = BuildFieldsProvider.provideBuildVersion() < Build.VERSION_CODES.JELLY_BEAN

    val handler: UiTraceHandler?
        get() = getOrCreateSingleInstance(
            ::handlerWeakReference,
            { handlerWeakReference = it },
            ::createUiTraceHandler
        )

    val wrapperFactory: ParameterizedFactory<UiTraceWrapper, Long>?
        @RequiresApi(Build.VERSION_CODES.JELLY_BEAN)
        get() = getOrCreateSingleInstance(
            ::wrapperFactoryWeakReference,
            { wrapperFactoryWeakReference = it },
            ::createUiTraceWrapperFactory
        )

    private val uiTracesRepo: UiTracesRepo
        @RequiresApi(Build.VERSION_CODES.JELLY_BEAN)
        get() = getOrCreateSingleton(
            ::_uiTracesRepo,
            { _uiTracesRepo = it },
            ::createUiTracesRepo
        )

    @JvmStatic
    val uiTracesExecutor: Executor by lazy {
        PoolProvider.getSingleThreadExecutor(UI_TRACES_ORDERED_EXECUTOR_ID)
    }

    @JvmStatic
    val manager: UiTracesManager?
        @SuppressLint("NewApi")
        get() = getOrCreateSingleton(
            ::_uiTracesManager,
            { _uiTracesManager = it },
            { if (isBelowJellybean) null else createUiTracesManager() }
        )

    @JvmStatic
    val apmUiTraceActivityCallbacks: APMUiTraceActivityCallbacks
        get() = if (isBelowJellybean) EmptyAPMUiTraceActivityCallbacks() else compositeApmUiTraceActivityCallbacks

    val compositeApmUiTraceActivityCallbacks: CompositeApmUiTraceActivityCallbacks
        get() = getOrCreateSingleton(
            ::_compositeApmUiTraceActivityCallbacks,
            { _compositeApmUiTraceActivityCallbacks = it },
            ::CompositeApmUiTraceActivityCallbacks
        )

    val apmChoreographer: APMChoreographer
        @RequiresApi(Build.VERSION_CODES.JELLY_BEAN)
        get() = getOrCreateSingleton(
            ::_APMChoreographer,
            { _APMChoreographer = it },
            ::createAPMChoreographer
        )

    @RequiresApi(Build.VERSION_CODES.JELLY_BEAN)
    fun getUiHangHandler(): UiHangHandler = UiHangHandlerImpl(
        ServiceLocator.getApmConfigurationProvider(),
        apmChoreographer,
        FrameDropsCalculatorImpl()
    )

    private fun createUiTraceHandler(): UiTraceHandler? = ServiceLocator
        .getSessionMetaDataCacheHandler()?.let { metadataCacheHandler ->
            UiTraceHandlerImpl(
                ServiceLocator.getUiTraceCacheHandler(),
                metadataCacheHandler,
                ServiceLocator.getApmConfigurationProvider(),
                ServiceLocator.getApmLogger()
            )
        }

    @RequiresApi(Build.VERSION_CODES.JELLY_BEAN)
    private fun createUiTraceWrapperFactory(): ParameterizedFactory<UiTraceWrapper, Long> =
        UiTraceWrapperFactory(
            createWebViewTraceListenerFactory(),
            UiTracePowerManagementCallbackFactory(),
            UiHangsHandlerFactory()
        )

    private fun createWebViewTraceListenerFactory(): ParameterizedFactory<WebViewTraceEventListener?, Long>? {
        val currentPlatform = SettingsManager
            .getInstance()
            .getEarlyCurrentPlatform(ServiceLocator.getContext())
        return if (currentPlatform == Platform.ANDROID) WebViewTraceListenerFactoryUiTraceDecorator(
            ServiceLocator.getWebViewTraceEventListenerFactory(),
            WebViewTraceConfigsProvider(),
            WebViewTraceManagerProvider()
        )
        else null
    }

    @RequiresApi(Build.VERSION_CODES.JELLY_BEAN)
    private fun createUiTracesRepo(): UiTracesRepo = UiTracesRepoImpl(
        UiTraceHandlerProvider(),
        UiTraceWrapperFactoryProvider(),
        ServiceLocator.getApmLogger(),
        createUiTracesWrapperHandler()
    )

    @RequiresApi(Build.VERSION_CODES.JELLY_BEAN)
    private fun createUiTracesManager() = UiTracesManagerImpl(
        compositeApmUiTraceActivityCallbacks,
        NativeAutomaticUiTraceHandlerProvider(),
        CPAutomaticUiTraceHandlerProvider(),
        CustomUiTraceHandlerActivityCallbacksProvider(),
        ServiceLocator.getApmConfigurationProvider(),
        uiTracesExecutor,
        uiTracesRepo,
        InstabugInternalTrackingDelegate.getInstance(),
        ServiceLocator.getDeviceStateProvider(),
        SettingsManager.getInstance(),
        ServiceLocator.getApmLogger(),
        ContextProvider()
    )

    @RequiresApi(Build.VERSION_CODES.JELLY_BEAN)
    fun createNativeAutomaticUiTraceHandler(): APMUiTraceActivityCallbacks =
        NativeAutomaticUiTraceHandler(
            uiTracesExecutor,
            uiTracesRepo,
            ServiceLocator.getApmConfigurationProvider(),
            ServiceLocator.getApmLogger(),
            ServiceLocator.getDeviceStateProvider()
        )

    @RequiresApi(Build.VERSION_CODES.JELLY_BEAN)
    fun createCPAutomaticUiTraceHandler(): CPAutomaticUiTracesHandlerImpl =
        CPAutomaticUiTracesHandlerImpl(
            InstabugInternalTrackingDelegate.getInstance(),
            uiTracesExecutor,
            ServiceLocator.getApmConfigurationProvider(),
            uiTracesRepo,
            ServiceLocator.getApmLogger(),
            ServiceLocator.getDeviceStateProvider()
        )

    @RequiresApi(Build.VERSION_CODES.JELLY_BEAN)
    private fun createUiTracesWrapperHandler() = UiTraceWrapperHandlerImpl(
        BatteryLevelChangeBroadcastProvider(),
        PowerSaveModeBroadcastProvider(),
        WebViewTraceManagerProvider(),
        ServiceLocator.getSessionHandler()
    )

    @RequiresApi(Build.VERSION_CODES.JELLY_BEAN)
    private fun createAPMChoreographer() = APMChoreographerImpl(
        // To avoid waiting on poolProvider.getInstance() lock
        Handler(Looper.getMainLooper()),
        uiTracesExecutor,
        ServiceLocator.getApmLogger()
    )
}