package com.instabug.apm.screenloading.handler

import android.app.Activity
import androidx.annotation.WorkerThread
import com.instabug.apm.cache.handler.uitrace.UiLoadingMetricCacheHandler
import com.instabug.apm.cache.model.UiLoadingModel
import com.instabug.apm.configuration.APMConfigurationProvider
import com.instabug.apm.constants.UiLoadingMetric.STAGE_CUSTOM_END_LOADING_DURATION_MICRO
import com.instabug.apm.constants.UiLoadingMetric.STAGE_CUSTOM_END_LOADING_TIMESTAMP_MICRO
import com.instabug.apm.logger.internal.Logger
import com.instabug.apm.model.EventTimeMetricCapture
import com.instabug.apm.uitrace.activitycallbacks.APMUiTraceActivityCallbacks
import com.instabug.apm.util.runOrReportAPMError
import java.util.concurrent.Executor

interface CPScreenLoadingHandler {
    fun reportScreenLoading(startTimeStampMicro: Long, durationMicro: Long, uiTraceId: Long)
    fun endScreenLoading(timeStampMicro: Long, uiTraceId: Long)

    @WorkerThread
    fun cacheSynchronous()

    fun cacheCurrentTrace()
}

class CPScreenLoadingHandlerImpl(
    private val configurationProvider: APMConfigurationProvider,
    private val cacheHandler: UiLoadingMetricCacheHandler,
    private val executor: Executor,
    private val logger: Logger
) : CPScreenLoadingHandler, APMUiTraceActivityCallbacks {

    private var currentUiTraceId: Long? = null
    private var currentScreenLoadingModel: UiLoadingModel? = null

    override fun reportScreenLoading(
        startTimeStampMicro: Long,
        durationMicro: Long,
        uiTraceId: Long
    ) = doScreenLoadingOperation(
        "reportScreenLoading",
        "error while reporting ScreenLoading"
    ) { handleScreenLoading(startTimeStampMicro, durationMicro, uiTraceId) }

    override fun endScreenLoading(timeStampMicro: Long, uiTraceId: Long) =
        runOrReportAPMError("error while reporting endScreenLoading", logger) {
            currentScreenLoadingModel?.let {
                when {
                    uiTraceId != currentUiTraceId -> logger.logSDKProtected("endScreenLoading not reported because of invalid uiTraceId")
                    timeStampMicro <= 0 -> it.handleEarlyScreenLoading()
                    else -> it.handleEndScreenLoading(timeStampMicro)
                }
            }
        }

    private fun UiLoadingModel.handleEndScreenLoading(timeStampMicro: Long) {
        val endScreenLoadingStartTimeStamp = startTimeStampMicro + durationInMicro
        val endScreenLoadingDuration = timeStampMicro - endScreenLoadingStartTimeStamp
        durationInMicro = timeStampMicro - startTimeStampMicro
        stages = mapOf(
            STAGE_CUSTOM_END_LOADING_TIMESTAMP_MICRO to endScreenLoadingStartTimeStamp,
            STAGE_CUSTOM_END_LOADING_DURATION_MICRO to endScreenLoadingDuration
        )
    }

    private fun UiLoadingModel.handleEarlyScreenLoading() {
        stages = mapOf(STAGE_CUSTOM_END_LOADING_DURATION_MICRO to 0)
    }

    override fun onActivityPaused(activity: Activity, timeMetric: EventTimeMetricCapture) =
        doScreenLoadingOperation(
            "onActivityPaused",
            "error while reporting ScreenLoading"
        ) { cacheUiLoadingModel() }

    @WorkerThread
    override fun cacheSynchronous() = cacheUiLoadingModel()

    override fun cacheCurrentTrace() = doScreenLoadingOperation(
        "cacheCurrentTrace",
        "Failed to cache current screenLoading trace"
    ) { cacheUiLoadingModel() }

    private inline fun doScreenLoadingOperation(
        method: String,
        @Suppress("SameParameterValue")
        errorMessage: String,
        crossinline operation: () -> Unit
    ) = executor.execute {
        runOrReportAPMError(errorMessage, logger) {
            if (configurationProvider.isAutoUiLoadingMetricsFullyEnabled) operation()
            else logger.logSDKProtected("CPScreenLoadingHandler.$method was not called because feature is disabled")
        }
    }

    private fun handleScreenLoading(
        startTimeStampMicro: Long,
        durationMicro: Long,
        uiTraceId: Long
    ) {
        cacheUiLoadingModel()
        createUiLoadingModel(startTimeStampMicro, durationMicro, uiTraceId)
    }

    private fun cacheUiLoadingModel() {
        currentScreenLoadingModel?.let { uiLoadingModel ->
            currentUiTraceId?.let { uiTraceId ->
                cacheHandler.insert(uiLoadingModel, uiTraceId)
            }
        }
        currentUiTraceId = null
        currentScreenLoadingModel = null
    }

    private fun createUiLoadingModel(
        startTimeStampMicro: Long,
        durationMicro: Long,
        uiTraceId: Long
    ) {
        currentScreenLoadingModel = UiLoadingModel().apply {
            this.startTimeStampMicro = startTimeStampMicro
            this.durationInMicro = durationMicro
        }
        currentUiTraceId = uiTraceId
    }
}