package com.instabug.apm.screenloading

import com.instabug.apm.configuration.APMConfigurationProvider
import com.instabug.apm.configuration.FeatureAvailabilityChecker
import com.instabug.apm.logger.internal.Logger

class ScreenLoadingFeatureAvailabilityChecker(
    private val configurationProvider: APMConfigurationProvider,
    private val logger: Logger
) : FeatureAvailabilityChecker {
    override fun isAvailable(apiName: String) = when {
        !configurationProvider.isAPMFeatureAvailable ->
            returnDisabledAndLogError("$apiName $FEATURE_NOT_AVAILABLE")

        !configurationProvider.isAPMSdkEnabled ->
            returnDisabledAndLogError("$apiName $APM_SDK_DISABLED")

        !configurationProvider.isUiTraceSdkEnabled ->
            returnDisabledAndLogError("$apiName $UI_TRACE_SDK_DISABLED")

        !configurationProvider.isUiLoadingMetricsFeatureEnabled ->
            returnDisabledAndLogError("$apiName $FEATURE_NOT_AVAILABLE")

        !configurationProvider.isUiLoadingMetricsSdkEnabled ->
            returnDisabledAndLogError("$apiName $SCREEN_LOADING_SDK_DISABLED")

        else -> true
    }

    private fun returnDisabledAndLogError(message: String): Boolean {
        logger.logSDKError(message)
        return false
    }

    companion object {
        private const val FEATURE_NOT_AVAILABLE =
            "wasn’t called as the feature seems to be disabled for your company on Instabug. Please contact Instabug’s support team from your dashboard for more information."
        private const val APM_SDK_DISABLED =
            "wasn't called. Please make sure to enable APM first by following the instructions at this link: https://docs.instabug.com/reference#enable-or-disable-apm"
        private const val UI_TRACE_SDK_DISABLED =
            "wasn't called. Please make sure to enable Auto UI Traces first by following the instructions at this link: https://docs.instabug.com/reference/enable-or-disable-auto-ui-trace"
        private const val SCREEN_LOADING_SDK_DISABLED =
            "wasn't called. Please make sure to enable Screen Loading first by following the instructions at this link: https://docs.instabug.com/reference/enable-or-disable-screen-loading"
    }
}