package com.instabug.apm.lifecycle;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.os.Bundle;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.instabug.apm.configuration.APMConfigurationProvider;
import com.instabug.apm.di.ServiceLocator;
import com.instabug.apm.handler.session.APMSessionObserver;
import com.instabug.apm.handler.session.SessionObserverHandler;
import com.instabug.apm.model.EventTimeMetricCapture;
import com.instabug.apm.screenloading.di.ScreenLoadingServiceLocator;
import com.instabug.apm.screenloading.manager.ScreenLoadingManager;
import com.instabug.apm.uitrace.activitycallbacks.APMUiTraceActivityCallbacks;
import com.instabug.apm.uitrace.di.UiTracesServiceLocator;
import com.instabug.apm.uitrace.manager.UiTracesManager;
import com.instabug.library.model.common.Session;

public class ActivityCallbacks implements Application.ActivityLifecycleCallbacks, APMSessionObserver {

    private static boolean isRegistered = false;
    @NonNull
    private final AppLaunchLifeCycleCallbacks appLaunchLifeCycleCallbacks;
    @NonNull
    private final APMUiTraceActivityCallbacks uiTraceCallbacks;
    private int activitiesInForegroundCount = 0;

    @SuppressLint({"ERADICATE_FIELD_NOT_INITIALIZED", "NewApi"})
    public ActivityCallbacks(
            Context context,
            boolean isRegisteredBeforeFirstActivityLaunch
    ) {
        setRegistered();
        SessionObserverHandler.register(this);
        appLaunchLifeCycleCallbacks = ServiceLocator.getAppLaunchLifeCycleCallbacks(context,
                isRegisteredBeforeFirstActivityLaunch);
        uiTraceCallbacks = UiTracesServiceLocator.getApmUiTraceActivityCallbacks();
        startUiTraces();
        initScreenLoading();
    }

    private static void startUiTraces() {
        UiTracesManager uiTracesManager = UiTracesServiceLocator.getManager();
        if (uiTracesManager != null) uiTracesManager.startSynchronous();
    }

    private static APMConfigurationProvider getApmConfigurationProvider() {
        return ServiceLocator.getApmConfigurationProvider();
    }

    private static void initScreenLoading() {
        if (!getApmConfigurationProvider().isAutoUiLoadingMetricsFullyEnabled()) return;
        ScreenLoadingManager screenLoadingManager = ScreenLoadingServiceLocator.getManager();
        if (screenLoadingManager != null) screenLoadingManager.startSynchronous();
    }

    public ActivityCallbacks(Context context) {
        this(context, false);
    }

    public static boolean isRegistered() {
        return isRegistered;
    }

    private static void setRegistered() {
        ActivityCallbacks.isRegistered = true;
    }

    @Override
    public void onActivityPreCreated(@NonNull Activity activity, @Nullable Bundle savedInstanceState) {
        EventTimeMetricCapture timeMetric = new EventTimeMetricCapture();

        uiTraceCallbacks.onActivityPreCreated(activity, savedInstanceState, timeMetric, timeMetric.getTimeStampMillis());
    }

    @Override
    public void onActivityCreated(@NonNull Activity activity, @Nullable Bundle savedInstanceState) {
        EventTimeMetricCapture timeMetric = new EventTimeMetricCapture();
        uiTraceCallbacks.onActivityCreated(activity, savedInstanceState, timeMetric, timeMetric.getTimeStampMillis());
        appLaunchLifeCycleCallbacks.onActivityCreated(activity, timeMetric);
    }

    @Override
    public void onActivityPostCreated(@NonNull Activity activity, @Nullable Bundle savedInstanceState) {
        EventTimeMetricCapture timeMetric = new EventTimeMetricCapture();
        uiTraceCallbacks.onActivityPostCreated(activity, savedInstanceState, timeMetric);
    }

    @Override
    public void onActivityPreStarted(@NonNull Activity activity) {
        EventTimeMetricCapture timeMetric = new EventTimeMetricCapture();
        uiTraceCallbacks.onActivityPreStarted(activity, timeMetric);
    }

    @Override
    public void onActivityStarted(@NonNull Activity activity) {
        activitiesInForegroundCount++;
        EventTimeMetricCapture timeMetric = new EventTimeMetricCapture();
        uiTraceCallbacks.onActivityStarted(activity, timeMetric);
        appLaunchLifeCycleCallbacks.onActivityStarted(activity, timeMetric);
    }

    @Override
    public void onActivityPostStarted(@NonNull Activity activity) {
        EventTimeMetricCapture timeMetric = new EventTimeMetricCapture();
        uiTraceCallbacks.onActivityPostStarted(activity, timeMetric);
    }

    @Override
    public void onActivityPreResumed(@NonNull Activity activity) {
        EventTimeMetricCapture timeMetric = new EventTimeMetricCapture();
        uiTraceCallbacks.onActivityPreResumed(activity, timeMetric);
    }

    @Override
    public void onActivityResumed(@NonNull final Activity activity) {
        EventTimeMetricCapture timeMetric = new EventTimeMetricCapture();
        uiTraceCallbacks.onActivityResumed(activity, timeMetric, timeMetric.getTimeStampMillis());
        appLaunchLifeCycleCallbacks.onActivityResumed(activity, timeMetric);
    }

    @Override
    public void onActivityPostResumed(@NonNull Activity activity) {
        EventTimeMetricCapture timeMetric = new EventTimeMetricCapture();
        uiTraceCallbacks.onActivityPostResumed(activity, timeMetric);
    }

    @Override
    public void onActivityPaused(@NonNull Activity activity) {
        EventTimeMetricCapture timeMetric = new EventTimeMetricCapture();
        uiTraceCallbacks.onActivityPaused(activity, timeMetric);
    }

    @Override
    public void onActivityStopped(@NonNull Activity activity) {
        if (activitiesInForegroundCount != 0) {
            activitiesInForegroundCount--;
        }
        uiTraceCallbacks.onActivityStopped(activity, activitiesInForegroundCount == 0);
        appLaunchLifeCycleCallbacks.onActivityStopped();
    }

    @Override
    public void onActivitySaveInstanceState(@NonNull Activity activity, @NonNull Bundle outState) {
        uiTraceCallbacks.onActivitySaveInstanceState(activity, outState);
    }

    @Override
    public void onActivityDestroyed(@NonNull Activity activity) {
        uiTraceCallbacks.onActivityDestroyed(activity);
    }

    @Override
    public synchronized void onNewSessionStarted(@NonNull final Session runningSession, @Nullable Session lastSession) {
        appLaunchLifeCycleCallbacks.onNewSessionStarted(runningSession);
    }
}
