package com.instabug.apm.appflow.manager

import com.instabug.apm.appStateDispacher.AppStateEventListener
import com.instabug.apm.appStateDispacher.AppStateEventDispatcher
import com.instabug.apm.appflow.configuration.AppFlowConfigurationProvider
import com.instabug.apm.appflow.model.AppFlowAttribute
import com.instabug.apm.appflow.usecases.UseCase
import com.instabug.apm.common.concurrent.catchingExecuteAndGet
import com.instabug.apm.di.Provider
import com.instabug.apm.handler.session.APMSessionObserver
import com.instabug.apm.handler.session.SessionObserverHandler
import com.instabug.apm.model.TimeCaptureBoundModel
import com.instabug.apm.v3_session_data_readiness.APMSessionLazyDataProvider
import com.instabug.apm.v3_session_data_readiness.APMSessionReadinessHandler
import com.instabug.apm.v3_session_data_readiness.APMSessionReadinessManager
import java.util.concurrent.ExecutorService

interface AppFlowManager {
    /**
     * Starts a new app flow
     * @param name of the flow to be started
     */
    fun startFlow(name: TimeCaptureBoundModel<String?>)

    /**
     * ends a flow with a given name
     * @param name of the flow to be ended
     */
    fun endFlow(name: TimeCaptureBoundModel<String?>)

    /**
     * Sets AppFlow attribute value
     * <br/>
     *
     * If attribute is not already added it adds the attribute and sets its value
     * <br/>
     *
     * If value is null the attribute will be removed
     * <br/>
     *
     * If no active flow with the passed name then attribute will not be set or modified
     * <br/>
     *
     * @param attribute, non null time bound AppFlowAttribute
     */
    fun setAttribute(attribute: TimeCaptureBoundModel<AppFlowAttribute>)

    /**
     * Starts app flow capturing configs
     */
    fun start()

    /**
     * checks appFlow feature state whether enabled or disabled
     * if enabled calls start logic
     * else it calls stop logic
     */
    fun onStateChanged()

    /**
     * Handles clear dangling sessions on new app launch
     */
    fun onNewAppLaunch()
}

class AppFlowManagerImpl(
    private val startAppFlowUseCaseProvider: Provider<UseCase<TimeCaptureBoundModel<String?>, Unit>?>,
    private val endAppFlowUseCaseProvider: Provider<UseCase<TimeCaptureBoundModel<String?>, Unit>?>,
    private val setFlowAttributeUseCaseProvider: Provider<UseCase<TimeCaptureBoundModel<AppFlowAttribute>, Unit>?>,
    private val appFlowDisabledUseCaseProvider: Provider<UseCase<Unit, Unit>?>,
    private val appFlowAppLaunchUseCaseProvider: Provider<UseCase<Unit, Unit>?>,
    private val appFlowConfigurationProvider: AppFlowConfigurationProvider,
    private val sessionObserver: APMSessionObserver,
    private val appFlowAppStateEventListenerProvider: Provider<AppStateEventListener>,
    private val appStateEventDispatcher: AppStateEventDispatcher,
    private val appFlowApmSessionReadinessHandlerProvider: Provider<APMSessionReadinessHandler?>,
    private val apmSessionLazyDataProvider: APMSessionLazyDataProvider,
    private val executorProvider: Provider<ExecutorService>
) : AppFlowManager, APMSessionReadinessManager {

    private val executor: ExecutorService
        get() = executorProvider.invoke()
    private var appFlowAppStateEventListener: AppStateEventListener? = null
    private var appFlowApmSessionReadinessHandler: APMSessionReadinessHandler? = null

    override fun onNewAppLaunch() = executor.execute {
        appFlowAppLaunchUseCaseProvider()?.invoke(Unit)
    }

    override fun startFlow(name: TimeCaptureBoundModel<String?>) = executor.execute {
        startAppFlowUseCaseProvider()?.invoke(name)
    }

    override fun endFlow(name: TimeCaptureBoundModel<String?>) = executor.execute {
        endAppFlowUseCaseProvider()?.invoke(name)
    }

    override fun setAttribute(attribute: TimeCaptureBoundModel<AppFlowAttribute>) =
        executor.execute {
            setFlowAttributeUseCaseProvider()?.invoke(attribute)
        }

    override fun start() = executor.execute {
        if (appFlowConfigurationProvider.enabled) {
            startSynchronous()
        }
    }

    private fun startSynchronous() {
        SessionObserverHandler.register(sessionObserver)
        if (appFlowAppStateEventListener == null) {
            appFlowAppStateEventListener = appFlowAppStateEventListenerProvider()
            appFlowAppStateEventListener?.let(appStateEventDispatcher::plusAssign)
            registerAppFlowAPMSessionReadinessHandlerSynchronous()
        }
    }

    override fun onStateChanged() = executor.execute {
        if (appFlowConfigurationProvider.enabled) startSynchronous()
        else disableAppFlowFeature()
    }

    private fun disableAppFlowFeature() {
        stopSynchronous()
        appFlowDisabledUseCaseProvider()?.invoke(Unit)
    }

    private fun stopSynchronous() {
        SessionObserverHandler.unregister(sessionObserver)
        appFlowAppStateEventListener?.let(appStateEventDispatcher::minusAssign)
        appFlowAppStateEventListener = null
        unregisterAppFlowAPMSessionReadinessHandlerSynchronous()
    }

    override fun registerReadinessHandler() {
        executor.catchingExecuteAndGet {
            if (appFlowConfigurationProvider.enabled) registerAppFlowAPMSessionReadinessHandlerSynchronous()
        }
    }

    private fun registerAppFlowAPMSessionReadinessHandlerSynchronous() {
        if (appFlowApmSessionReadinessHandler == null) {
            appFlowApmSessionReadinessHandler = appFlowApmSessionReadinessHandlerProvider()
                ?.also(apmSessionLazyDataProvider::registerHandler)
        }
    }

    private fun unregisterAppFlowAPMSessionReadinessHandlerSynchronous() {
        appFlowApmSessionReadinessHandler?.also(apmSessionLazyDataProvider::unregisterHandler)
        appFlowApmSessionReadinessHandler = null
    }
}