package com.instabug.library.apm_okhttp_event_listener

import com.instabug.apm.model.EventTimeMetricCapture
import com.instabug.library.apm_okhttp_event_listener.NetworkLatencyConstants.DNS_LOOKUP_SPAN_KEY
import com.instabug.library.apm_okhttp_event_listener.NetworkLatencyConstants.DURATION_MICROS
import com.instabug.library.apm_okhttp_event_listener.NetworkLatencyConstants.EVENT_NAME
import com.instabug.library.apm_okhttp_event_listener.NetworkLatencyConstants.FAILED
import com.instabug.library.apm_okhttp_event_listener.NetworkLatencyConstants.HANDSHAKE_SPAN_KEY
import com.instabug.library.apm_okhttp_event_listener.NetworkLatencyConstants.REQUEST_SPAN_KEY
import com.instabug.library.apm_okhttp_event_listener.NetworkLatencyConstants.RESPONSE_SPAN_KEY
import com.instabug.library.apm_okhttp_event_listener.NetworkLatencyConstants.SERVER_PROCESSING_SPAN_KEY
import com.instabug.library.apm_okhttp_event_listener.NetworkLatencyConstants.START_TIME_MICROS
import com.instabug.library.apm_okhttp_event_listener.NetworkLatencyConstants.TLS_SPAN_KEY
import com.instabug.library.diagnostics.IBGDiagnostics
import com.instabug.library.map.Mapper
import org.json.JSONArray
import org.json.JSONObject

class NetworkLatencySpansMapper : Mapper<Array<EventTimeMetricCapture?>, String?> {

    private val Array<EventTimeMetricCapture?>.requestFailed
        get() = getOrNull(LatencyEvent.REQUEST_BODY_START) != null
                && getOrNull(LatencyEvent.REQUEST_BODY_END) == null
                && getOrNull(LatencyEvent.REQUEST_FAILED) != null
    private val Array<EventTimeMetricCapture?>.responseFailed
        get() = getOrNull(LatencyEvent.RESPONSE_BODY_START) != null
                && getOrNull(LatencyEvent.RESPONSE_BODY_END) == null
                && getOrNull(LatencyEvent.REQUEST_FAILED) != null

    override fun map(from: Array<EventTimeMetricCapture?>): String? {
        if (from.all { it == null }) return null
        val spansJsonArray = JSONArray()
        spansJsonArray.runCatching {
            putDnsLookupSpan(from)
            putHandshakeSpan(from)
            putTLSSpan(from)
            putRequestSpan(from)
            putServerProcessingSpan(from)
            putResponseSpan(from)
        }.onFailure {
            IBGDiagnostics.reportNonFatal(
                it, "${NetworkLatencyConstants.MAPPING_ERROR_MESSAGE} ${it.message}"
            )
        }
        return spansJsonArray.toString()
    }

    private fun JSONArray.putTLSSpan(eventTimeMetrics: Array<EventTimeMetricCapture?>) =
        putSpan(
            eventTimeMetrics = eventTimeMetrics,
            spanName = TLS_SPAN_KEY,
            startEvents = arrayOf(LatencyEvent.SECURE_CONNECT_START),
            endEvents = arrayOf(LatencyEvent.SECURE_CONNECT_END, LatencyEvent.REQUEST_FAILED)
        )

    private fun JSONArray.putDnsLookupSpan(eventTimeMetrics: Array<EventTimeMetricCapture?>) =
        putSpan(
            eventTimeMetrics = eventTimeMetrics,
            spanName = DNS_LOOKUP_SPAN_KEY,
            startEvents = arrayOf(LatencyEvent.DNS_START),
            endEvents = arrayOf(LatencyEvent.DNS_END, LatencyEvent.REQUEST_FAILED)
        )

    private fun JSONArray.putHandshakeSpan(eventTimeMetrics: Array<EventTimeMetricCapture?>) =
        putSpan(
            eventTimeMetrics = eventTimeMetrics,
            spanName = HANDSHAKE_SPAN_KEY,
            startEvents = arrayOf(LatencyEvent.CONNECT_START),
            endEvents = arrayOf(
                LatencyEvent.SECURE_CONNECT_START,
                LatencyEvent.CONNECT_END,
                LatencyEvent.REQUEST_FAILED
            )
        )

    private fun JSONArray.putRequestSpan(eventTimeMetrics: Array<EventTimeMetricCapture?>) =
        putSpan(
            eventTimeMetrics = eventTimeMetrics,
            spanName = REQUEST_SPAN_KEY,
            startEvents = arrayOf(LatencyEvent.REQUEST_HEADERS_START),
            endEvents = if (eventTimeMetrics.requestFailed) arrayOf(LatencyEvent.REQUEST_FAILED)
            else arrayOf(
                LatencyEvent.REQUEST_BODY_END,
                LatencyEvent.REQUEST_HEADERS_END,
                LatencyEvent.REQUEST_FAILED
            )
        )

    private fun JSONArray.putResponseSpan(eventTimeMetrics: Array<EventTimeMetricCapture?>) =
        putSpan(
            eventTimeMetrics = eventTimeMetrics,
            spanName = RESPONSE_SPAN_KEY,
            startEvents = arrayOf(LatencyEvent.RESPONSE_HEADERS_START),
            endEvents = if (eventTimeMetrics.responseFailed) arrayOf(LatencyEvent.REQUEST_FAILED)
            else arrayOf(
                LatencyEvent.RESPONSE_BODY_END,
                LatencyEvent.RESPONSE_HEADERS_END,
                LatencyEvent.REQUEST_FAILED
            )
        )

    private fun JSONArray.putServerProcessingSpan(eventTimeMetrics: Array<EventTimeMetricCapture?>) =
        putSpan(
            eventTimeMetrics = eventTimeMetrics,
            spanName = SERVER_PROCESSING_SPAN_KEY,
            startEvents = if (eventTimeMetrics.requestFailed) emptyArray()
            else arrayOf(LatencyEvent.REQUEST_BODY_END, LatencyEvent.REQUEST_HEADERS_END),
            endEvents = arrayOf(LatencyEvent.RESPONSE_HEADERS_START, LatencyEvent.REQUEST_FAILED)
        )

    private fun JSONArray.putSpan(
        eventTimeMetrics: Array<EventTimeMetricCapture?>,
        spanName: String,
        startEvents: Array<Int>,
        endEvents: Array<Int>
    ) = eventTimeMetrics.getSpanAsJson(
        spanName = spanName,
        startEvents = startEvents,
        endEvents = endEvents
    )?.let { put(it) }

    private fun Array<EventTimeMetricCapture?>.getSpanAsJson(
        spanName: String,
        startEvents: Array<Int>,
        endEvents: Array<Int>
    ): JSONObject? {
        val startTime = getOrFallBack(startEvents)
        val endTime = getOrFallBack(endEvents)
        return getSpanAsJson(spanName, startTime, endTime)
    }

    private fun Array<EventTimeMetricCapture?>.getOrFallBack(events: Array<Int>):
            EventTimeMetricCapture? = events.firstNotNullOfOrNull { get(it) }

    private fun Array<EventTimeMetricCapture?>.getSpanAsJson(
        spanName: String,
        startEventMetric: EventTimeMetricCapture?,
        endEventMetric: EventTimeMetricCapture?
    ): JSONObject? = if (startEventMetric == null) null
    else JSONObject().also { jsonObject ->
        startEventMetric.let { startTime ->
            jsonObject.put(EVENT_NAME, spanName)
            jsonObject.put(START_TIME_MICROS, startTime.getTimeStampMicro())
            endEventMetric?.let { endTime ->
                val duration = endTime.getMicroTime() - startTime.getMicroTime()
                jsonObject.put(DURATION_MICROS, duration)
                markStageFailed(endEventMetric, jsonObject)
            } ?: also {
                jsonObject.put(DURATION_MICROS, 0)
                jsonObject.put(FAILED, true)
            }
        }
    }

    private fun Array<EventTimeMetricCapture?>.markStageFailed(
        endEventMetric: EventTimeMetricCapture,
        jsonObject: JSONObject
    ) = (endEventMetric.getNanoTime() == getOrNull(LatencyEvent.REQUEST_FAILED)?.getNanoTime())
        .also { if (it) jsonObject.put(FAILED, true) }

}