/*
 * Decompiled with CFR 0.152.
 */
package com.indeed.util.io;

import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.indeed.util.core.io.Closeables2;
import com.indeed.util.io.SafeOutputStream;
import java.io.Closeable;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermissions;
import javax.annotation.Nonnegative;
import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.NotThreadSafe;
import org.apache.log4j.Logger;

@ParametersAreNonnullByDefault
public final class SafeFiles {
    private static final Logger LOG = Logger.getLogger(SafeFiles.class);

    public static void rename(Path oldName, Path newName) throws IOException {
        Preconditions.checkNotNull((Object)oldName);
        Preconditions.checkNotNull((Object)newName);
        boolean sameDir = Files.isSameFile(oldName.getParent(), newName.getParent());
        Files.move(oldName, newName, StandardCopyOption.ATOMIC_MOVE);
        SafeFiles.fsync(newName.getParent());
        if (!sameDir) {
            SafeFiles.fsync(oldName.getParent());
        }
    }

    public static void ensureDirectoryExists(Path path) throws IOException {
        if (Files.exists(path, new LinkOption[0])) {
            if (!Files.isDirectory(path, new LinkOption[0])) {
                throw new IOException("path is not a directory: " + path);
            }
        } else {
            Files.createDirectories(path, new FileAttribute[0]);
            SafeFiles.fsyncLineage(path.getParent());
        }
    }

    @Nonnegative
    public static int fsyncRecursive(Path root) throws IOException {
        FsyncingSimpleFileVisitor visitor = new FsyncingSimpleFileVisitor();
        Files.walkFileTree(root, visitor);
        return visitor.getFileCount();
    }

    public static void fsync(Path path) throws IOException {
        if (!Files.isDirectory(path, new LinkOption[0]) && !Files.isRegularFile(path, new LinkOption[0])) {
            throw new IllegalArgumentException("fsync is only supported for regular files and directories: " + path);
        }
        try (FileChannel channel = FileChannel.open(path, StandardOpenOption.READ);){
            channel.force(true);
        }
    }

    private static void fsyncLineage(Path path) throws IOException {
        for (Path cursor = path.toRealPath(new LinkOption[0]); cursor != null; cursor = cursor.getParent()) {
            SafeFiles.fsync(cursor);
        }
    }

    public static void writeUTF8(String value, Path path) throws IOException {
        SafeFiles.write(value.getBytes(Charsets.UTF_8), path);
    }

    public static void write(byte[] data, Path path) throws IOException {
        try (SafeOutputStream out = SafeFiles.createAtomicFile(path);){
            out.write(ByteBuffer.wrap(data));
            out.commit();
        }
    }

    @Nonnull
    public static SafeOutputStream createAtomicFile(Path path) throws IOException {
        Path dir = path.getParent();
        Path name = path.getFileName();
        Path tempFile = Files.createTempFile(dir, name.toString(), ".tmp", PosixFilePermissions.asFileAttribute(PosixFilePermissions.fromString("rw-r--r--")));
        FileChannel fc = null;
        try {
            fc = (FileChannel)Files.newByteChannel(tempFile, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.WRITE);
        }
        catch (Exception e) {
            SafeFiles.deleteIfExistsQuietly(tempFile);
            if (fc != null) {
                Closeables2.closeQuietly(fc, (Logger)LOG);
            }
            Throwables.propagateIfInstanceOf((Throwable)e, IOException.class);
            throw Throwables.propagate((Throwable)e);
        }
        return new SafeFileOutputStream(path, tempFile, fc);
    }

    public static void deleteIfExistsQuietly(Path path) {
        try {
            Files.deleteIfExists(path);
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private SafeFiles() {
    }

    @ParametersAreNonnullByDefault
    @NotThreadSafe
    private static class SafeFileOutputStream
    extends SafeOutputStream {
        @Nonnull
        private final Path path;
        @Nonnull
        private final Path tempFile;
        @Nonnull
        private final OutputStream out;
        @Nonnull
        private final FileChannel fileChannel;
        private boolean closed = false;

        private SafeFileOutputStream(Path path, Path tempFile, FileChannel fileChannel) {
            this.path = path;
            this.tempFile = tempFile;
            this.fileChannel = fileChannel;
            this.out = Channels.newOutputStream(fileChannel);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void commit() throws IOException {
            if (this.closed) {
                return;
            }
            try {
                try {
                    this.out.flush();
                    this.fileChannel.force(true);
                }
                finally {
                    this.fileChannel.close();
                }
                Files.move(this.tempFile, this.path, StandardCopyOption.ATOMIC_MOVE);
            }
            catch (Exception e) {
                SafeFiles.deleteIfExistsQuietly(this.tempFile);
                Throwables.propagateIfInstanceOf((Throwable)e, IOException.class);
                this.closed = true;
                throw Throwables.propagate((Throwable)e);
            }
            this.closed = true;
            SafeFiles.fsync(this.tempFile.getParent());
        }

        @Override
        public void close() throws IOException {
            if (!this.closed) {
                Closeables2.closeQuietly((Closeable)this.fileChannel, (Logger)LOG);
                SafeFiles.deleteIfExistsQuietly(this.tempFile);
                this.closed = true;
            }
        }

        @Override
        public boolean isOpen() {
            return !this.closed;
        }

        @Override
        public int write(ByteBuffer src) throws IOException {
            return SafeFileOutputStream.writeFully(this.fileChannel, src);
        }

        private static int writeFully(WritableByteChannel ch, ByteBuffer bb) throws IOException {
            int total = 0;
            while (bb.remaining() > 0) {
                int n = ch.write(bb);
                if (n <= 0) {
                    throw new RuntimeException("no bytes written");
                }
                total += n;
            }
            return total;
        }

        @Override
        public void write(int b) throws IOException {
            this.checkNotClosed();
            this.out.write(b);
        }

        @Override
        public void write(byte[] b) throws IOException {
            this.checkNotClosed();
            this.out.write(b);
        }

        @Override
        public void write(byte[] b, int off, int len) throws IOException {
            this.checkNotClosed();
            this.out.write(b, off, len);
        }

        @Override
        public void flush() throws IOException {
            this.checkNotClosed();
            this.out.flush();
        }

        private void checkNotClosed() throws IllegalStateException {
            if (this.closed) {
                throw new IllegalStateException("operation not permitted once output is closed");
            }
        }
    }

    private static class FsyncingSimpleFileVisitor
    extends SimpleFileVisitor<Path> {
        @Nonnegative
        private int fileCount = 0;

        private FsyncingSimpleFileVisitor() {
        }

        @Nonnegative
        public int getFileCount() {
            return this.fileCount;
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
            if (attrs.isRegularFile()) {
                SafeFiles.fsync(file);
                ++this.fileCount;
            }
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
            if (attrs.isDirectory()) {
                SafeFiles.fsync(dir);
            }
            return FileVisitResult.CONTINUE;
        }
    }
}

