/*
 * Decompiled with CFR 0.152.
 */
package com.indeed.lsmtree.recordlog;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;
import com.google.common.collect.Lists;
import com.indeed.lsmtree.recordlog.BlockCompressedRecordFile;
import com.indeed.lsmtree.recordlog.RecordFile;
import com.indeed.util.compress.CompressionCodec;
import com.indeed.util.compress.Decompressor;
import com.indeed.util.core.io.Closeables2;
import com.indeed.util.core.reference.SharedReference;
import com.indeed.util.serialization.Serializer;
import fj.data.Option;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import org.apache.log4j.Logger;

public final class RecordLogDirectory<E>
implements RecordFile<E> {
    private static final Logger log = Logger.getLogger(RecordLogDirectory.class);
    public static final int DEFAULT_FILE_INDEX_BITS = 28;
    public static final int DEFAULT_RECORD_INDEX_BITS = 10;
    public static final int DEFAULT_PAD_BITS = 6;
    public static final int DEFAULT_BLOCK_SIZE = 16384;
    private final FileCache fileCache;
    private final int segmentShift;
    private final long segmentMask;
    private final File dir;
    private final Serializer<E> serializer;
    private final int maxCachedFiles;
    private final CompressionCodec codec;
    private final int blockSize;
    private final int fileIndexBits;
    private final int recordIndexBits;
    private final int padBits;

    public RecordLogDirectory(File dir, Serializer<E> serializer, int maxCachedFiles, CompressionCodec codec, int blockSize, int fileIndexBits, int recordIndexBits, int padBits) {
        this(dir, serializer, maxCachedFiles, codec, blockSize, fileIndexBits, recordIndexBits, padBits, false);
    }

    public RecordLogDirectory(File dir, Serializer<E> serializer, int maxCachedFiles, CompressionCodec codec, int blockSize, int fileIndexBits, int recordIndexBits, int padBits, boolean mlockFiles) {
        this.dir = dir;
        this.serializer = serializer;
        this.maxCachedFiles = maxCachedFiles;
        this.codec = codec;
        this.blockSize = blockSize;
        this.fileIndexBits = fileIndexBits;
        this.recordIndexBits = recordIndexBits;
        this.padBits = padBits;
        this.segmentShift = 64 - fileIndexBits;
        this.segmentMask = (1L << this.segmentShift) - 1L;
        this.fileCache = new FileCache(mlockFiles);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public E get(long address) throws IOException {
        Object ret;
        int segmentNum = (int)(address >>> this.segmentShift);
        Option option = this.fileCache.get(segmentNum);
        if (option.isNone()) {
            throw new IOException("address is invalid: " + address);
        }
        SharedReference reference = (SharedReference)option.some();
        try {
            ret = ((BlockCompressedRecordFile)reference.get()).get(address & this.segmentMask);
        }
        finally {
            Closeables2.closeQuietly((Closeable)reference, (Logger)log);
        }
        return ret;
    }

    @Override
    public RecordFile.Reader<E> reader() throws IOException {
        return new Reader();
    }

    @Override
    public RecordFile.Reader<E> reader(long address) throws IOException {
        return new Reader(address);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Option<RecordFile.Reader<E>> getFileReader(final long segmentNum) throws IOException {
        RecordFile.Reader segmentReader;
        Option option = this.fileCache.get((int)segmentNum);
        if (option.isNone()) {
            return Option.none();
        }
        SharedReference reference = (SharedReference)option.some();
        try {
            segmentReader = ((BlockCompressedRecordFile)reference.get()).reader();
        }
        finally {
            Closeables2.closeQuietly((Closeable)reference, (Logger)log);
        }
        return Option.some((Object)new RecordFile.Reader<E>(){
            final long segmentShift;
            final long maxSegmentPosition;
            {
                this.segmentShift = 64 - RecordLogDirectory.this.fileIndexBits;
                this.maxSegmentPosition = (1L << (int)this.segmentShift) - 1L;
            }

            @Override
            public boolean next() throws IOException {
                return segmentReader.next();
            }

            @Override
            public long getPosition() {
                long segmentPosition = segmentReader.getPosition();
                if (segmentPosition > this.maxSegmentPosition) {
                    throw new IllegalStateException("position in segment file" + segmentNum + " is too high to be addressable in record log directory with " + RecordLogDirectory.this.fileIndexBits + " file index bits");
                }
                return (segmentNum << (int)this.segmentShift) + segmentPosition;
            }

            @Override
            public E get() {
                return segmentReader.get();
            }

            @Override
            public void close() throws IOException {
                Closeables2.closeQuietly((Closeable)segmentReader, (Logger)log);
            }
        });
    }

    public void garbageCollect(long address) {
        ArrayList filesToDelete = Lists.newArrayList();
        for (int segmentNum = this.getSegmentNum(address) - 1; segmentNum >= 0; --segmentNum) {
            File f = RecordLogDirectory.getSegmentPath(this.dir, segmentNum);
            if (!f.exists()) break;
            filesToDelete.add(f);
        }
        for (int i = filesToDelete.size() - 1; i >= 0; --i) {
            ((File)filesToDelete.get(i)).delete();
        }
    }

    @Override
    public void close() throws IOException {
        this.fileCache.close();
    }

    public int getSegmentNum(long address) {
        return (int)(address >>> 64 - this.fileIndexBits);
    }

    public long getAddress(long segNum) {
        return segNum << 64 - this.fileIndexBits;
    }

    public int getMaxSegmentNum() throws IOException {
        return RecordLogDirectory.getMaxSegmentNum(this.dir);
    }

    public long getSegmentTimestamp(int segmentNum) throws IOException {
        return RecordLogDirectory.getSegmentPath(this.dir, segmentNum, false).lastModified();
    }

    public static File getSegmentPath(File file, int segmentNum, boolean mkDirs) {
        File segmentDir = file;
        for (int divisor = 1000000; divisor > 1; divisor /= 1000) {
            segmentDir = new File(segmentDir, String.format("%03d", segmentNum / divisor % 1000));
        }
        if (mkDirs) {
            segmentDir.mkdirs();
        }
        return new File(segmentDir, String.format("%09d", segmentNum) + ".rec");
    }

    public static int getMaxSegmentNum(File path) throws IOException {
        int maxSegmentNum = -1;
        int maxDir = -1;
        if (path.exists()) {
            for (File f : path.listFiles()) {
                int segmentNum;
                String name = f.getName();
                if (name.matches("\\d+") && f.isDirectory()) {
                    int dirNum = Integer.parseInt(name);
                    if (dirNum <= maxDir) continue;
                    maxDir = dirNum;
                    continue;
                }
                if (!name.matches("\\d+\\.rec") || (segmentNum = Integer.parseInt(name.substring(0, name.length() - 4))) <= maxSegmentNum) continue;
                maxSegmentNum = segmentNum;
            }
            if (maxSegmentNum >= 0) {
                return maxSegmentNum;
            }
            if (maxDir >= 0) {
                return RecordLogDirectory.getMaxSegmentNum(new File(path, String.format("%03d", maxDir)));
            }
        }
        return -1;
    }

    public static int getMinSegmentNum(File path) throws IOException {
        int minSegmentNum = Integer.MAX_VALUE;
        int minDir = Integer.MAX_VALUE;
        if (path.exists()) {
            for (File f : path.listFiles()) {
                int segmentNum;
                String name = f.getName();
                if (name.matches("\\d+") && f.isDirectory() && f.list().length > 0) {
                    int dirNum = Integer.parseInt(name);
                    if (dirNum >= minDir) continue;
                    minDir = dirNum;
                    continue;
                }
                if (!name.matches("\\d+\\.rec") || (segmentNum = Integer.parseInt(name.substring(0, name.length() - 4))) >= minSegmentNum) continue;
                minSegmentNum = segmentNum;
            }
            if (minSegmentNum < Integer.MAX_VALUE) {
                return minSegmentNum;
            }
            if (minDir < Integer.MAX_VALUE) {
                return RecordLogDirectory.getMinSegmentNum(new File(path, String.format("%03d", minDir)));
            }
        }
        return -1;
    }

    public static File getSegmentPath(File root, int segment) {
        return RecordLogDirectory.getSegmentPath(root, segment, false);
    }

    private class FileCache
    implements Closeable {
        private final LoadingCache<Integer, Option<SharedReference<BlockCompressedRecordFile<E>>>> readerCache;
        private final BlockingQueue<Decompressor> decompressorPool;
        private final boolean mlockFiles;
        private final CacheLoader<Integer, Option<SharedReference<BlockCompressedRecordFile<E>>>> open = new CacheLoader<Integer, Option<SharedReference<BlockCompressedRecordFile<E>>>>(){

            public Option<SharedReference<BlockCompressedRecordFile<E>>> load(Integer segmentNum) {
                try {
                    File segmentFile = RecordLogDirectory.getSegmentPath(RecordLogDirectory.this.dir, segmentNum, false);
                    if (!segmentFile.exists()) {
                        return Option.none();
                    }
                    long start = System.nanoTime();
                    BlockCompressedRecordFile recordFile = new BlockCompressedRecordFile.Builder(segmentFile, RecordLogDirectory.this.serializer, RecordLogDirectory.this.codec).setDecompressorPool(FileCache.this.decompressorPool).setBlockSize(RecordLogDirectory.this.blockSize).setRecordIndexBits(RecordLogDirectory.this.recordIndexBits).setPadBits(RecordLogDirectory.this.padBits).setMlockFiles(FileCache.this.mlockFiles).build();
                    log.debug((Object)("segment open time: " + (double)(System.nanoTime() - start) / 1000.0 + " us"));
                    return Option.some((Object)SharedReference.create(recordFile));
                }
                catch (IOException e) {
                    log.error((Object)("error opening file with segment number " + segmentNum), (Throwable)e);
                    return Option.none();
                }
            }
        };

        public FileCache(boolean mlockFiles) {
            this.mlockFiles = mlockFiles;
            this.decompressorPool = new LinkedBlockingQueue<Decompressor>();
            this.readerCache = CacheBuilder.newBuilder().maximumSize((long)RecordLogDirectory.this.maxCachedFiles).removalListener(new RemovalListener<Integer, Option<SharedReference<BlockCompressedRecordFile<E>>>>(){

                public void onRemoval(RemovalNotification<Integer, Option<SharedReference<BlockCompressedRecordFile<E>>>> notification) {
                    Integer segmentNum = (Integer)notification.getKey();
                    Option referenceOption = (Option)notification.getValue();
                    for (SharedReference reference : referenceOption) {
                        try {
                            reference.close();
                        }
                        catch (IOException e) {
                            log.error((Object)"error on block cleanup", (Throwable)e);
                        }
                    }
                }
            }).build(this.open);
        }

        public Option<SharedReference<BlockCompressedRecordFile<E>>> get(Integer key) {
            Option option = (Option)this.readerCache.getUnchecked((Object)key);
            if (option.isNone()) {
                this.readerCache.invalidate((Object)key);
                return option;
            }
            SharedReference copy = ((SharedReference)option.some()).tryCopy();
            if (copy == null) {
                return this.get(key);
            }
            return Option.some((Object)copy);
        }

        @Override
        public void close() throws IOException {
            this.readerCache.invalidateAll();
        }
    }

    private class Reader
    implements RecordFile.Reader<E> {
        private RecordFile.Reader<E> currentReader = null;
        private int currentSegmentNum = 0;
        private boolean done = false;

        private Reader() throws IOException {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public Reader(long address) throws IOException {
            this.currentSegmentNum = (int)(address >>> RecordLogDirectory.this.segmentShift);
            Option recordFile = RecordLogDirectory.this.fileCache.get(this.currentSegmentNum);
            if (recordFile.isNone()) {
                throw new IOException("address is invalid: " + address);
            }
            SharedReference reference = (SharedReference)recordFile.some();
            try {
                this.currentReader = ((BlockCompressedRecordFile)reference.get()).reader(address & RecordLogDirectory.this.segmentMask);
            }
            finally {
                Closeables2.closeQuietly((Closeable)reference, (Logger)log);
            }
        }

        @Override
        public boolean next() throws IOException {
            if (this.done) {
                return false;
            }
            if (this.currentReader == null && !this.getSegmentReader(this.currentSegmentNum)) {
                this.done = true;
                return false;
            }
            while (!this.currentReader.next()) {
                if (!this.getSegmentReader(this.currentSegmentNum + 1)) {
                    this.done = true;
                    return false;
                }
                ++this.currentSegmentNum;
            }
            return true;
        }

        @Override
        public long getPosition() {
            return ((long)this.currentSegmentNum << RecordLogDirectory.this.segmentShift) + this.currentReader.getPosition();
        }

        @Override
        public E get() {
            return this.currentReader.get();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private boolean getSegmentReader(int segmentNum) throws IOException {
            Closeables2.closeQuietly(this.currentReader, (Logger)log);
            Option option = RecordLogDirectory.this.fileCache.get(segmentNum);
            Iterator i$ = option.iterator();
            if (i$.hasNext()) {
                SharedReference reference = (SharedReference)i$.next();
                try {
                    this.currentReader = ((BlockCompressedRecordFile)reference.get()).reader();
                }
                finally {
                    Closeables2.closeQuietly((Closeable)reference, (Logger)log);
                }
                return true;
            }
            return false;
        }

        @Override
        public void close() {
            Closeables2.closeQuietly(this.currentReader, (Logger)log);
        }
    }

    public static final class Builder<E> {
        private File dir;
        private Serializer<E> serializer;
        private int maxCachedFiles = 1024;
        private CompressionCodec codec;
        private int blockSize = 16384;
        private int fileIndexBits = 28;
        private int recordIndexBits = 10;
        private int padBits = 6;
        private boolean mlockFiles;

        public Builder(File dir, Serializer<E> serializer, CompressionCodec codec) {
            this.dir = dir;
            this.serializer = serializer;
            this.codec = codec;
        }

        public Builder<E> setDir(File dir) {
            this.dir = dir;
            return this;
        }

        public Builder<E> setSerializer(Serializer<E> serializer) {
            this.serializer = serializer;
            return this;
        }

        public Builder<E> setMaxCachedFiles(int maxCachedFiles) {
            this.maxCachedFiles = maxCachedFiles;
            return this;
        }

        public Builder<E> setCodec(CompressionCodec codec) {
            this.codec = codec;
            return this;
        }

        public Builder<E> setBlockSize(int blockSize) {
            this.blockSize = blockSize;
            return this;
        }

        public Builder<E> setFileIndexBits(int fileIndexBits) {
            this.fileIndexBits = fileIndexBits;
            return this;
        }

        public Builder<E> setRecordIndexBits(int recordIndexBits) {
            this.recordIndexBits = recordIndexBits;
            return this;
        }

        public Builder<E> setPadBits(int padBits) {
            this.padBits = padBits;
            return this;
        }

        public Builder<E> setMlockFiles(boolean mlockFiles) {
            this.mlockFiles = mlockFiles;
            return this;
        }

        public File getDir() {
            return this.dir;
        }

        public Serializer<E> getSerializer() {
            return this.serializer;
        }

        public int getMaxCachedFiles() {
            return this.maxCachedFiles;
        }

        public CompressionCodec getCodec() {
            return this.codec;
        }

        public int getBlockSize() {
            return this.blockSize;
        }

        public int getFileIndexBits() {
            return this.fileIndexBits;
        }

        public int getRecordIndexBits() {
            return this.recordIndexBits;
        }

        public int getPadBits() {
            return this.padBits;
        }

        public boolean isMlockFiles() {
            return this.mlockFiles;
        }

        public RecordLogDirectory<E> build() {
            return new RecordLogDirectory<E>(this.dir, this.serializer, this.maxCachedFiles, this.codec, this.blockSize, this.fileIndexBits, this.recordIndexBits, this.padBits, this.mlockFiles);
        }
    }

    public static class Writer<E>
    implements RecordFile.Writer<E> {
        private final int blockSize;
        private final int recordIndexBits;
        private final int padBits;
        private final int segmentShift;
        private final File path;
        private final Serializer<E> serializer;
        private final CompressionCodec codec;
        private final File tmpPath;
        private File currentWriterPath;
        private final long rollFrequency;
        private long lastRollTime;
        private RecordFile.Writer currentWriter;
        private int currentSegmentNum;

        public static <E> Writer<E> create(File file, Serializer<E> serializer, CompressionCodec codec, long rollFrequency) throws IOException {
            return new Writer<E>(file, serializer, codec, rollFrequency, 16384, 28, 10, 6, -1);
        }

        public static <E> Writer<E> create(File file, Serializer<E> serializer, CompressionCodec codec, long rollFrequency, int blockSize, int fileIndexBits, int recordIndexBits, int padBits) throws IOException {
            return new Writer<E>(file, serializer, codec, rollFrequency, blockSize, fileIndexBits, recordIndexBits, padBits, -1);
        }

        public static <E> Writer<E> create(File file, Serializer<E> serializer, CompressionCodec codec, long rollFrequency, int maxSegment) throws IOException {
            return new Writer<E>(file, serializer, codec, rollFrequency, 16384, 28, 10, 6, maxSegment);
        }

        public static <E> Writer<E> create(File file, Serializer<E> serializer, CompressionCodec codec, long rollFrequency, int blockSize, int fileIndexBits, int recordIndexBits, int padBits, int maxSegment) throws IOException {
            return new Writer<E>(file, serializer, codec, rollFrequency, blockSize, fileIndexBits, recordIndexBits, padBits, maxSegment);
        }

        private Writer(File file, Serializer<E> serializer, CompressionCodec codec, long rollFrequency, int blockSize, int fileIndexBits, int recordIndexBits, int padBits, int maxSegment) throws IOException {
            this.path = file;
            this.serializer = serializer;
            this.codec = codec;
            this.blockSize = blockSize;
            this.recordIndexBits = recordIndexBits;
            this.padBits = padBits;
            this.segmentShift = 64 - fileIndexBits;
            this.tmpPath = new File(file, "tmp");
            this.rollFrequency = rollFrequency;
            this.tmpPath.mkdirs();
            if (maxSegment < 0) {
                this.currentSegmentNum = RecordLogDirectory.getMaxSegmentNum(this.path);
                if (this.currentSegmentNum == -1 || this.verifySegmentIntegrity(this.path, this.currentSegmentNum)) {
                    ++this.currentSegmentNum;
                }
            } else {
                this.currentSegmentNum = maxSegment + 1;
            }
            log.info((Object)("current segment num: " + this.currentSegmentNum));
            this.currentWriter = this.createWriter(this.currentSegmentNum);
            this.lastRollTime = System.currentTimeMillis();
        }

        private RecordFile.Writer createWriter(int segmentNum) throws IOException {
            this.currentWriterPath = new File(this.tmpPath, String.valueOf(segmentNum) + ".rec");
            return BlockCompressedRecordFile.Writer.open(this.currentWriterPath, this.serializer, this.codec, this.blockSize, this.recordIndexBits, this.padBits);
        }

        @Override
        public long append(E entry) throws IOException {
            long writerAddress;
            if (System.currentTimeMillis() - this.lastRollTime > this.rollFrequency) {
                this.roll();
            }
            if ((writerAddress = this.currentWriter.append(entry)) >= 1L << this.segmentShift) {
                throw new IOException("current writer has exceeded maximum size");
            }
            return ((long)this.currentSegmentNum << this.segmentShift) + writerAddress;
        }

        public void roll() throws IOException {
            this.currentWriter.sync();
            this.currentWriter.close();
            File segmentFile = RecordLogDirectory.getSegmentPath(this.path, this.currentSegmentNum, true);
            this.currentWriterPath.renameTo(segmentFile);
            this.currentWriter = this.createWriter(++this.currentSegmentNum);
            this.lastRollTime = System.currentTimeMillis();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public boolean verifySegmentIntegrity(File file, int segmentNum) {
            BlockCompressedRecordFile<E> recordFile = null;
            try {
                recordFile = new BlockCompressedRecordFile.Builder<E>(RecordLogDirectory.getSegmentPath(file, segmentNum, false), this.serializer, this.codec).setBlockSize(this.blockSize).setRecordIndexBits(this.recordIndexBits).setPadBits(this.padBits).build();
            }
            catch (IOException e) {
                boolean bl = false;
                return bl;
            }
            finally {
                if (recordFile != null) {
                    try {
                        recordFile.close();
                    }
                    catch (IOException e) {}
                }
            }
            return true;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void close() throws IOException {
            try {
                this.roll();
            }
            finally {
                this.currentWriter.close();
            }
        }

        public void sync() throws IOException {
            this.currentWriter.sync();
        }
    }
}

