/************************************************************
 *  * EaseMob CONFIDENTIAL 
 * __________________ 
 * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
 *  
 * NOTICE: All information contained herein is, and remains 
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material 
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import com.hyphenate.util.EMLog;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.Iterator;

/**
 * this class is used to encode the domain EMMessage object into json string
 * payload which will be transfered in xmpp.chat.body attribute it also
 * responsible decode the received xmpp payload in json back to EMMessage object
 * 
 */
public class MessageEncoder {
	private static final String TAG = "encoder";
	/*
	 * txt message:
	 * {"from":"test2","to":"test1","bodies":[{"type":"txt","msg":"hello from test2"
	 * }]} image message:
	 * {"from":"test1","to":"test2","bodies":[{"type":"img","url":
	 * "http://storage.hyphenate.com/uploads/chatdemo/chat/image/test120140208T123347.jpg"
	 * ,"filename":"test1.jpg","thumb":
	 * "http://storage.hyphenate.com/getthumbnail.php?filename=chatdemo/chat/image/test120140208T123347.jpg"
	 * }]} voice message:
	 * {"from":"test2","to":"test1","bodies":[{"type":"audio","url":
	 * "http://storage.hyphenate.com/uploads/chatdemo/chat/audio/test120140209T190431.amr"
	 * ,"filename":"test1.amr","length":10}]} location message:
	 * {"from":"test1","to"
	 * :"test2","bodies":[{"type":"loc","addr":"西城区西便门桥 ","lat"
	 * :39.9053,"lng":116.36302}]} command message: command:
	 * {"from":"admin","to"
	 * :"test1","bodies":[{"type":"cmd","action":"swipe","param"
	 * :["arg1","arg2"]}]} command
	 * resp(optional):{"from":"test1","to":"admin","type"
	 * :"cmd","action":"swipe","id":"command id","resp":"cmd resp"} video
	 * message: {"from":"test1","to":"test2","bodies":[{"type":"video","url":
	 * "ttp://a1.hyphenate.com/hyphenate-demo/chatdemoui/chatfiles/b41ef1ee-0363-11e4-a63f-4bc7a9d1767a"
	 * ,"filename":"video","thumb":
	 * "http://a1.hyphenate.com/hyphenate-demo/chatdemoui/chatfiles/b455e05a-0363-11e4-bfa7-b99877111c26"
	 * , "length":3,"secret":"tCE7ygNjEeSHnqGXOavxcGVbVeSBy4olQ3YtO4CDhnUBavRk",
	 * "thumb_secret":"tGjMEANjEeSah4vR-7wLUu8znf-1bP02H1223vryebtbYeUU"}]}
	 */

	/**
	 * notice that the json msg send on the wire is different from json string
	 * saved in local db. for example. image message, the local db will save
	 * localurl attribute, whitch wont send over
	 */
	public final static String ATTR_TYPE = "type";
	public final static String ATTR_TO = "to";
	public final static String ATTR_FROM = "from";
	public final static String ATTR_MSG = "msg";
	public final static String ATTR_URL = "url";
	public final static String ATTR_LOCALURL = "localurl";
	public final static String ATTR_THUMB_LOCALURL = "thumblocalurl";
	public final static String ATTR_FILENAME = "filename";
	public final static String ATTR_THUMBNAIL = "thumb";
	public final static String ATTR_SECRET = "secret";
	public final static String ATTR_SIZE = "size";
	public final static String ATTR_IMG_WIDTH = "width";
	public final static String ATTR_IMG_HEIGHT = "height";
	public final static String ATTR_THUMBNAIL_SECRET = "thumb_secret";
	public final static String ATTR_LENGTH = "length";
	public final static String ATTR_ADDRESS = "addr";
	public final static String ATTR_LATITUDE = "lat";
	public final static String ATTR_LONGITUDE = "lng";
	public final static String ATTR_ACTION = "action";
	public final static String ATTR_PARAM = "param";
	public final static String ATTR_FILE_LENGTH = "file_length";

	// optional json obj for additional user defined attributes
	public final static String ATTR_EXT = "ext";

	private final static String ATTR_TYPE_TXT = "txt";
	private final static String ATTR_TYPE_IMG = "img";
	private final static String ATTR_TYPE_VOICE = "audio";
	private final static String ATTR_TYPE_VIDEO = "video";
	private final static String ATTR_TYPE_LOCATION = "loc";
	private final static String ATTR_TYPE_CMD = "cmd";
	private final static String ATTR_TYPE_file = "file";



	/**
	 * create message object from json string
	 * 
	 * @param jsonString
	 * @return
	 */
	@SuppressWarnings("UnnecessaryUnboxing")
	public static EMMessage getMsgFromJson(String jsonString) {
		try {
			JSONObject jsonPayload = new JSONObject(jsonString);
			EMContact from = new EMContact(jsonPayload.getString(ATTR_FROM));
			EMContact to = new EMContact(jsonPayload.getString(ATTR_TO));

			JSONArray bodies = jsonPayload.getJSONArray("bodies");
			if (bodies.length() < 1) {
				EMLog.d(TAG, "wrong msg without body");
				return null;
			}
			// for now, we only deal with first body in bodies[]
			JSONObject bodyJson = bodies.getJSONObject(0);
			String type = bodyJson.getString(ATTR_TYPE);

			EMMessage.Type _type = EMMessage.Type.TXT;
			if (type.equals(ATTR_TYPE_TXT)) {
				_type = EMMessage.Type.TXT;
			} else if (type.equals(ATTR_TYPE_IMG)) {
				_type = EMMessage.Type.IMAGE;
			} else if (type.equals(ATTR_TYPE_file)) {
				_type = EMMessage.Type.FILE;
			} else if (type.equals(ATTR_TYPE_VIDEO)) {
				_type = EMMessage.Type.VIDEO;
			} else if (type.equals(ATTR_TYPE_VOICE)) {
				_type = EMMessage.Type.VOICE;
			} else if (type.equals(ATTR_TYPE_LOCATION)) {
				_type = EMMessage.Type.LOCATION;
			}  else if (type.equals(ATTR_TYPE_CMD)) {
				_type = EMMessage.Type.CMD;
			}
				
			EMMessage msg = null;
			if (from.username.equals(EMClient.getInstance().getCurrentUser())) {
				msg = EMMessage.createSendMessage(_type);
			} else {
				msg = EMMessage.createReceiveMessage(_type);
			}
			
			if (type.equals(ATTR_TYPE_TXT)) {
				// convert xmpp message to EMMessage
				String bodyString = bodyJson.getString(ATTR_MSG);
				String decodeString = bodyString.replaceAll("%22", "\"");
				EMTextMessageBody body = new EMTextMessageBody(decodeString);
				msg.addBody(body);
			} else if (type.equals(ATTR_TYPE_IMG)) {
				String remoteUrl = bodyJson.getString(ATTR_URL);
				String fileName = bodyJson.getString(ATTR_FILENAME);
				String thumbnailUrl=remoteUrl;
				if(bodyJson.has(ATTR_THUMBNAIL))
				{
					thumbnailUrl = bodyJson.getString(ATTR_THUMBNAIL);
				}
				EMImageMessageBody body = new EMImageMessageBody(fileName, remoteUrl, thumbnailUrl);

				if (bodyJson.has(ATTR_LOCALURL)) {
					body.setLocalUrl(bodyJson.getString(ATTR_LOCALURL));
				}
				if (bodyJson.has(ATTR_SECRET)) {
					body.setSecret(bodyJson.getString(ATTR_SECRET));
				}
				if (bodyJson.has(ATTR_THUMBNAIL_SECRET)) {
					body.setThumbnailSecret(bodyJson.getString(ATTR_THUMBNAIL_SECRET));
				}
				if(bodyJson.has(ATTR_SIZE)){
					JSONObject sizeJson = bodyJson.getJSONObject(ATTR_SIZE);
					body.setSize(sizeJson.getInt(ATTR_IMG_WIDTH), sizeJson.getInt(ATTR_IMG_HEIGHT));
				}
				
				msg.addBody(body);
			} else if (type.equals(ATTR_TYPE_file)) {
				String remoteUrl = bodyJson.getString(ATTR_URL);
				String fileName = bodyJson.getString(ATTR_FILENAME);
				EMNormalFileMessageBody body = new EMNormalFileMessageBody(fileName, remoteUrl);

				body.setFileLength(Integer.parseInt(bodyJson.getString(ATTR_FILE_LENGTH)));
				if (bodyJson.has(ATTR_LOCALURL)) {
					body.setLocalUrl(bodyJson.getString(ATTR_LOCALURL));
				}
				if (bodyJson.has(ATTR_SECRET)) {
					body.setSecret(bodyJson.getString(ATTR_SECRET));
				}

				msg.addBody(body);
			} else if (type.equals(ATTR_TYPE_VIDEO)) {
				String remoteUrl = bodyJson.getString(ATTR_URL);
				String fileName = bodyJson.getString(ATTR_FILENAME);
				String thumbnailUrl = bodyJson.getString(ATTR_THUMBNAIL);
				int length = bodyJson.getInt(ATTR_LENGTH);
				EMVideoMessageBody body = new EMVideoMessageBody(fileName, remoteUrl, thumbnailUrl, length);
				if (bodyJson.has(ATTR_LOCALURL)) {
					body.setLocalUrl(bodyJson.getString(ATTR_LOCALURL));
				}
				if (bodyJson.has(ATTR_FILE_LENGTH)) {
					body.setFileLength(bodyJson.getLong(ATTR_FILE_LENGTH));
				}
				if (bodyJson.has(ATTR_THUMB_LOCALURL)) {
					body.setLocalThumb(bodyJson.getString(ATTR_THUMB_LOCALURL));
				}
				if (bodyJson.has(ATTR_SECRET)) {
					body.setSecret(bodyJson.getString(ATTR_SECRET));
				}
				if (bodyJson.has(ATTR_THUMBNAIL_SECRET)) {
					body.setThumbnailSecret(bodyJson.getString(ATTR_THUMBNAIL_SECRET));
				}
				if(bodyJson.has(ATTR_SIZE)){
					JSONObject sizeJson = bodyJson.getJSONObject(ATTR_SIZE);
					body.setThumbnailSize(sizeJson.getInt(ATTR_IMG_WIDTH), sizeJson.getInt(ATTR_IMG_HEIGHT));
				}
				msg.addBody(body);
			} else if (type.equals(ATTR_TYPE_VOICE)) {
				String remoteUrl = bodyJson.getString(ATTR_URL);
				String fileName = bodyJson.getString(ATTR_FILENAME);
				int length = bodyJson.getInt(ATTR_LENGTH);
				EMVoiceMessageBody body = new EMVoiceMessageBody(fileName, remoteUrl, length);
				if (bodyJson.has(ATTR_LOCALURL)) {
					body.setLocalUrl(bodyJson.getString(ATTR_LOCALURL));
				}
				if (bodyJson.has(ATTR_SECRET)) {
					body.setSecret(bodyJson.getString(ATTR_SECRET));
				}
				msg.addBody(body);
			} else if (type.equals(ATTR_TYPE_LOCATION)) {
				String address = bodyJson.getString(ATTR_ADDRESS);
				double latitude = bodyJson.getDouble(ATTR_LATITUDE);
				double longitude = bodyJson.getDouble(ATTR_LONGITUDE);

				EMLocationMessageBody body = new EMLocationMessageBody(address, latitude, longitude);
				msg.addBody(body);
			}  else if (type.equals(ATTR_TYPE_CMD)) {
			    // CMD message will not be stored in DB, those part code will not be excuted
//				HashMap<String, String> params = new HashMap<String, String>();
//				if (bodyJson.has(ATTR_PARAM)) {
//					// process cmd params
//					JSONArray array = bodyJson.getJSONArray(ATTR_PARAM);
//					for(int i = 0; i < array.length(); i++){
//						JSONObject jObj = array.getJSONObject(i);
//						String key = (String) jObj.keys().next();
//						String value = (String) jObj.get(key);
//						params.put(key, value);
//					}
//					
//				}
//				EMCmdMessageBody body = new EMCmdMessageBody(bodyJson.getString(ATTR_ACTION), params);
//				msg.addBody(body);
			}

			if (msg != null) {
				msg.setFrom(from.getUsername());
				msg.setTo(to.getUsername());
			}

			// process extra attributes if any
			if (jsonPayload.has(ATTR_EXT)) {
				JSONObject extAttrObj = jsonPayload.getJSONObject(ATTR_EXT);
				Iterator itor = extAttrObj.keys();
				while (itor.hasNext()) {
					String key = (String) itor.next();
					Object obj = extAttrObj.get(key);
					if (obj instanceof String) {
						msg.setAttribute(key, (String) obj);
					} else if (obj instanceof Integer) {
						msg.setAttribute(key, ((Integer) obj).intValue());
					} else if (obj instanceof Boolean) {
						msg.setAttribute(key, ((Boolean) obj).booleanValue());
					} else if (obj instanceof JSONObject) {
						msg.setAttribute(key, (JSONObject) obj);
					} else if (obj instanceof JSONArray) {
						msg.setAttribute(key, (JSONArray) obj);
					} else {
						EMLog.e("msg", "unknow additonal msg attr:" + obj.getClass().getName());
					}
				}
			}

			return msg;
		} catch (Exception e) {
			e.printStackTrace();
			return null;
		}
	}
}
