/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import com.hyphenate.EMCallBack;
import com.hyphenate.EMChatRoomChangeListener;
import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.adapter.EMAChatRoom;
import com.hyphenate.chat.adapter.EMAChatRoomManager;
import com.hyphenate.chat.adapter.EMAChatRoomManagerListener;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.exceptions.HyphenateException;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

public class EMChatRoomManager {
    EMAChatRoomManager emaObject;
	private List<EMChatRoomChangeListener> chatRoomListeners = Collections.synchronizedList(new ArrayList<EMChatRoomChangeListener>());
	private ExecutorService threadPool = null;
	private List<EMChatRoom> chatRooms = Collections.synchronizedList(new ArrayList<EMChatRoom>());
	
	EMClient mClient;
	public EMChatRoomManager(EMClient client, EMAChatRoomManager manager) {
	    emaObject = manager;
	    emaObject.addListener(chatRoomListenerImpl);
		mClient = client;
		threadPool = Executors.newCachedThreadPool();
	}

	/**
	 * \~chinese
	* 注册聊天室事件监听
	* @param listener
	* 
	* \~english
	* register chat room change listener
	* @param listener
	*/
	public void addChatRoomChangeListener(EMChatRoomChangeListener listener) {
		chatRoomListeners.add(listener);
	}

    /**
     * \~chinese
     * 移除聊天室事件监听
	 * @param listener
     * @deprecated  使用 {@link EMChatRoomManager#removeChatRoomListener(EMChatRoomChangeListener)} 替代
	 *
	 * \~english
	 * remove chat room change listener
	 * @param listener
     * @deprecated  use {@link EMChatRoomManager#removeChatRoomListener(EMChatRoomChangeListener)} instead
	 */
    @Deprecated
	public void removeChatRoomChangeListener(EMChatRoomChangeListener listener) {
	    removeChatRoomListener(listener);
    }

	/**
	 * \~chinese
	 * 移除聊天室事件监听
	 * @param listener
	 *
	 * \~english
	 * remove chat room change listener
	 * @param listener
	 */
	public void removeChatRoomListener(EMChatRoomChangeListener listener)
	{
		chatRoomListeners.remove(listener);
	}

	/**
	 * \~chinese
	 * 加入聊天室,异步方法
	 * @param roomId 
     * @param callback
	 * 
	 * \~english
     * join the chat room asynchronously
	 * @param roomId chat room id to join
	 * @param callback callback for success or fail 
	 */
	public void joinChatRoom(final String roomId, final EMValueCallBack<EMChatRoom> callback) {
		threadPool.submit(new Runnable() {                                         

			@Override                                                              
			public void run() { 
				EMAError error =  new EMAError();
				EMAChatRoom r = EMChatRoomManager.this.emaObject.joinChatRoom(roomId, error);
				EMChatRoom room = new EMChatRoom(r);
				if (error.errCode() == EMAError.EM_NO_ERROR || error.errCode() == EMAError.CHATROOM_ALREADY_JOINED) {
					callback.onSuccess(room);
				} else {
					callback.onError(error.errCode(), error.errMsg());
				}
			}
		});

	}
	
	/**
	* \~chinese
	* 离开聊天室,异步方法
	* @param roomId 
	* 
	* \~english
	* leave the chat room asynchronously
	* @param roomId 
	*/
	public void leaveChatRoom(final String roomId) {
		EMChatRoom room = getChatRoom(roomId);
		if (room == null) {
		    return;
		}

        EMOptions options = EMClient.getInstance().getOptions();

        boolean allowed = options.isChatroomOwnerLeaveAllowed();
        String owner = room.getOwner();
        if(!allowed && owner.equals(EMSessionManager.getInstance().getLastLoginUser())){
            return;
        }

        // temp workaround to delete conversation in case of leaving the room
        // but deleteConversation could be a time consuming operation?
        EMClient.getInstance().chatManager().deleteConversation(roomId,true);
		threadPool.submit(new Runnable() {                                         

			@Override                                                              
			public void run() { 
				EMAError error =  new EMAError();
				EMChatRoomManager.this.emaObject.leaveChatRoom(roomId, error);
			}
		});
	}

	/**
	 * \~chinese
	 * 分页从服务器获取聊天室
	 * 耗时操作
	 * @param pageNum 当前要拉取的页数，从1开始
	 * @param pageSize 当前要拉取页的数量
	 * @return  分页获取结果 {@link EMPageResult}
	 * @throws HyphenateException
	 *
	 * \~english
	 * fetch the chatroom in page synchronoulsy
	 * this is a time consuming ooperation
	 *
	 * @param pageNum --page number, start from 1
	 * @param pageSize -- size of page
	 * @return      fetch result, {@link EMPageResult}
	 * @throws HyphenateException
	 */
	public EMPageResult<EMChatRoom> fetchPublicChatRoomsFromServer(int pageNum, int pageSize) throws HyphenateException {
		EMAError error = new EMAError();
		EMPageResult<EMAChatRoom> result = emaObject.fetchChatroomsWithPage(pageNum, pageSize, error);
		handleError(error);

		List<EMAChatRoom> rooms = result.getData();
		int count = result.getPageCount();

		EMPageResult<EMChatRoom> ret = new EMPageResult<EMChatRoom>();
		List<EMChatRoom> data = new ArrayList<EMChatRoom>();
		for (EMAChatRoom room : rooms) {
			data.add(new EMChatRoom(room));
		}
		ret.setPageCount(count);
		ret.setData(data);

		chatRooms.clear();
		chatRooms.addAll(data);
		return ret;
	}

	/**
	* \~chinese
	* 翻页从服务器获取聊天室，在获取最后一页时，EMCursorResult.getCursor()返回一个空字符串。
	* 耗时操作
	* @param pageSize 一次取多少条
    * @param cursor 从这个游标位置开始取
    * @return   分页获取结果 {@link EMCursorResult}
    * @throws HyphenateException
    * @deprecated use fetchPublicChatRoomsFromServer(int pageNum, int pageSize)
	*
	* \~english
	* fetch the chat rooms in page synchronously, when fetch the last page, EMCursorResult.getCursor() return an empty string.
	* this is a time consuming operation
	*
	* @param pageSize how many chat room to get
	* @param cursor cursor position
	* @return fetch result {@link EMCursorResult}
	* @throws HyphenateException
	* @deprecated use fetchPublicChatRoomsFromServer(int pageNum, int pageSize)
	*/
	public EMCursorResult<EMChatRoom> fetchPublicChatRoomsFromServer(int pageSize, String cursor) throws HyphenateException {
		EMAError error = new EMAError();
		EMCursorResult<EMAChatRoom> cursorResult = emaObject.fetchChatroomsWithCursor(cursor, pageSize, error);
		handleError(error);
		
		EMCursorResult<EMChatRoom> ret = new EMCursorResult<EMChatRoom>();
		List<EMChatRoom> data = new ArrayList<EMChatRoom>();
		for (EMAChatRoom room : cursorResult.getData()) {
			data.add(new EMChatRoom(room));
		}
		ret.setCursor(cursorResult.getCursor());
		ret.setData(data);
		
		chatRooms.clear();
		chatRooms.addAll(data);
		return ret;
	}
	
	/**
     * \~chinese
     * 翻页从服务器获取聊天室
     * 耗时操作
     * @param pageSize 一次取多少条
     * @param cursor 从这个游标位置开始取
     * @param callback
     * @deprecated use asyncFetchPublicChatRoomsFromServer(final int pageNum, final int pageSize, final EMValueCallBack&#60;EMPageResult&#60;EMChatRoom&#62;&#62; callback)
     *
     * \~english
     * fetch the chatroom in page synchronoulsy
     * this is a time consuming ooperation
     *
     * @param pageSize how many chat room to get
     * @param cursor cursor position
	 * @param callback
     * @deprecated use asyncFetchPublicChatRoomsFromServer(final int pageNum, final int pageSize, final EMValueCallBack&#60;EMPageResult&#60;EMChatRoom&#62;&#62; callback)
     */
    public void asyncFetchPublicChatRoomsFromServer(final int pageSize, final String cursor, final EMValueCallBack<EMCursorResult<EMChatRoom>> callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    EMCursorResult<EMChatRoom> result = fetchPublicChatRoomsFromServer(pageSize, cursor);
                    callback.onSuccess(result);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }
	
    
    /**
     * \~chinese
     * 翻页从服务器获取聊天室
     * 耗时操作
     * @param pageNum 当前要拉取的页数，从1开始
     * @param pageSize 当前要拉取页的数量
     * @param callback
     *
     * \~english
     * fetch the chatroom in page asynchronously
     * this is a time consuming operation
     *
     * @param pageNum --page number, start from 1
     * @param pageSize -- size of page
     * @param callback
     */

	public void asyncFetchPublicChatRoomsFromServer(final int pageNum, final int pageSize, final EMValueCallBack<EMPageResult<EMChatRoom>> callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    EMPageResult<EMChatRoom> result = fetchPublicChatRoomsFromServer(pageNum, pageSize);
                    callback.onSuccess(result);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

	/**
	 * \~chinese
	 * 从服务器获取聊天室详情，默认不取成员列表
	 * 耗时操作
	 * @param roomId
	 * @return EMChatRoom
	 * @throws HyphenateException
	 *
	 * \~english
	 * fetch the chat room by room id, default behaviour is not fetch
	 *
	 * @param roomId
	 * @return EMChatRoom
	 * @throws HyphenateException
	 */
	public EMChatRoom fetchChatRoomFromServer(String roomId) throws HyphenateException {
		return fetchChatRoomFromServer(roomId, false);
	}
	
	/**
	 * \~chinese
	 * 从服务器获取聊天室详情，如果需要取成员列表，默认最多取200个成员，超出部分，请调用{@link EMChatRoomManager#fetchChatRoomMembers(String, String, int)}
	 * 耗时操作
	 * @param roomId 聊天室id
	 * @param fetchMembers 是否需要获取聊天室成员
	 * @return EMChatRoom
	 * @throws HyphenateException
	 *
	 * \~english
	 * fetch the chat room by room id. If fetchMembers is true, default maximum of fetched member count is 200,
	 * to continue fetch members, call {@link EMChatRoomManager#fetchChatRoomMembers(String, String, int)}
	 *
	 * @param roomId chat room id
	 * @param fetchMembers Whether you need to get chat room members
	 * @return EMChatRoom
	 * @throws HyphenateException
	 */
 	public EMChatRoom fetchChatRoomFromServer(String roomId, boolean fetchMembers)throws HyphenateException {
 	   EMAError error = new EMAError();
       EMAChatRoom room = emaObject.fetchChatroomSpecification(roomId, error, fetchMembers);
       handleError(error);
       return new EMChatRoom(room);
 	}
	
	/**
    * \~chinese
    * 从服务器获取聊天室详情
    * 耗时操作
    * @param roomId
	 * @param callback
    *
    * \~english
    * fetch the chat room by room id
    *
    * @param roomId
	 * @param callback
    */

    public void asyncFetchChatRoomFromServer(final String roomId, final EMValueCallBack<EMChatRoom> callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    EMChatRoom chatRoom = fetchChatRoomFromServer(roomId);
                    callback.onSuccess(chatRoom);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });   
    }
	
	/**
	 * \~chinese
	 * 从内存中获取聊天室
	 * 非耗时操作
	 * @param roomId
     * @return
	 *
	 * \~english
	 * get the chat room in the cache
	 * @param roomId
	 * @return
	 */
	public EMChatRoom getChatRoom(String roomId) {
		EMAChatRoom room = emaObject.getChatroom(roomId);
		if(room == null){
			return null;
		}
		return new EMChatRoom(room);
	}
	
	/**
	 * \~chinese
	 * 获取当前内存的聊天室
	 * @return
	 * 
	 * \~english
	 * get all chat rooms in the cache
	 * @return
	 */
	public List<EMChatRoom> getAllChatRooms() {
		return Collections.unmodifiableList(chatRooms);
	}

	EMAChatRoomManagerListener chatRoomListenerImpl = new EMAChatRoomManagerListener() {

		@Override
		public void onLeaveChatRoom(EMAChatRoom chatroom, int leaveReason) {
            EMClient.getInstance().chatManager().caches.remove(chatroom.getId());
		    synchronized (chatRoomListeners) {
			    try {
			        for (EMChatRoomChangeListener listener : chatRoomListeners) {
			            if (leaveReason == DESTROYED) {
			                listener.onChatRoomDestroyed(chatroom.getId(), chatroom.getName());
			            } else {
							listener.onRemovedFromChatRoom(leaveReason, chatroom.getId(), chatroom.getName(), EMClient.getInstance().getCurrentUser());
						}
			        }
			    } catch (Exception e) {
				    e.printStackTrace();
			    }
		    }
		}

		@Override
		public void onMemberJoinedChatRoom(EMAChatRoom chatroom, String member) {
		    synchronized (chatRoomListeners) {
			    try {
				    for (EMChatRoomChangeListener listener : chatRoomListeners) {
					    listener.onMemberJoined(chatroom.getId(), member);
				    }
			    } catch (Exception e) {
				    e.printStackTrace();
			    }
		    }
		}

		@Override
		public void onMemberLeftChatRoom(EMAChatRoom chatroom, String member) {
		    synchronized (chatRoomListeners) {
			    try {
			        for (EMChatRoomChangeListener listener : chatRoomListeners) {
			            listener.onMemberExited(chatroom.getId(), chatroom.getName(), member);
			        }
			    } catch (Exception e) {
				    e.printStackTrace();
			    }
		    }
		}

		public void onAddMuteList(EMAChatRoom chatRoom, List<String> mutes, long expireTime) {
			synchronized (chatRoomListeners) {
				try {
					for (EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onMuteListAdded(chatRoom.getId(), mutes, expireTime);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onRemoveMutes(EMAChatRoom chatRoom, List<String> mutes) {
			synchronized (chatRoomListeners) {
				try {
					for (EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onMuteListRemoved(chatRoom.getId(), mutes);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onWhiteListAdded(EMAChatRoom chatRoom, List<String> members) {
			synchronized (chatRoomListeners) {
				try {
					for(EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onWhiteListAdded(chatRoom.getId(), members);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onWhiteListRemoved(EMAChatRoom chatRoom, List<String> members) {
			synchronized (chatRoomListeners) {
				try {
					for(EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onWhiteListRemoved(chatRoom.getId(), members);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onAllMemberMuteStateChanged(EMAChatRoom chatRoom, final boolean isMuted) {
			synchronized (chatRoomListeners) {
				try {
					for(EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onAllMemberMuteStateChanged(chatRoom.getId(), isMuted);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onAddAdmin(EMAChatRoom chatRoom, String admin) {
			synchronized (chatRoomListeners) {
				try {
					for (EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onAdminAdded(chatRoom.getId(), admin);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onRemoveAdmin(EMAChatRoom chatRoom, String admin) {
			synchronized (chatRoomListeners) {
				try {
					for (EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onAdminRemoved(chatRoom.getId(), admin);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		public void onOwnerChanged(EMAChatRoom chatRoom, String newOwner, String oldOwner) {
			synchronized (chatRoomListeners) {
				try {
					for (EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onOwnerChanged(chatRoom.getId(), newOwner, oldOwner);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onAnnouncementChanged(EMAChatRoom chatRoom, String announcement) {
			synchronized (chatRoomListeners) {
				try {
					for (EMChatRoomChangeListener listener : chatRoomListeners) {
						listener.onAnnouncementChanged(chatRoom.getId(), announcement);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}
	};


	/**
	 * \~chinese
	 * 创建聊天室，聊天室最大人数上限10000
	 * @param subject           名称
	 * @param description       描述
	 * @param welcomeMessage    邀请成员加入聊天室的消息
	 * @param maxUserCount      允许加入聊天室的最大成员数
	 * @param members           邀请加入聊天室的成员列表
	 * @return EMChatRoom 聊天室
     * @throws HyphenateException
	 *
	 * \~english
	 * create chat room, maximum number of 10,000
	 * @param subject           subject
	 * @param description       description
	 * @param welcomeMessage    welcome message to invite member join the chat room
	 * @param maxUserCount      max allowed member limit
	 * @param members           invite members to join
	 * @return EMChatRoom
	 * @throws HyphenateException
     */
	public EMChatRoom createChatRoom(String subject, String description, String welcomeMessage,
	                                 int maxUserCount, List<String> members)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.createChatRoom(subject, description, welcomeMessage, EMChatRoom.EMChatRoomStyle.EMChatRoomStylePublicOpenJoin.ordinal(), maxUserCount, members, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 创建聊天室，聊天室最大人数上限10000。异步方法
	 * @param subject           名称
	 * @param description       描述
	 * @param welcomeMessage    邀请成员加入聊天室的消息
	 * @param maxUserCount      允许加入聊天室的最大成员数
	 * @param members           邀请加入聊天室的成员列表
	 * @param callBack          程序回调函数，onSuccess会包含该聊天室
	 *
	 * \~english
	 * create chat room, maximum number of 10,000
	 * @param subject           subject
	 * @param description       description
	 * @param welcomeMessage    welcome message to invite member join the chat room
	 * @param maxUserCount      max allowed member limit
	 * @param members           members to join
	 */
	public void asyncCreateChatRoom(final String subject, final String description, final String welcomeMessage,
	                                final int maxUserCount, final List<String> members, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(createChatRoom(subject, description, welcomeMessage, maxUserCount, members));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 销毁聊天室，需要owner权，同步方法
	 * @param chatRoomId
	 * @throws HyphenateException
	 *
	 * \~english
	 * destroy chat room, owner‘s authority is required, synchronized operation
	 * @param chatRoomId
	 * @throws HyphenateException
     */
	public void destroyChatRoom(String chatRoomId)
			throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.destroyChatroom(chatRoomId, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 销毁聊天室，需要owner权，同步方法
	 * @param chatRoomId
	 * @param callBack
	 *
	 * \~english
	 * destroy chat room, owner‘s authority is required, asynchronous method
	 * @param chatRoomId
	 * @param callBack
	 */
	public void asyncDestroyChatRoom(final String chatRoomId, final EMCallBack callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					destroyChatRoom(chatRoomId);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 修改聊天室标题，需要owner权，同步方法
	 * @param chatRoomId
	 * @param newSubject
	 * @return
	 * @throws HyphenateException
	 *
	 * \~english
	 * change chat room subject, owner‘s authority is required, synchronized operation
	 * @param chatRoomId
	 * @param newSubject
	 * @return
	 * @throws HyphenateException
     */
	public EMChatRoom changeChatRoomSubject(String chatRoomId, String newSubject)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.changeChatroomSubject(chatRoomId, newSubject, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 修改聊天室标题，需要owner权，同步方法
	 * @param chatRoomId
	 * @param newSubject
	 *
	 * \~english
	 * change chat room subject, owner‘s authority is required, asynchronous method
	 * @param chatRoomId
	 * @param newSubject
	 */
	public void asyncChangeChatRoomSubject(final String chatRoomId, final String newSubject, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(changeChatRoomSubject(chatRoomId, newSubject));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 修改群描述信息，需要owner权，同步方法
	 * @param chatRoomId
	 * @param newDescription
	 * @return
	 * @throws HyphenateException
	 *
	 * \~english
	 * change chat room description, owner‘s authority is required, synchronized operation
	 * @param chatRoomId
	 * @param newDescription
	 * @return
	 * @throws HyphenateException
     */
	public EMChatRoom changeChatroomDescription(String chatRoomId, String newDescription)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.changeChatroomDescription(chatRoomId, newDescription, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 修改群描述信息，需要owner权，同步方法
	 * @param chatRoomId
	 * @param newDescription
	 *
	 * \~english
	 * change chat room description, owner‘s authority is required, asynchronous method
	 * @param chatRoomId
	 * @param newDescription
	 * @param callBack
	 */
	public void asyncChangeChatroomDescription(final String chatRoomId, final String newDescription, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(changeChatroomDescription(chatRoomId, newDescription));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 获取聊天室成员列表， 获取最后一页成员列表时，EMCursorResult.getCursor()返回一个空字符串.
	 * @param chatRoomId
	 * @param cursor
	 * @param pageSize
	 * @return
	 * @throws HyphenateException
	 *
	 * \~english
	 * fetch chat room member list, synchronized operation,  for the last page, EMCursorResult.getCursor() return empty string.
	 * @param chatRoomId
	 * @param cursor
	 * @param pageSize
	 * @return
	 * @throws HyphenateException
     */
	public EMCursorResult<String> fetchChatRoomMembers(String chatRoomId, String cursor, int pageSize)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMCursorResult result = emaObject.fetchChatroomMembers(chatRoomId, cursor, pageSize, error);
		handleError(error);
		return result;
	}

	/**
	 * \~chinese
	 * 获取聊天室成员列表
	 * @param chatRoomId
	 * @param cursor
	 * @param pageSize
	 * @param callBack
	 *
	 * \~english
	 * fetch chat room member list, asynchronous method
	 * @param chatRoomId
	 * @param cursor
	 * @param pageSize
	 * @param callBack
	 */

	public void asyncFetchChatRoomMembers(final String chatRoomId, final String cursor, final int pageSize, final EMValueCallBack<EMCursorResult<String>> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchChatRoomMembers(chatRoomId, cursor, pageSize));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 禁止聊天室成员发言，需要聊天室拥有者或者管理员权限
	 * @param chatRoomId
	 * @param muteMembers 禁言的用户列表
	 * @param duration 禁言的时间，单位是毫秒
	 * @return
	 * @throws HyphenateException
	 *
	 * \~english
	 * Stop members post messages for some time, entry value is mute duration, need owner or administrator's authority to access.
	 * @param chatRoomId
	 * @param muteMembers mute member list
	 * @param duration mute duration
	 * @return
	 * @throws HyphenateException
     */
	public EMChatRoom muteChatRoomMembers(String chatRoomId, List<String> muteMembers, long duration)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.muteChatroomMembers(
				chatRoomId, muteMembers, duration, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 禁止聊天室成员发言，需要聊天室拥有者或者管理员权限
	 * @param chatRoomId
	 * @param muteMembers   需要禁言的群组成员
	 * @param duration 禁言持续时间，单位是毫秒
	 * @param callBack
	 *
	 * \~english
	 * Stop members post message for some time, entry value is mute duration, need owner or administrator's authority to access.
	 * @param chatRoomId
	 * @param muteMembers mute group members
	 * @param duration mute duration, in milli-seconds
	 * @param callBack
	 */
	public void asyncMuteChatRoomMembers(final String chatRoomId, final List<String> muteMembers, final long duration, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(muteChatRoomMembers(chatRoomId, muteMembers, duration));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 取消禁言，需要聊天室拥有者或者管理员权限
	 * @param chatRoomId
	 * @param members
	 * @return
	 * @throws HyphenateException
	 *
	 * \~english
	 * unmute chat room members, need owner or administrator's authority to access.
	 * @param chatRoomId
	 * @param members
	 * @return
	 * @throws HyphenateException
     */
	public EMChatRoom unMuteChatRoomMembers(String chatRoomId, List<String> members)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.unmuteChatRoomMembers(chatRoomId, members, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 取消禁言，需要聊天室拥有者或者管理员权限，异步方法
	 * @param chatRoomId
	 * @param members
	 * @param callBack
	 *
	 * \~english
	 * cancel mute chat room member, need owner or administrator's authority to access.
	 * @param chatRoomId
	 * @param members
	 * @param callBack
	 */
	public void asyncUnMuteChatRoomMembers(final String chatRoomId, final List<String> members,
	                                             final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(unMuteChatRoomMembers(chatRoomId, members));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 转移聊天室的所有权，需要聊天室拥有者权限
	 * @param chatRoomId
	 * @param newOwner
	 * @return
	 * @throws HyphenateException
	 *
	 * \~english
	 * transfer chat room ownership to others, need owner's authority to access.
	 * @param chatRoomId
	 * @param newOwner
	 * @return
	 * @throws HyphenateException
	 */
	public EMChatRoom changeOwner(String chatRoomId, String newOwner)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.transferChatroomOwner(chatRoomId, newOwner, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 转移聊天室的所有权
	 * @param chatRoomId
	 * @param newOwner
	 * @param callBack
	 *
	 * \~english
	 * transfer chat room ownership to others, need owner's authority to access.
	 * @param chatRoomId
	 * @param newOwner
	 * @param callBack
	 */
	public void asyncChangeOwner(final String chatRoomId, final String newOwner, final EMValueCallBack<EMChatRoom> callBack)
			throws HyphenateException {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(changeOwner(chatRoomId, newOwner));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 为聊天室添加管理员，需要拥有者权限
	 * @param chatRoomId
	 * @param admin
	 * @return
	 * @throws HyphenateException
	 *
	 * \~english
	 * add chat room administrator, need owner's authority to access.
	 * @param chatRoomId
	 * @param admin
	 * @return
	 * @throws HyphenateException
     */
	public EMChatRoom addChatRoomAdmin(String chatRoomId, String admin)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.addChatroomAdmin(chatRoomId, admin, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 为聊天室添加管理员，需要拥有者权限
	 * @param chatRoomId
	 * @param admin
	 * @param callBack
	 *
	 * \~english
	 * add chat room administrator, need owner's authority to access.
	 * @param chatRoomId
	 * @param admin
	 * @param callBack
	 */
	public void asyncAddChatRoomAdmin(final String chatRoomId, final String admin, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(addChatRoomAdmin(chatRoomId, admin));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 删除聊天室管理员，需要拥有者权限
	 * @param chatRoomId
	 * @param admin
	 * @return
	 * @throws HyphenateException
	 *
	 * \~english
	 * remove chat room administrator, need owner's authority to access.
	 * @param chatRoomId
	 * @param admin
	 * @return
	 * @throws HyphenateException
     */
	public EMChatRoom removeChatRoomAdmin(String chatRoomId, String admin)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.removeChatRoomAdmin(chatRoomId, admin, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 删除聊天室管理员，需要拥有着权限
	 * @param chatRoomId
	 * @param admin
	 * @param callBack
	 *
	 * \~english
	 * remove chat room administrator, need owner's authority to access.
	 * @param chatRoomId
	 * @param admin
	 * @param callBack
	 */
	public void asyncRemoveChatRoomAdmin(final String chatRoomId, final String admin,
	                                final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(removeChatRoomAdmin(chatRoomId, admin));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 获取聊天室的禁言列表，需要拥有者或者管理员权限
	 * @param chatRoomId
	 * @param pageNum
	 * @param pageSize
	 * @return  Map.entry.key 是禁言的成员id，Map.entry.value是禁言动作失效的时间，单位是毫秒
	 * @throws HyphenateException
	 *
	 * \~english
	 * fetch mute list, which contains mute members and mute time, need owner or administrator's authority to access.
	 * @param chatRoomId
	 * @param pageNum
	 * @param pageSize
	 * @return Map.entry.key is username of mute action, Map.entry.value is expired time of banning post action, in milli-seconds
	 * @throws HyphenateException
     */
	public Map<String, Long> fetchChatRoomMuteList(String chatRoomId, int pageNum, int pageSize)
			throws HyphenateException {
		EMAError error = new EMAError();
		Map<String, Long> members = emaObject.fetchChatRoomMuteList(chatRoomId, pageNum, pageSize, error);
		handleError(error);
		return members;
	}

	/**
	 * \~chinese
	 * 获取聊天室的禁言列表，需要拥有者或者管理员权限
	 * @param chatRoomId
	 * @param pageNum
	 * @param pageSize
	 * @param callBack
	 *
	 * \~english
	 * fetch mute list, which contains mute members and mute time, need owner or administrator's authority to access.
	 * @param chatRoomId
	 * @param pageNum
	 * @param pageSize
	 * @param callBack
	 */
	public void asyncFetchChatRoomMuteList(final String chatRoomId, final int pageNum, final int pageSize,
	                                       final EMValueCallBack<Map<String, Long>> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchChatRoomMuteList(chatRoomId, pageNum, pageSize));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 删除聊天室成员，需要拥有者或者管理员权限
	 * @param chatRoomId
	 * @param members
	 * @return
	 * @throws HyphenateException
	 *
	 * \~english
	 * remove chat room members, need owner or administrator's authority to access.
	 * @param chatRoomId
	 * @param members
	 * @return
	 * @throws HyphenateException
     */
	public EMChatRoom removeChatRoomMembers(String chatRoomId, List<String> members)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.removeChatRoomMembers(chatRoomId, members, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 删除聊天室成员，需要拥有者或者管理员权限
	 * @param chatRoomId
	 * @param members
	 * @param callBack
	 *
	 * \~english
	 * remove chat room members, need owner or administrator's authority to access.
	 * @param chatRoomId
	 * @param members
	 * @param callBack
	 */
	public void asyncRemoveChatRoomMembers(final String chatRoomId, final List<String> members,
	                                       final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(removeChatRoomMembers(chatRoomId, members));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 添加成员到黑名单，禁止成员继续加入聊天室，需要拥有者或者管理员权限
	 * @param chatRoomId
	 * @param members
	 * @return
	 * @throws HyphenateException
	 *
	 * \~english
	 * add members to chat room's black list, need owner or administrator's authority to access.
	 * @param chatRoomId
	 * @param members
	 * @return
	 * @throws HyphenateException
     */
	public EMChatRoom blockChatroomMembers(String chatRoomId, List<String> members)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.blockChatroomMembers(chatRoomId, members, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 添加成员到黑名单，禁止成员继续加入聊天室，需要拥有者或者管理员权限
	 * @param chatRoomId
	 * @param members
	 * @param callBack
	 *
	 * \~english
	 * add members to chat room's black list, need owner or administrator's authority to access.
	 * @param chatRoomId
	 * @param members
	 * @param callBack
	 */
	public void asyncBlockChatroomMembers(final String chatRoomId, final List<String> members, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(blockChatroomMembers(chatRoomId, members));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 将成员从黑名单种移除，需要拥有者或者管理员权限
	 * @param chatRoomId
	 * @param members
	 * @return
	 * @throws HyphenateException
	 *
	 * \~english
	 * remove members from chat room's black list, need owner or administrator's authority to access.
	 * @param chatRoomId
	 * @param members
	 * @return
	 * @throws HyphenateException
     */
	public EMChatRoom unblockChatRoomMembers(String chatRoomId, List<String> members)
			throws HyphenateException {
		EMAError error = new EMAError();
		EMAChatRoom chatRoom = emaObject.unblockChatRoomMembers(chatRoomId, members, error);
		handleError(error);
		return new EMChatRoom(chatRoom);
	}

	/**
	 * \~chinese
	 * 将成员从黑名单种移除，需要拥有者或者管理员权限
	 * @param chatRoomId
	 * @param members
	 * @param callBack
	 *
	 * \~english
	 * remove members from chat room's black list, need owner or administrator's authority to access.
	 * @param chatRoomId
	 * @param members
	 * @param callBack
	 */
	public void asyncUnBlockChatRoomMembers(final String chatRoomId, final List<String> members,
	                                        final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(unblockChatRoomMembers(chatRoomId, members));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 获取群组黑名单列表，分页显示，需要拥有者或者管理员权限
	 * @param chatRoomId
	 * @param pageNum
	 * @param pageSize
	 * @return
	 * @throws HyphenateException
	 *
	 * \~english
	 * fetch chat room black list members in pages, need owner or administrator's authority to access.
	 * @param chatRoomId
	 * @param pageNum
	 * @param pageSize
	 * @return
	 * @throws HyphenateException
     */
	public List<String> fetchChatRoomBlackList(String chatRoomId, int pageNum, int pageSize)
			throws HyphenateException {
		EMAError error = new EMAError();
		List<String> blockList = emaObject.fetchChatRoomBlackList(chatRoomId, pageNum, pageSize, error);
		handleError(error);
		return blockList;
	}

	/**
	 * \~chinese
	 * 获取群组黑名单列表，分页显示，需要拥有者或者管理员权限
	 * @param chatRoomId
	 * @param pageNum
	 * @param pageSize
	 * @param callBack
	 *
	 * \~english
	 * fetch chat room black list members in pages, need owner or administrator's authority to access.
	 * @param chatRoomId
	 * @param pageNum
	 * @param pageSize
	 * @param callBack
	 */
	public void asyncFetchChatRoomBlackList(final String chatRoomId, final int pageNum, final int pageSize,
	                                        final EMValueCallBack<List<String>> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchChatRoomBlackList(chatRoomId, pageNum, pageSize));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 添加用户到白名单
	 * @param chatRoomId 聊天室id
	 * @param members 成员id列表
	 * @throws HyphenateException
	 *
	 * \~english
	 * add members to whitelist
	 * @param chatRoomId chat room id
	 * @param members member list
	 * @throws HyphenateException
	 */
	public void addToChatRoomWhiteList(final String chatRoomId, final List<String> members, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					EMAChatRoom chatRoom = emaObject.addToWhiteList(chatRoomId, members, error);
					handleError(error);
					callBack.onSuccess(new EMChatRoom(chatRoom));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 将用户从白名单移除
	 * @param chatRoomId 聊天室id
	 * @param members 成员id列表
	 * @throws HyphenateException
	 *
	 * \~english
	 * remove members from whitelist
	 * @param chatRoomId chat room id
	 * @param members member list
	 * @throws HyphenateException
	 */
	public void removeFromChatRoomWhiteList(final String chatRoomId, final List<String> members, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					EMAChatRoom chatRoom = emaObject.removeFromWhiteList(chatRoomId, members, error);
					handleError(error);
					callBack.onSuccess(new EMChatRoom(chatRoom));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 检查自己是否在白名单中
	 * @param chatRoomId 聊天室id
	 * @throws HyphenateException
	 *
	 * \~english
	 * remove members from whitelist
	 * @param chatRoomId chat room id
	 * @throws HyphenateException
	 */
	public void checkIfInChatRoomWhiteList(final String chatRoomId, EMValueCallBack<Boolean> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					Boolean re = emaObject.checkIfInWhiteList(chatRoomId, error);
					handleError(error);
					callBack.onSuccess(re);

				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});

	}

	/**
	 * \~chinese
	 * 从服务器获取白名单成员列表
	 * @param chatRoomId 聊天室id
	 * @throws HyphenateException
	 *
	 * \~english
	 * fetch whitelist
	 * @param chatRoomId chat room id
	 * @throws HyphenateException
	 */
	public void fetchChatRoomWhiteList(final String chatRoomId, final EMValueCallBack<List<String>> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					List<String> result = emaObject.fetchChatRoomWhiteList(chatRoomId, error);
					handleError(error);
					callBack.onSuccess(result);
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 禁言所有成员
	 * @param chatRoomId 聊天室id
	 *
	 * \~english
	 * mute all members
	 * @param chatRoomId chat room id
	 */
	public void muteAllMembers(final String chatRoomId, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					EMAChatRoom chatRoom = emaObject.muteAllMembers(chatRoomId, error);
					handleError(error);
					callBack.onSuccess(new EMChatRoom(chatRoom));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 解除所有成员禁言
	 * @param chatRoomId 聊天室id
	 *
	 * \~english
	 * unmute all members
	 * @param chatRoomId chat room id
	 */
	public void unmuteAllMembers(final String chatRoomId, final EMValueCallBack<EMChatRoom> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					EMAError error = new EMAError();
					EMAChatRoom chatRoom = emaObject.unmuteAllMembers(chatRoomId, error);
					handleError(error);
					callBack.onSuccess(new EMChatRoom(chatRoom));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 更新聊天室公告
	 * @param chatRoomId 聊天室id
	 * @param announcement 公告内容
	 * @throws HyphenateException
	 *
	 * \~english
	 * update chat room announcement
	 * @param chatRoomId chat room id
	 * @param announcement announcement content
	 * @throws HyphenateException
	 */
	public void updateChatRoomAnnouncement(String chatRoomId, String announcement) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.updateChatRoomAnnouncement(chatRoomId, announcement, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 异步更新聊天室公告
	 * @param chatRoomId 聊天室id
	 * @param announcement 公告内容
	 * @param callBack 回调
	 * @param callBack
	 *
	 * \~english
	 * async update chat room announcement
	 * @param chatRoomId chat room id
	 * @param announcement announcement content
	 * @param callBack callback
	 * @param callBack
	 */
	public void asyncUpdateChatRoomAnnouncement(final String chatRoomId, final String announcement, final EMCallBack callBack){
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					updateChatRoomAnnouncement(chatRoomId, announcement);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从服务器获取聊天室公告内容
	 * @param chatRoomId 聊天室id
	 * @return
	 * @throws HyphenateException
	 *
	 * \~english
	 * get the chatroom announcement from server
	 * @param chatRoomId chatroom id
	 * @return
	 * @throws HyphenateException
	 */
	public String fetchChatRoomAnnouncement(String chatRoomId) throws HyphenateException {
		EMAError error = new EMAError();
		String result = emaObject.fetchChatRoomAnnouncement(chatRoomId, error);
		handleError(error);
		return result;
	}

	/**
	 * \~chinese
	 * 从服务器获取聊天室公告内容
	 * @param chatRoomId 聊天室id
	 * @param callBack  回调
	 * @param callBack
	 *
	 * \~english
	 * get the chatroom announcement from server
	 * @param chatRoomId chatroom id
	 * @param callBack  callback
	 * @param callBack
	 */
	public void asyncFetchChatRoomAnnouncement(final String chatRoomId, final EMValueCallBack<String> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchChatRoomAnnouncement(chatRoomId));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	// ============================= chatRoom_reform new add api end
	private void handleError(EMAError error)  throws HyphenateException {
	    if (error.errCode() != EMAError.EM_NO_ERROR) {
	        throw new HyphenateException(error);
	    }
	}
	
	void onLogout() {
	    chatRoomListeners.clear();
	}
}
