/************************************************************
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2013-2015 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import android.graphics.BitmapFactory.Options;
import android.net.Uri;
import android.text.TextUtils;
import android.util.Log;

import com.hyphenate.EMCallBack;
import com.hyphenate.EMConversationListener;
import com.hyphenate.EMError;
import com.hyphenate.EMMessageListener;
import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.EMConversation.EMConversationType;
import com.hyphenate.chat.EMMessage.Status;
import com.hyphenate.chat.EMMessage.Type;
import com.hyphenate.chat.adapter.EMAChatManager;
import com.hyphenate.chat.adapter.EMAChatManagerListener;
import com.hyphenate.chat.adapter.EMAConversation;
import com.hyphenate.chat.adapter.EMAConversation.EMAConversationType;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.chat.adapter.EMAGroupReadAck;
import com.hyphenate.chat.adapter.message.EMAFileMessageBody;
import com.hyphenate.chat.adapter.message.EMAMessage;
import com.hyphenate.chat.adapter.message.EMAMessageBody;
import com.hyphenate.chat.core.EMAdvanceDebugManager;
import com.hyphenate.cloud.EMCloudOperationCallback;
import com.hyphenate.cloud.EMHttpClient;
import com.hyphenate.exceptions.HyphenateException;
import com.hyphenate.notification.core.EMNotificationHelper;
import com.hyphenate.util.EMLog;
import com.hyphenate.util.ImageUtils;
import com.hyphenate.util.PathUtil;
import com.hyphenate.util.UriUtils;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Hashtable;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;

import static com.hyphenate.chat.EMMessage.self;

/**
 * send a message
 * EMMessage msg = EMMessage;
 * EMClient.getInstance().chatManager().sendMessage(msg);
 */
public class EMChatManager {

	EMAChatManager emaObject;

	private static final String TAG = "EMChatManager";

	private final static String INTERNAL_ACTION_PREFIX = "em_";

	EMClient mClient;

	Map<String, EMConversation.MessageCache> caches = new Hashtable<String, EMConversation.MessageCache>();

	protected EMChatManager(){}

	protected EMChatManager(EMClient client, EMAChatManager manager) {
		mClient = client;
		emaObject = manager;

		emaObject.addListener(chatManagerListenerImpl);
	}

	private List<EMMessageListener> messageListeners = new CopyOnWriteArrayList<EMMessageListener>();
	private List<EMConversationListener> conversationListeners = Collections.synchronizedList(new ArrayList<EMConversationListener>());

	EMAChatManagerListener chatManagerListenerImpl = new EMAChatManagerListener() {
//		private final Object lockObj = new Object();
//
//        List<EMMessageListener> cloneSyncedList(List<EMMessageListener> list) {
//            if (list == null) {
//                return new ArrayList<>();
//            } else {
//                synchronized (lockObj) {
//                    return list.subList(0, list.size());
//                }
//            }
//        }

		@Override
		public void onReceiveMessages(final List<EMAMessage> messages) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
					List<EMMessage> msgs = new ArrayList<EMMessage>();
					for (EMAMessage msg : messages) {
						msgs.add(new EMMessage(msg));
					}

					List<EMMessage> remainingMsgs = new ArrayList<EMMessage>();
					for (EMMessage msg : msgs) {
				    	/*if(msg.getChatType() == EMMessage.ChatType.ChatRoom){
                            EMChatRoom room = EMClient.getInstance().chatroomManager().getChatRoom(msg.conversationId());
                            if(room == null){
                            	continue;
                            }
				    	}*/

	                    EMConversation conv = getConversation(msg.conversationId(), EMConversation.msgType2ConversationType(msg.getFrom(), msg.getChatType()), false);
	                    if(conv == null){
	                    	EMLog.d(TAG, "no conversation");
	                    	continue;
	                    }
	                    
	                    // CMD not put into cache
	                    if (msg.getType() != Type.CMD) {
	                        conv.getCache().addMessage(msg);
	                    }
	                    remainingMsgs.add(msg);
				    }
				    
				    if(remainingMsgs.size() <= 0){
						EMLog.d(TAG, "no remainingMsgs");
				    	return;
				    }

                    try {
                        for (EMMessageListener l : messageListeners) {
                        	EMLog.d(TAG, "onMessageReceived： " + l);
                            l.onMessageReceived(remainingMsgs);
                        }
                    } catch (Exception e) {
                    	EMLog.d(TAG, "onMessageReceived has problem: " + e.getMessage());
                        e.printStackTrace();
                    }
				}
			});
		}

		@Override
		public void onReceiveCmdMessages(final List<EMAMessage> messages) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
					List<EMMessage> msgs = new ArrayList<EMMessage>();
					for (EMAMessage msg : messages) {
						EMMessage message = new EMMessage(msg);
						String action = ((EMCmdMessageBody)message.getBody()).action();
						if(isAdvanceDebugMessage(action)) {
							EMAdvanceDebugManager.getInstance().handleDebugMessage(message, EMAdvanceDebugManager.Type.valueOf(action));
						} else {
							msgs.add(message);
						}
					}
                    try {
                        for (EMMessageListener l : messageListeners) {
                            l.onCmdMessageReceived(msgs);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }

				}
			});
		}

		@Override
		public void onMessageStatusChanged(final EMAMessage message, final EMAError error) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
                    try {
                        EMMessage msg = new EMMessage(message);
                        for (EMMessageListener l : messageListeners) {
                            l.onMessageChanged(msg, null);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
				}
			});
		}

		@Override
	    public void onMessageAttachmentsStatusChanged(final EMAMessage message, final EMAError error) {
            try {
                EMMessage msg = new EMMessage(message);
                for (EMMessageListener l : messageListeners) {
                    l.onMessageChanged(msg, null);
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
		}
		
		
        @Override
        public void onReceiveRecallMessages(final List<EMAMessage> messages) {
            mClient.executeOnMainQueue(new Runnable() {

                @Override
                public void run() {
                    List<EMMessage> msgs = new ArrayList<EMMessage>();
                    for (EMAMessage msg : messages) {
                        msgs.add(new EMMessage(msg));
                        getConversation(msg.conversationId()).getCache().removeMessage(msg.msgId());
                    }

                    try {
                        for (EMMessageListener l : messageListeners) {
                            l.onMessageRecalled(msgs);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            });
        }
		@Override
		public void onReceiveHasReadAcks(final List<EMAMessage> messages) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
                    List<EMMessage> msgs = new ArrayList<EMMessage>();
                    for (EMAMessage msg : messages) {
                        msgs.add(new EMMessage(msg));
                    }

                    try {
                        for (EMMessageListener l : messageListeners) {
                            l.onMessageRead(msgs);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
				}
			});
		}

		@Override
		public void onReceiveHasDeliveredAcks(final List<EMAMessage> messages) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
					List<EMMessage> msgs = new ArrayList<EMMessage>();
					for (EMAMessage msg : messages) {
						msgs.add(new EMMessage(msg));
					}

                    try {
                        for (EMMessageListener l : messageListeners) {
                            l.onMessageDelivered(msgs);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
				}
			});
		}

		@Override
		public void onUpdateConversationList(final List<EMAConversation> conversations) {
			mClient.executeOnMainQueue(new Runnable() {

				@Override
				public void run() {
					EMLog.d(TAG, "onUpdateConversationList");
                    synchronized (conversationListeners) {
	                    try {
	                        for (EMConversationListener l : conversationListeners) {
	                            l.onCoversationUpdate();
	                        }
	                    } catch (Exception e) {
		                    e.printStackTrace();
	                    }
                    }
				}
			});
		}

		@Override
		public void onReceivePrivateMessages(List<EMAMessage> messages) {
			mClient.executeOnMainQueue(new Runnable() {
				@Override
				public void run() {
					EMLog.d(TAG, "onPrivateMessages");
					for (EMAMessage msg : messages) {
						EMMessage message = new EMMessage(msg);
						EMNotificationHelper.getInstance().analyzeCmdMessage(message);
					}
				}
			});
		}

		@Override
		public void onReceiveReadAcksForGroupMessage(List<EMAGroupReadAck> acks) {
            mClient.executeOnMainQueue(new Runnable() {
				@Override
				public void run() {
					EMLog.d(TAG, "onReceiveReadAcksForGroupMessage");

					List<EMGroupReadAck> groupAcks = new ArrayList<EMGroupReadAck>();
					for (EMAGroupReadAck ack : acks) {
						groupAcks.add(new EMGroupReadAck(ack));
					}

					try {
						for (EMMessageListener l : messageListeners) {
							l.onGroupMessageRead(groupAcks);
						}
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
			});
		}

		@Override
		public void onUpdateGroupAcks() {
        	mClient.executeOnMainQueue(new Runnable() {
				@Override
				public void run() {
					EMLog.d(TAG, "onUpdateGroupAcks");
					try {
						for (EMMessageListener l : messageListeners) {
							l.onReadAckForGroupMessageUpdated();
						}
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
			});
		}

	};

	private boolean isAdvanceDebugMessage(String action){
		if(action.startsWith(INTERNAL_ACTION_PREFIX)){
			try {
				EMAdvanceDebugManager.Type.valueOf(action);
				return true;
			} catch (Exception e) {
				//if it is not a field of Type, throws exception
				e.printStackTrace();
			}
		}
		return false;
	}

	/**
	 * \~chinese
	 * 异步发送消息 如果是语音，图片类有附件的消息，sdk 会自动上传附件
	 *
	 * @param msg					待发送消息对象
	 *
	 * \~english
	 * send message asynchronously and you can set this message status callback through EMMessage.setMessageStatusCallback
	 *
	 * @param msg
	 */
	public void sendMessage(final EMMessage msg) {
		msg.makeCallbackStrong();

		boolean createConv = msg.getType() != Type.CMD;
		final EMConversation conv = getConversation(msg.conversationId(), EMConversation.msgType2ConversationType(msg.getTo(), msg.getChatType()), createConv);

		//add the message at first
		if (conv != null) {
			boolean exists = conv.getCache().getMessage(msg.getMsgId()) != null;
			if (!exists) {
				// sent message time is lastMsgTime + 1ms
				long lastMsgTime = System.currentTimeMillis();
				EMMessage lastMsg = conv.getLastMessage();
				if (lastMsg != null) {
					lastMsgTime = lastMsgTime < lastMsg.getMsgTime() ? lastMsg.getMsgTime() : lastMsgTime;
				}
				msg.setMsgTime(lastMsgTime + 1);

				conv.getCache().addMessage(msg);
			}
		}

		class HandleError {
			HandleError(int code, String desc) {
				EMMessage.EMCallbackHolder holder = msg.messageStatusCallBack;
				if (holder != null) {
					holder.onError(code, desc);
				}
			}
		}

    mClient.executeOnSendQueue(new Runnable() {
            
        @Override
        public void run() {
				try {
					String originPath = null;
					// If message body is image, check scale request, set size, and file name
					if (msg.getType() == Type.IMAGE) {
						//the default status is fail status,
						//which lead to message show fail and then show success in need scale image
						msg.setStatus(Status.INPROGRESS);
						EMImageMessageBody body = (EMImageMessageBody) msg.getBody();
						if (body == null) {
							new HandleError(EMError.GENERAL_ERROR, "Message body can not be null");
							return;
						}

						Uri filePathUri = body.getLocalUri();
						if(!UriUtils.isFileExistByUri(mClient.getContext(), filePathUri)) {
							new HandleError(EMError.FILE_INVALID, "File not exists or can not be read");
							return;
						}
						String localUri = filePathUri.toString();
						if(!body.isSendOriginalImage()) {
							String scaledImagePath = ImageUtils.getScaledImageByUri(mClient.getContext(), localUri);
							if(!TextUtils.equals(scaledImagePath, localUri)) {
							    originPath = localUri;
								long originalSize = ImageUtils.getFileLength(mClient.getContext(), localUri);
								long scaledSize = ImageUtils.getFileLength(mClient.getContext(), scaledImagePath);
								if (originalSize == 0) {
									EMLog.d(TAG, "original image size:" + originalSize);
									new HandleError(EMError.FILE_INVALID, "original image size is 0");
									return;
								}
								EMLog.d(TAG, "original image size:" + originalSize + " scaled image size:" + scaledSize
										+ " ratio:" + (int) (scaledSize / originalSize) + "%");
								localUri = scaledImagePath;
								body.setLocalUrl(UriUtils.getLocalUriFromString(localUri));
							}
							body.setFileName(ImageUtils.getFilename(mClient.getContext(), localUri));
						}
						// get image width and height
						Options options = ImageUtils.getBitmapOptions(mClient.getContext(), localUri);
						if(options != null) {
							int width = options.outWidth;
							int height = options.outHeight;
							body.setSize(width, height);
						}
					}else if (msg.getType() == Type.VIDEO){
						msg.setStatus(Status.INPROGRESS);
						EMVideoMessageBody body = (EMVideoMessageBody) msg.getBody();
						if (body == null) {
							new HandleError(EMError.GENERAL_ERROR, "Message body can not be null");
							return;
						}
						Uri filePathUri = body.getLocalUri();
						if(!UriUtils.isFileExistByUri(mClient.getContext(), filePathUri)) {
							new HandleError(EMError.FILE_INVALID, "File not exists or can not be read");
							return;
						}
						// get video width and height
						String thumbPath = UriUtils.getFilePath(EMClient.getInstance().getContext(), body.getLocalThumbUri());
						if(!TextUtils.isEmpty(thumbPath)) {
							Options options = ImageUtils.getBitmapOptions(thumbPath);
							int width = options.outWidth;
							int height = options.outHeight;
							body.setThumbnailSize(width, height);
						}
					}

					String oldId = msg.getMsgId();
					//set callback to replace old id
					setMessageSendCallback(msg, conv, oldId, originPath);
					emaObject.sendMessage(msg.emaObject);
				} catch (Exception e) {
					e.printStackTrace();
					new HandleError(EMError.GENERAL_ERROR, "send message failed");
				}
			}
		});
	}

	private void setMessageSendCallback(final EMMessage msg,
										final EMConversation conv,
										final String oldId,
										final String originImagePath) {
		if (msg == null) {
			return;
		}

		msg.setInnerCallback(new EMCallBack(){

			@Override
			public void onSuccess() {
				if (conv != null) {
					conv.getCache().removeMessage(oldId);
					conv.getCache().addMessage(msg);
				}
				if (originImagePath != null){
					if (msg.getBody() instanceof EMImageMessageBody) {
						Uri localUri = ((EMImageMessageBody)msg.getBody()).getLocalUri();
						String scaleImagePath = UriUtils.getUriString(localUri);
						EMLog.d(TAG, "origin: + " + originImagePath + ", scale:" + scaleImagePath);
						//if scaleImagePath is not origin image path, should delete scale image file
						if(scaleImagePath != null && !scaleImagePath.equals(originImagePath)){
                            Uri scaleImageUri = UriUtils.getLocalUriFromString(scaleImagePath);
                            //只删除私有目录下的缩放文件（正常情况下，需要压缩的文件均在私有目录下生成）
                            if(UriUtils.isFileExistByUri(EMClient.getInstance().getContext(), scaleImageUri)) {
                                String filePath = UriUtils.getFilePath(EMClient.getInstance().getContext(), scaleImageUri);
                                if(!TextUtils.isEmpty(filePath)) {
                                    File scaleFile = new File(filePath);
                                    if(scaleFile.exists()) {
                                        scaleFile.delete();
                                    }
                                }
                            }
						}
						((EMImageMessageBody)msg.getBody()).setLocalUrl(originImagePath);
						updateMessage(msg);
					}
				}
			}

			@Override
			public void onProgress(int progress, String status) {
			}

			@Override
			public void onError(int code, String error) {
			}
		});
	}

	/**
	 * \~chinese
	 * 发送消息已读回执
	 *
	 * @param to			接收方的用户名
	 * @param messageId		消息的ID
	 * @throws HyphenateException
	 *
	 * \~english
	 * send read ack to server
	 * @param to			the recipient id
	 * @param messageId		message id
	 * @throws HyphenateException
	 */
	public void ackMessageRead(String to, String messageId) throws HyphenateException {
		EMOptions chatOptions = EMClient.getInstance().getChatConfigPrivate().getOptions();
		if (!chatOptions.getRequireAck()) {
			EMLog.d(TAG, "chat option reqire ack set to false. skip send out ask msg read");
			return;
		}
		EMAMessage msg = emaObject.getMessage(messageId);
		if (msg != null) {
			emaObject.sendReadAckForMessage(msg);
		} else { // just for Xinju since there is no local storage
			EMAMessage _msg = EMAMessage.createReceiveMessage("", self(), null, EMMessage.ChatType.Chat.ordinal());
			_msg.setMsgId(messageId);
			_msg.setFrom(to);

			emaObject.sendReadAckForMessage(_msg);
		}
	}

	public void ackGroupMessageRead(String to, String messageId, String ext) throws HyphenateException {
		EMOptions chatOptions = EMClient.getInstance().getChatConfigPrivate().getOptions();
		if (!chatOptions.getRequireAck()) {
			EMLog.d(TAG, "chat option reqire ack set to false. skip send out ask msg read");
			return;
		}
		EMAMessage msg = emaObject.getMessage(messageId);
		if (msg != null) {
			if (msg.isNeedGroupAck()) {
				emaObject.sendReadAckForGroupMessage(msg, ext);
			} else {
				EMLog.d(TAG, "normal group message, do not ack it");
			}
		}
	}
	
    /**
     * \~chinese
     * 撤回发送成功的消息对象
     *
     * @param message 消息对象
     * @return
	 *
	 * \~english
	 * Recall the sent message
	 * @param message message
	 * @return
	 *
     */
    public void recallMessage(EMMessage message) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.recallMessage(message.emaObject, error);
        handleError(error);
        getConversation(message.getTo()).getCache().removeMessage(message.getMsgId());
    }

    public void aysncRecallMessage(final EMMessage message,
                                   final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {

            @Override
            public void run() {
                try {
                    recallMessage(message);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

	/**
	 * \~chinese
	 * 获取指定ID 的消息对象
	 *
	 * @param messageId		消息ID
	 * @return
	 *
	 * \~english
	 * get message through message id
	 *
	 * @param messageId 	message id
	 * @return
	 */
	public EMMessage getMessage(String messageId) {
		synchronized (caches) {
			for(EMConversation.MessageCache cache : caches.values()) {
				EMMessage msg = cache.getMessage(messageId);
				if (msg != null) {
					return msg;
				}
			}
		}
		EMAMessage message =  emaObject.getMessage(messageId);
		if (message == null) {
			return null;
		}
		EMMessage msg = new EMMessage(message);
		return msg;
	}


	/**
	 * \~chinese
	 * 获取会话，没有则返回null, 没找到则返回空
	 * @param id user id or group id
	 * @return 会话
	 *
	 * \~english
	 * get conversation by id
	 *
	 * @param id user id, group id or chatroom id
	 * @return EMConversation the existing conversation found by conversation, null if not found
	 */
	public EMConversation getConversation(String id){
		EMAConversation conversation = emaObject.conversationWithType(id, EMAConversationType.CHAT, false);
		if (conversation == null) {
			conversation = emaObject.conversationWithType(id, EMAConversationType.GROUPCHAT, false);
		}
		if (conversation == null) {
			conversation = emaObject.conversationWithType(id, EMAConversationType.CHATROOM, false);
		}
		if (conversation == null) {
			conversation = emaObject.conversationWithType(id, EMAConversationType.DISCUSSIONGROUP, false);
		}
		if (conversation == null) {
			conversation = emaObject.conversationWithType(id, EMAConversationType.HELPDESK, false);
		}

		return conversation == null ? null : new EMConversation(conversation);
	}

	/**
	 * \~chinese
	 * 根据用户或群组id以及会话类型获取会话，没有找到则返回空
	 *
	 * @param id 		用户或群组id
	 * @param type 		会话类型
	 * @return          会话
	 *
	 * \~english
	 * get conversation by conversation id and conversation type, create a empty conversation if not found
	 *
	 * @param id 			user id, group id or chatroom id
	 * @param type @see  	EMConversationType the conversation type
	 * @return 				EMConversation the conversation found according to the id and type, create a empty conversation if not found
	 */
	public EMConversation getConversation(String id, EMConversationType type) {
		return getConversation(id, type, false);
	}

	/**
	 * \~chinese
	 * 根据用户或群组id以及会话类型获取会话
	 *
	 * @param username 用户或群组id
	 * @param type 会话类型
	 * @param createIfNotExists 没找到相应会话时是否自动创建
	 * @return
	 *
	 * \~english
	 * get conversation by conversation id and conversation type
	 *
	 * @param username user id, group id or chatroom id
	 * @param type @see  EMConversationType the conversation type
	 * @param createIfNotExists create conversation if not exists
	 * @return
	 */
	public EMConversation getConversation(String username, EMConversationType type, boolean createIfNotExists) {
		EMAConversationType t = EMAConversationType.CHAT;
		if (type == EMConversationType.Chat) {
			t = EMAConversationType.CHAT;
		} else if (type == EMConversationType.GroupChat) {
			t = EMAConversationType.GROUPCHAT;
		} else if (type == EMConversationType.ChatRoom) {
			t = EMAConversationType.CHATROOM;
		} else if (type == EMConversationType.DiscussionGroup) {
			t = EMAConversationType.DISCUSSIONGROUP;
		} else if (type == EMConversationType.HelpDesk) {
			t = EMAConversationType.HELPDESK;
		}
		EMAConversation conversation = emaObject.conversationWithType(username, t, createIfNotExists);
		if (conversation == null) {
			return null;
		}
		Log.d(TAG, "convID:" + conversation.conversationId());
		return new EMConversation(conversation);
	}


	/**
	 * \~chinese
	 * 把所有的会话都设成已读
	 *
	 * \~english
	 * mark all messages as read
	 */
	public void markAllConversationsAsRead() {
		List<EMAConversation> conversations = emaObject.loadAllConversationsFromDB();
		for (EMAConversation conversation : conversations) {
			conversation.markAllMessagesAsRead(true);
		}
	}

	/**
	 * \~chinese
	 * 获取未读消息计数
	 * @deprecated  使用 {@link EMChatManager#getUnreadMessageCount()} 替代
	 * @return
	 *
	 * \~english
	 * get unread message count
	 * @deprecated  use {@link EMChatManager#getUnreadMessageCount()} instead
	 * @return
	 */
	@Deprecated
	public int getUnreadMsgsCount() {
		return getUnreadMessageCount();
	}

	/**
	 * \~chinese
	 * 获取未读消息计数
	 * @return
	 *
	 * \~english
	 * get unread message count
	 * @return
	 */
	public int getUnreadMessageCount()
	{
		List<EMAConversation> conversations = emaObject.getConversations();
		int unreadCount = 0;
		for (EMAConversation conversation : conversations) {
			if (conversation._getType() != EMAConversationType.CHATROOM) {
				unreadCount += conversation.unreadMessagesCount();
			}
		}
		return unreadCount;
	}

	/**
	 * \~chinese
	 * 保存用户app 生成的消息，比如系统提示 消息会存到内存中的conversation 和数据库
	 * CMD类型数据不保存在本地
	 *
	 * @param message	待存储的消息
	 *
	 * \~english
	 * save the message to memory and local database
	 * CMD type message to will stored in database
	 *
	 * @param message
	 */
	public void saveMessage(EMMessage message) {
		EMMessage.ChatType type = message.getChatType();
		EMConversationType t = EMConversationType.Chat;
		switch (type) {
			case Chat:
				t = EMConversationType.Chat;
				break;
			case GroupChat:
				t = EMConversationType.GroupChat;
				break;
			case ChatRoom:
				t = EMConversationType.ChatRoom;
				break;
		}
		String convId = message.getTo();
		//for group, chatroom, conversation id is group id for both receive and send message
		if (t == EMConversationType.Chat && message.direct() == EMMessage.Direct.RECEIVE) {
			convId = message.getFrom();
		}
		if (message.getType() == Type.CMD) {
			return;
		}
		EMConversation conv = getConversation(convId, t, true);
		// when send message out, appendMessage will update time to lastMsgTime + 1ms
		conv.insertMessage(message);
	}

	/**
	 * \~chinese
	 * 更新消息，消息的内容会被保存到本地
	 *
	 * @param message
	 *
	 * \~english
	 * update the message
	 *
	 * @param message
	 */
	public boolean updateMessage(EMMessage message) {
		String id = message.direct() == EMMessage.Direct.RECEIVE ? message.getFrom() : message.getTo();
		if (message.getType() == Type.CMD) {
			return false;
		}
		EMConversation conv = getConversation(message.conversationId(), EMConversation.msgType2ConversationType(id, message.getChatType()), true);
		return conv.updateMessage(message);
	}

	/**
	 * \~chinese
	 * 下载消息的附件，未成功下载的附件，可调用此方法再次下载
	 *
	 * @param msg
	 *
	 * \~english
	 * download the message attachement
	 *
	 * @param msg 	message to be downloaded
	 */
	public void downloadAttachment(final EMMessage msg) {
        if (msg == null) {
            return;
        }
        msg.makeCallbackStrong();
		checkContentAttachmentExist(msg);
		emaObject.downloadMessageAttachments(msg.emaObject);
	}

	/**
	 * 检查附件是否存在
	 * @param msg
	 */
	private void checkContentAttachmentExist(EMMessage msg) {
		List<EMAMessageBody> bodies = msg.emaObject.bodies();
		if(bodies != null && !bodies.isEmpty()) {
		    for (EMAMessageBody body: bodies) {
		    	switch (body.type()) {
		    	    case EMAMessageBody.EMAMessageBodyType_IMAGE :
		    	    case EMAMessageBody.EMAMessageBodyType_VIDEO :
		    	    case EMAMessageBody.EMAMessageBodyType_VOICE :
		    	    case EMAMessageBody.EMAMessageBodyType_FILE :
						EMAFileMessageBody fileBody = (EMAFileMessageBody) body;
						String localUrl = fileBody.getLocalUrl();
						EMLog.d(TAG, "download before check path = "+localUrl);
						Uri fileUri = UriUtils.getLocalUriFromString(localUrl);
						if(fileUri != null && UriUtils.uriStartWithContent(fileUri)
								&& !UriUtils.isFileExistByUri(mClient.getContext(), fileUri)) {
							String filename = fileBody.displayName();
							String newLocalPath = null;
							switch (body.type()) {
							    case EMAMessageBody.EMAMessageBodyType_IMAGE :
									newLocalPath = PathUtil.getInstance().getImagePath()+File.separator+filename;
							        break;
							    case EMAMessageBody.EMAMessageBodyType_VIDEO :
									newLocalPath = PathUtil.getInstance().getVideoPath()+File.separator+filename;
							        break;
							    case EMAMessageBody.EMAMessageBodyType_VOICE :
									newLocalPath = PathUtil.getInstance().getVoicePath()+File.separator+filename;
							        break;
							    case EMAMessageBody.EMAMessageBodyType_FILE :
									newLocalPath = PathUtil.getInstance().getFilePath()+File.separator+filename;
							        break;
							}
							if(!TextUtils.isEmpty(newLocalPath)) {
							    fileBody.setLocalPath(newLocalPath);
							    updateMessage(msg);
								EMLog.d(TAG, "download:create new path , path is "+newLocalPath);
							}
						}
						break;
		    	}
			}
		}
	}

	/**
	 * \~chinese
	 * 下载消息的缩略图
	 *
	 * @param msg
	 *
	 * \~english
	 * download the thumbnail if not downloaded before
	 *
	 * @param msg  	the message to be downloaded
	 */
	public void downloadThumbnail(final EMMessage msg) {
        msg.makeCallbackStrong();
		emaObject.downloadMessageThumbnail(msg.emaObject);
	}

	/**
	 * \~chinese
	 * 向消息数据库导入多条聊天记录
	 * 在调用次函数时要保证，消息的发送方或者接收方是当前用户
	 * 已经对函数做过速度优化， 推荐一次导入1000条数据
	 *
	 * @param msgs 需要导入数据库的消息
	 *
	 * \~english
	 * import messages to local database. Make sure the meessage's sender or receiver is current user
	 * Recommend import up to 1000 messages per operation
	 *
	 * @param msgs the message list of be imported
	 */
	public synchronized void importMessages(List<EMMessage> msgs) {
		List<EMAMessage> dummy = new ArrayList<EMAMessage>();
		for (EMMessage msg : msgs) {
			dummy.add(msg.emaObject);
		}
		EMClient.getInstance().getChatConfigPrivate().importMessages(dummy);
	}

	/**
	 * \~chinese
	 * 获取某个类型所有的会话
	 *
	 * @return 会话list
	 *
	 * \~english
	 * get list of conversations by conversation type
	 *
	 * @param type	converation type
	 * @return 		list of conversation in specified type
	 */
	public List<EMConversation> getConversationsByType(EMConversationType type) {
		List<EMAConversation> conversations = emaObject.getConversations();
		List<EMConversation> result = new ArrayList<EMConversation>();
		for (EMAConversation conv : conversations) {
			if (type.ordinal() == conv._getType().ordinal()) {
				result.add(new EMConversation(conv));
			}
		}
		return result;
	}

	/**
	 * \~chinese
	 * 从服务器下载文件，推荐使用downloadAttachment(EMMessage msg)替代此函数
	 *
	 * @param remoteUrl 服务器上的远程文件
	 * @param localFilePath 本地要生成的文件
	 * @param headers Http Request Header
	 * @param callback EMCallBack
	 *
	 * \~english
	 * download the file from server. Recommend using downloadAttachment(EMMessage msg) instead of this function.
	 *
	 * @param remoteUrl 		the remote file url
	 * @param localFilePath 	the local file path
	 * @param headers 			Http Request Headers
	 * @param callback 			EMCallBack
	 */
	@Deprecated
	public void downloadFile(final String remoteUrl,
							 final String localFilePath,
							 final Map<String, String> headers,
							 final EMCallBack callback) {
		/*
		if (!headers.containsKey("Authorization")) {
			headers.put("Authorization", EMChatConfig.getInstance().getAccessToken());
		}
		int responseCode = EMARHttpAPI.download(remoteUrl, localFilePath, headers);

		if (200 <= responseCode && responseCode <= 299) {
			if (callback != null) {
				callback.onSuccess();
			}
		} else {
			callback.onError(responseCode, "");
		}
		*/

		EMHttpClient.getInstance().downloadFile(remoteUrl, localFilePath, headers, new EMCloudOperationCallback(){

			@Override
			public void onSuccess(String result) {
				if(callback != null){
					callback.onSuccess();
				}
			}

			@Override
			public void onError(String msg) {
				if(callback != null){
					callback.onError(EMError.GENERAL_ERROR, msg);
				}
			}

			@Override
			public void onProgress(int progress) {
				if(callback != null){
					callback.onProgress(progress, null);
				}
			}

		});
	}

	/**
	 * \~chinese
	 * 获取当前所有的会话
	 *
	 * @return
	 *
	 * \~english
	 * get all conversations in the cache
	 *
	 * @return conversations which is a map with key&#45;&#62;conversation id : value&#45;&#62;EMConversation
	 */
	public Map<String, EMConversation> getAllConversations() {
		List<EMAConversation> conversations = emaObject.getConversations();
		Hashtable<String, EMConversation> result = new Hashtable<String, EMConversation>();
		for (EMAConversation conversation : conversations) {
			/*if (conversation._getType() != EMAConversationType.CHATROOM) {
				result.put(conversation.conversationId(), new EMConversation(conversation));
			}*/
			result.put(conversation.conversationId(), new EMConversation(conversation));
		}
		return result;
	}

	/**
	 * \~chinese
	 * 同步加载所有的会话，并且每条会话读入EMChatOptions.getNumberOfMessagesLoaded()条消息, 默认是20条以保持兼容
	 *
	 * \~english
	 * load all conversatons from local database
	 */
	public void loadAllConversations() {
		emaObject.loadAllConversationsFromDB();
	}

	/**
	 * \~chinese
	 * 删除和指定用户或者群聊的对话(包括删除本地的聊天记录)
	 *
	 * @param username 			用户名或者群聊id
	 * @param deleteMessages	是否删除消息
	 * @return 					删除失败或者不存在此user的conversation返回false
	 *
	 * \~english
	 * delete conversation and messages from local database
	 *
	 * @param username 			user id or group id
	 * @param deleteMessages 	whether delete messages
	 * @return 					return true if delete successfully
	 */
	public boolean deleteConversation(String username, boolean deleteMessages) {
		EMConversation conv = getConversation(username);
		if (conv == null) {
			return false;
		}
		if (!deleteMessages) {
			conv.clear();
		} else {
			conv.clearAllMessages();
		}
		emaObject.removeConversation(username, deleteMessages);
		return true;
	}

	/**
	 * \~chinese
	 * 注册消息监听
	 * @param listener
	 *
	 * \~english
	 * add message listener @see EMMessageListener
	 *
	 * @param listener the message listener which is used to monitor the incoming messages
	 */
	public void addMessageListener(EMMessageListener listener) {
		if(listener == null){
			EMLog.d(TAG, "addMessageListener: listener is null");
			return;
		}

		if(!messageListeners.contains(listener)){
			EMLog.d(TAG, "add message listener: " + listener);
			messageListeners.add(listener);
		}
	}

	/**
	 * \~chinese
	 * 移除消息监听
	 *
	 * @param listener 要移除的监听
	 *
	 * \~english
	 * remove the message listener
	 *
	 * @param listener the message listener set before
	 */
	public void removeMessageListener(EMMessageListener listener) {
		if(listener == null){
			return;
		}

		messageListeners.remove(listener);
	}

	/**
	 * \~chinese
	 * 注册会话监听
	 *
	 * @param listener
	 *
	 * \~english
	 * add conversation listener @see EMConversationListener
	 * refresh the conversation list when receiving the conversation refresh event
	 *
	 * @param listener the conversation listener
	 */
	public void addConversationListener(EMConversationListener listener){
		if(!conversationListeners.contains(listener)){
			conversationListeners.add(listener);
		}
	}

	/**
	 * \~chinese
	 * 移除会话监听
	 *
	 * @param listener
	 *
	 * \~english
	 * remove the conversation listener
	 *
	 * @param listener the conversation listener set before
	 */
	public void removeConversationListener(EMConversationListener listener){
		if(listener == null){
			return;
		}

		conversationListeners.remove(listener);
	}

	/**
	 * \~chinese
	 * 设置消息为已听，一般用于语音消息
	 *
	 * @deprecated  使用{@link EMChatManager#setVoiceMessageListened(EMMessage)}替代
	 *
	 * \~english
	 * set message to listened, used for voice message
	 *
	 * @deprecated  use {@link EMChatManager#setVoiceMessageListened(EMMessage)} instead
	 */
	@Deprecated
	public void setMessageListened(EMMessage message) {
		setVoiceMessageListened(message);
	}

	/**
	 * \~chinese
	 * 设置消息为已听，一般用于语音消息
	 *
	 * \~english
	 * set message to listened, used for voice message
	 */
	public void setVoiceMessageListened(EMMessage message)
	{
		message.setListened(true);
		updateMessage(message);
	}

	void onLogout() {
		caches.clear();
	}

	synchronized void  loadAllConversationsFromDB() {
		emaObject.loadAllConversationsFromDB();
	}

	/**
	 * \~chinese
	 * 将数据库中的某个联系人相关信息变更成另外一个联系人
	 * 与变更相关的表单包含消息表单，会话表单，联系人表单，黑名单表单
	 * 注意：该操作不会更新内存中数据
	 *
	 * @param from
	 * @param changeTo
	 * @return 返回更新结果，任何表单更新失败，都会返回false
	 *
	 * \~english
	 * update database records, change username 'from' to 'changeTo', take effect on message table, conversation table, contacts, blacklist table
	 * note: this operation does not update data stored in memory cache.
	 *
	 * @param from
	 * @param changeTo
	 * @return operation result, if any update operations on those tables failed, result is false
	 */
	public boolean updateParticipant(String from, String changeTo) {
		return emaObject.updateParticipant(from, changeTo);
	}

	/**
	 * \~chinese
	 * 从服务器获取群组消息回执详情
	 * @param msgId 消息id
	 * @param pageSize 获取的页面大小
	 * @param startAckId 已读回执的id，如果为空，从最新的回执向前开始获取
	 * @return 返回消息列表和用于继续获取群消息回执的Cursor
	 *
	 * \~english
	 * fetch ack details for group messages from server.
	 * @param msgId message id
	 * @param pageSize the page size.
	 * @param startAckId the start id for fetch acks, if empty start from the server lastest ack.
	 * @return server return ack and cursor for next fetch action.
	 */
	public EMCursorResult<EMGroupReadAck> fetchGroupReadAcks(String msgId, int pageSize, String startAckId) throws HyphenateException {
		EMAError error = new EMAError();
        EMCursorResult<EMGroupReadAck> cusorResult = new EMCursorResult<>();

		EMMessage msg = getMessage(msgId);
		String groupId = null;
		if (msg.getChatType() == EMMessage.ChatType.GroupChat && msg.isNeedGroupAck()) {
			groupId = msg.conversationId();
		} else {
			EMLog.e(TAG, "not group msg or don't need ack");
			return cusorResult;
		}

		EMCursorResult<EMAGroupReadAck> _cusorResult = emaObject.fetchGroupReadAcks(msgId, groupId, error, pageSize, startAckId);
		handleError(error);
		cusorResult.setCursor(_cusorResult.getCursor());

		List<EMGroupReadAck> groupReadAcks = new ArrayList<>();

		for(EMAGroupReadAck _ack: _cusorResult.getData()) {
			groupReadAcks.add(new EMGroupReadAck(_ack));
		}
		cusorResult.setData(groupReadAcks);
		return cusorResult;
	}

	/**
	 * \~chinese
	 * 从服务器获取群组消息回执详情
	 * @param msgId 消息id
	 * @param pageSize 获取的页面大小
	 * @param startAckId 已读回执的id，如果为空，从最新的回执向前开始获取
	 * @return 返回消息列表和用于继续获取群消息回执的Cursor
	 *
	 * \~english
	 * fetch ack details for group messages from server.
	 * @param msgId message id
	 * @param pageSize the page size.
	 * @param startAckId the start id for fetch acks, if empty start from the server lastest ack.
	 * @return server return ack and cursor for next fetch action.
	 */
	public void asyncFetchGroupReadAcks(final String msgId, final int pageSize,
										final String startAckId, final EMValueCallBack<EMCursorResult<EMGroupReadAck>> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchGroupReadAcks(msgId, pageSize, startAckId));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从服务器获取历史消息
	 * @param conversationId 会话名称
	 * @param type 会话类型
	 * @param pageSize 获取的页面大小
	 * @param startMsgId 漫游消息的开始消息id，如果为空，从最新的消息向前开始获取
	 * @return 返回消息列表和用于继续获取历史消息的Cursor
	 *
	 * \~english
	 * fetch conversation roam messages from server.
	 * @param conversationId the conversation id which select to fetch roam message.
	 * @param type the conversation type which select to fetch roam message.
	 * @param pageSize the page size.
	 * @param startMsgId the start search roam message, if empty start from the server lastest message.
	 * @return server return messages and cursor for next fetch action.
	 */
	public EMCursorResult<EMMessage> fetchHistoryMessages(String conversationId, EMConversationType type, int pageSize,
	                                 String startMsgId) throws HyphenateException {
		EMAError error = new EMAError();
		EMCursorResult<EMAMessage> _cursorResult = emaObject.fetchHistoryMessages(conversationId,
				type.ordinal(), pageSize, startMsgId, error);
		handleError(error);
		EMCursorResult<EMMessage> cursorResult = new EMCursorResult<>();
		cursorResult.setCursor(_cursorResult.getCursor());
		List<EMMessage> msgs = new ArrayList<>();
		for (EMAMessage _msg : _cursorResult.getData()) {
			msgs.add(new EMMessage(_msg));
		}
		cursorResult.setData(msgs);
		return cursorResult;
	}

	/**
	 * \~chinese
	 * 从服务器获取历史消息
	 * @param conversationId 会话名称
	 * @param type 会话类型
	 * @param pageSize 获取的页面大小
	 * @param startMsgId 漫游消息的开始消息id，如果为空，从最新的消息向前开始获取
	 * @param callBack 返回消息列表和用于继续获取历史消息的Cursor
	 *
	 * \~english
	 * fetch conversation roam messages from server.
	 * @param conversationId the conversation id which select to fetch roam message.
	 * @param type the conversation type which select to fetch roam message.
	 * @param pageSize the page size.
	 * @param startMsgId the start search roam message, if empty start from the server lastest message.
	 * @param callBack server return messages and cursor for next fetch action.
	 */
	public void asyncFetchHistoryMessage(final String conversationId, final EMConversationType type, final int pageSize,
	                                final String startMsgId, final EMValueCallBack<EMCursorResult<EMMessage>> callBack) {
		EMClient.getInstance().execute(new Runnable() {
			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchHistoryMessages(conversationId, type, pageSize, startMsgId));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	private void handleError(EMAError error) throws HyphenateException {
		if (error.errCode() != EMAError.EM_NO_ERROR) {
			throw new HyphenateException(error);
		}
	}

	private static final int LIST_SIZE = 512;

	/**
	 * \~chinese
	 * 根据传入的参数从本地存储中搜索指定数量的message。
	 * 注意：当maxCount非常大时，需要考虑内存消耗
	 *
	 * @param type       消息类型，文本、图片、语音等等
	 * @param timeStamp  搜索消息的时间点
	 * @param maxCount   搜索结果的最大条数
	 * @param from       搜索来自某人的消息，适用于搜索群组里的消息
	 * @return           消息列表
	 *
	 * \~english
	 * search message from database according the parameter
	 * Note: Cautious about memory usage when the maxCount is large
	 *
	 * @param type       message type, TXT、VOICE、IMAGE etc.
	 * @param timeStamp  the timestamp for search
	 * @param maxCount   the max number of message to search
	 * @param from       who the message from, used to search in group
	 * @return           the list of searched messages
	 */
	public List<EMMessage> searchMsgFromDB(EMMessage.Type type, long timeStamp, int maxCount, String from, EMConversation.EMSearchDirection direction) {
		EMAConversation.EMASearchDirection d = direction == EMConversation.EMSearchDirection.UP ? EMAConversation.EMASearchDirection.UP : EMAConversation.EMASearchDirection.DOWN;

		List<EMAMessage> msgs = emaObject.searchMessages(type.ordinal(), timeStamp, maxCount, from, d);
		// to avoid resizing issue of array list, used linked list when size > 512
		List<EMMessage> result;
		if (msgs.size() > LIST_SIZE) {
			result = new LinkedList<EMMessage>();
		} else {
			result = new ArrayList<EMMessage>();
		}
		for (EMAMessage msg : msgs) {
			if (msg != null) {
				result.add(new EMMessage(msg));
			}
		}
		return result;
	}

	/**
	 * \~chinese
	 * 根据传入的参数从本地存储中搜索指定数量的message。
	 * 注意：当maxCount非常大时，需要考虑内存消耗
	 *
	 * @param keywords   搜索消息中的关键词
	 * @param timeStamp  搜索消息的时间点
	 * @param maxCount   搜索结果的最大条数
	 * @param from       搜索来自某人的消息，适用于搜索群组里的消息
	 * @return           消息列表
	 *
	 * \~english
	 * search message from database based the parameters
	 * Note: Cautious about memory usage when the maxCount is large
	 *
	 * @param keywords   the keywords in message.
	 * @param timeStamp  the timestamp for search
	 * @param maxCount   the max number of message to search
	 * @param from       who the message from, used to search in group
	 * @return           the list of searched messages
	 */
	public List<EMMessage> searchMsgFromDB(String keywords, long timeStamp, int maxCount, String from, EMConversation.EMSearchDirection direction) {
		EMAConversation.EMASearchDirection d = direction == EMConversation.EMSearchDirection.UP ? EMAConversation.EMASearchDirection.UP : EMAConversation.EMASearchDirection.DOWN;

		List<EMAMessage> msgs = emaObject.searchMessages(keywords, timeStamp, maxCount, from, d);
		// to avoid resizing issue of array list, used linked list when size > 512
		List<EMMessage> result;
		if (msgs.size() > LIST_SIZE) {
			result = new LinkedList<EMMessage>();
		} else {
			result = new ArrayList<EMMessage>();
		}
		for (EMAMessage msg : msgs) {
			if (msg != null) {
				result.add(new EMMessage(msg));
			}
		}
		return result;
	}

}
