/************************************************************
 *  * EaseMob CONFIDENTIAL 
 * __________________ 
 * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
 *
 * NOTICE: All information contained herein is, and remains 
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material 
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import android.os.Parcel;
import android.os.Parcelable;

import com.hyphenate.chat.adapter.message.EMAFileMessageBody.EMADownloadStatus;
import com.hyphenate.chat.adapter.message.EMAMessageBody;
import com.hyphenate.chat.adapter.message.EMAVideoMessageBody;
import com.hyphenate.util.EMLog;

/**
 * \~chinese
 * 视频消息体
 *
 * EMVideoMessageBody body = new EMVideoMessageBody(videoFilePath,thumbPath, duration, filelength);
 *
 * \~english
 * Video message body
 *
 * EMVideoMessageBody body = new EMVideoMessageBody(videoFilePath, thumbPath, duration, filelength);
 */
public class EMVideoMessageBody extends EMFileMessageBody implements Parcelable {

    public EMVideoMessageBody() {
        super("", EMAMessageBody.EMAMessageBodyType_VIDEO);
    }

    public EMVideoMessageBody(EMAVideoMessageBody body) {
        super(body);
    }

    /**
     * \~chinese
     * 创建一个视频消息体
     *
     * @param videoFilePath 视频文件路径
     * @param thumbPath     预览图路径
     * @param duration      视频时间
     * @param filelength    视频文件长度
     *
     * \~english
     *                      create a video message body
     * @param videoFilePath the path of the video file
     * @param thumbPath     the path of the video first frame
     * @param duration      video duration, in seconds
     * @param filelength    the length of the video file
     */
    public EMVideoMessageBody(String videoFilePath, String thumbPath, int duration, long filelength) {
        super(videoFilePath, EMAMessageBody.EMAMessageBodyType_VIDEO);
        ((EMAVideoMessageBody) emaObject).setThumbnailLocalPath(thumbPath);
        ((EMAVideoMessageBody) emaObject).setDuration(duration);
        ((EMAVideoMessageBody) emaObject).setFileLength(filelength);
        EMLog.d("videomsg", "create video, message body for:" + videoFilePath);
    }

    /**
     * create video message body from incoming xmpp message json
     */
    EMVideoMessageBody(String fileName, String remoteUrl, String thumbnailUrl, int length) {
        super(fileName, EMAMessageBody.EMAMessageBodyType_VIDEO);
        ((EMAVideoMessageBody) emaObject).setThumbnailLocalPath(thumbnailUrl);
        ((EMAVideoMessageBody) emaObject).setLocalPath(fileName);
        ((EMAVideoMessageBody) emaObject).setRemotePath(remoteUrl);
        ((EMAVideoMessageBody) emaObject).setThumbnailRemotePath(thumbnailUrl);
        ((EMAVideoMessageBody) emaObject).setFileLength(length);
    }

    /**
     * \~chinese
     * 获取视频文件的文件大小
     *
     * @return
     *
     * \~english
     * File length of a thumbnail, in bytes
     *
     * @return
     */
    public long getVideoFileLength() {
        return ((EMAVideoMessageBody) emaObject).fileLength();
    }

    /**
     * \~chinese
     * 视频文件的文件大小。用户在做数据库迁移的时候可能使用。
     *
     * \~english
     * set file length of a thumbnail, in bytes. Might need it during database migration
     */
    public void setVideoFileLength(long filelength) {
        ((EMAVideoMessageBody) emaObject).setFileLength(filelength);
    }

    /**
     *  \~chinese
     *  缩略图在服务器的路径
     *
     *  \~english
     *  Local path of thumbnail
     */
    public String getThumbnailUrl() {
        return ((EMAVideoMessageBody) emaObject).thumbnailRemotePath();
    }

    /**
     *  \~chinese
     *  缩略图在服务器的路径
     *
     *  \~english
     *  Server url path of thumbnail
     */
    public void setThumbnailUrl (String thumbnailUrl) {
        ((EMAVideoMessageBody) emaObject).setThumbnailRemotePath(thumbnailUrl);
    }

    public void setThumbnailSize(int width, int height) {
        ((EMAVideoMessageBody)emaObject).setSize(width, height);
    }
    /**
     * \~chinese
     * 获取视频缩略图的宽度
     * @return
     *
     * \~english
     * get video thumbnail width
     * @return
     */
    public int getThumbnailWidth() {
        return ((EMAVideoMessageBody)emaObject).width();
    }

    /**
     * \~chinese
     * 获取视频缩略图的高度
     * @return
     *
     * \~english
     * get video thumbnail height
     * @return
     */
    public int getThumbnailHeight() {
        return ((EMAVideoMessageBody)emaObject).height();
    }
    /**
     *  \~chinese
     *  获取缩略图的本地路径
     *
     *  UIImage *image = [UIImage imageWithContentsOfFile:thumbnailLocalPath];
     *  @return
     *
     *  \~english
     *  Local path of thumbnail
     *
     *  UIImage *image = [UIImage imageWithContentsOfFile:thumbnailLocalPath];
     *
     *  @return
     */
    public String getLocalThumb() {
        return ((EMAVideoMessageBody) emaObject).thumbnailLocalPath();
    }

    public void setLocalThumb(String localThumbPath) {
        ((EMAVideoMessageBody) emaObject).setThumbnailLocalPath(localThumbPath);
    }

    /**
     * \~chinese
     * 获得视频时长, 秒为单位
     *
     * @return
     *
     * \~english
     * Video duration, in seconds
     *
     * @return
     */
    public int getDuration() {
        return (int) ((EMAVideoMessageBody) emaObject).duration();
    }

    public String toString() {
        return "video: " + ((EMAVideoMessageBody) emaObject).displayName() +
                ", localUrl: " + ((EMAVideoMessageBody) emaObject).getLocalUrl() +
                ", remoteUrl: " + ((EMAVideoMessageBody) emaObject).getRemoteUrl() +
                ", thumbnailUrl: " + ((EMAVideoMessageBody) emaObject).thumbnailLocalPath() +
                ", length: " + ((EMAVideoMessageBody) emaObject).fileLength();
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        // members from FileMessageBody
        dest.writeString(((EMAVideoMessageBody) emaObject).displayName());
        dest.writeString(((EMAVideoMessageBody) emaObject).getLocalUrl());
        dest.writeString(((EMAVideoMessageBody) emaObject).getRemoteUrl());
        dest.writeString(((EMAVideoMessageBody) emaObject).thumbnailLocalPath());
        dest.writeString(((EMAVideoMessageBody) emaObject).thumbnailLocalPath());
        dest.writeInt(((EMAVideoMessageBody) emaObject).duration());
        dest.writeLong(((EMAVideoMessageBody) emaObject).fileLength());
        dest.writeInt(((EMAVideoMessageBody) emaObject).width());
        dest.writeInt(((EMAVideoMessageBody) emaObject).height());
    }

    public static final Parcelable.Creator<EMVideoMessageBody> CREATOR = new Creator<EMVideoMessageBody>() {

        @Override
        public EMVideoMessageBody[] newArray(int size) {
            return new EMVideoMessageBody[size];
        }

        @Override
        public EMVideoMessageBody createFromParcel(Parcel in) {

            return new EMVideoMessageBody(in);
        }
    };

    private EMVideoMessageBody(Parcel in) {
        super("", EMAMessageBody.EMAMessageBodyType_VIDEO);
        // members from FileMessageBody
        ((EMAVideoMessageBody) emaObject).setDisplayName(in.readString());
        ((EMAVideoMessageBody) emaObject).setLocalPath(in.readString());
        ((EMAVideoMessageBody) emaObject).setRemotePath(in.readString());
        ((EMAVideoMessageBody) emaObject).setThumbnailRemotePath(in.readString());
        ((EMAVideoMessageBody) emaObject).setThumbnailLocalPath(in.readString());
        ((EMAVideoMessageBody) emaObject).setDuration(in.readInt());
        ((EMAVideoMessageBody) emaObject).setFileLength(in.readLong());
        int width = in.readInt();
        int height = in.readInt();
        ((EMAVideoMessageBody) emaObject).setSize(width, height);
    }

    /**
     *  \~chinese
     *  缩略图的密钥, 下载缩略图时需要密匙做校验
     *
     *  \~english
     *  Secret key of thumbnail, required to download a thumbnail
     */
    public String getThumbnailSecret() {
        return ((EMAVideoMessageBody) emaObject).thumbnailSecretKey();
    }

    public void setThumbnailSecret(String secret) {
        ((EMAVideoMessageBody) emaObject).setThumbnailSecretKey(secret);
    }

    /**
     * \~chinese
     * 缩略图下载状态
     *
     * @return
     *
     * \~english
     * Download status of thumbnail
     *
     * @return
     */
    public EMDownloadStatus thumbnailDownloadStatus() {
        EMADownloadStatus _status = ((EMAVideoMessageBody) emaObject).thumbnailDownloadStatus();
        switch (_status) {
            case DOWNLOADING:
                return EMDownloadStatus.DOWNLOADING;
            case SUCCESSED:
                return EMDownloadStatus.SUCCESSED;
            case FAILED:
                return EMDownloadStatus.FAILED;
            case PENDING:
                return EMDownloadStatus.PENDING;
        }
        return EMDownloadStatus.SUCCESSED;
    }

    void setThumbnailDownloadStatus(EMDownloadStatus status) {
        ((EMAVideoMessageBody) emaObject).setThumbnailDownloadStatus(EMADownloadStatus.valueOf(status.name()));
    }
}
