package com.hyphenate.chat;

import android.graphics.Bitmap;
import android.text.TextUtils;

import com.hyphenate.EMConferenceListener;
import com.hyphenate.EMError;
import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.adapter.EMACallConferenceListener;
import com.hyphenate.chat.adapter.EMACallManager;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.media.EMCallSurfaceView;
import com.hyphenate.util.EMLog;
import com.hyphenate.util.EasyUtils;
import com.superrtc.mediamanager.EMediaDefines;
import com.superrtc.mediamanager.EMediaEntities;
import com.superrtc.mediamanager.EMediaManager;
import com.superrtc.mediamanager.EMediaPublishConfiguration;
import com.superrtc.mediamanager.EMediaSession;
import com.superrtc.mediamanager.EMediaStream;
import com.superrtc.mediamanager.EMediaTalker;
import com.superrtc.sdk.RtcConnection;
import com.superrtc.sdk.VideoViewRenderer;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * Created by lzan13 on 2017/8/16.
 * \~chinese
 * 多人音视频会议管理类，封装多人音视频会议操作方法，创建，加入，邀请等
 *
 * \~english
 * Multi person conference manager, Encapsulation of multi-person audio and video conference operation method,
 * create, join, invite and so on
 */
public class EMConferenceManager {
    private final String TAG = this.getClass().getSimpleName();

    static class Role {
        private EMConferenceRole role = EMConferenceRole.Audience;
        private String roleToken;

        public Role updateRole(EMConferenceRole role) {
            this.role = role;
            return this;
        }

        public Role updateToken(String token) {
            this.roleToken = token;
            return this;
        }

        public void update(EMConferenceRole role, String token) {
            this.role = role;
            this.roleToken = token;
        }

        public EMConferenceRole getRole() {
            return role;
        }

        public String getRoleToken() {
            return roleToken;
        }
    }

    public enum EMConferenceRole {
        Admin(7), //管理员，授权，订阅流，发布流
        Talker(3), //主播，订阅流和发布流
        Audience(1); //观众，只能订阅流

        public int code;

        EMConferenceRole(int code) {
            this.code = code;
        }

        static EMConferenceRole from(int code) {
            switch (code) {
                case 1:
                    return Audience;
                case 3:
                    return Talker;
                case 7:
                    return Admin;
            }
            return null;
        }
    }

    public enum EMConferenceType {
        SmallCommunication(10), // <= 4， 默认都是speaker
        LargeCommunication(11), // > 4 && < 20， 默认都是speaker
        LiveStream(12);         // > 20, 加入后默认为观众

        public int code;

        EMConferenceType(int code) {
            this.code = code;
        }

        static EMConferenceType from(int code) {
            switch (code) {
                case 10:
                    return SmallCommunication;
                case 11:
                    return LargeCommunication;
                case 12:
                    return LiveStream;
            }
            return null;
        }
    }

    private EMACallManager emaCallManager;

    private EMediaManager mediaManager = null;
    private EMediaSession mediaSession = null;
    private Role role = new Role();

    private ExecutorService executorService;

    private List<EMConferenceListener> conferenceListeners = Collections.synchronizedList(new ArrayList<EMConferenceListener>());
    private List<EMConferenceMember> memberList = Collections.synchronizedList(new ArrayList<EMConferenceMember>());
    private Map<String, EMConferenceStream> availableStreamMap = new ConcurrentHashMap<>();
    private Map<String, EMConferenceStream> subscribedStreamMap = new ConcurrentHashMap<>();

    private EMConferenceListener.ConferenceMode conferenceMode = EMConferenceListener.ConferenceMode.NORMAL;

    private String accessToken;
    private String appKey;
    private String username;

    /**
     * 设置单独使用视音频功能时所需参数
     *
     * @param accessToken
     * @param appKey
     * @param username
     */
    public void set(String accessToken, String appKey, String username) {
        this.accessToken = accessToken;
        this.appKey = appKey;
        this.username = username;
    }

    /**
     * preventing to instantiate this EMConferenceManager
     */
    protected EMConferenceManager(EMACallManager manager) {
        emaCallManager = manager;
        if (!EMediaManager.isInit()) {
            EMediaManager.initGlobal(EMClient.getInstance().getContext());
        }
        mediaManager = EMediaManager.getInstance();
        executorService = Executors.newCachedThreadPool();
        emaCallManager.setCallConferenceListener(new EMACallConferenceListener() {
            @Override public void onReceiveInvite(String confId, String password, String extension) {
                synchronized (conferenceListeners) {
                    try {
                        for (EMConferenceListener listener : conferenceListeners) {
                            listener.onReceiveInvite(confId, password, extension);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }
        });
        setSubscribeAudioMixEnable();
    }

    /**
     * \~chinese
     * 添加会议监听
     *
     * \~english
     * Add conference listener
     */
    public void addConferenceListener(EMConferenceListener listener) {
        if (listener != null && !conferenceListeners.contains(listener)) {
            conferenceListeners.add(listener);
        }
    }

    /**
     * \~chinese
     * 移除会议监听
     *
     * \~english
     * Remove conference listener
     */
    public void removeConferenceListener(EMConferenceListener listener) {
        if (listener != null) {
            conferenceListeners.remove(listener);
        }
    }

    /**
     * \~chinese
     * 查询会议信息
     *
     * @param confId   会议id
     * @param password 会议密码
     * @param callback 获取结果回调
     *
     * \~english
     * Get the conference info from server.
     *
     * @param confId   conference id
     * @param password conference password
     * @param callback result callback
     */
    public void getConferenceInfo(final String confId, final String password,
                                          final EMValueCallBack<EMConference> callback) {
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                EMAError error = new EMAError();
                Map<String, Object> params = new HashMap<>();
                params.put("confrId", confId);
                params.put("password", password);
                String body = convertMapToJSONObject(params).toString();
                String result = emaCallManager.requestMediaFromServer(4, body, error);
                if (error.errCode() != EMError.EM_NO_ERROR) {
                    if (callback != null)
                        callback.onError(error.errCode(), error.errMsg());
                    return;
                }
                try {
                    JSONObject object = new JSONObject(result);
                    JSONObject subObj = object.optJSONObject("confr");

                    EMConference conference = new EMConference();
                    conference.setPassword(password);
                    conference.setConferenceId(subObj.optString("id"));
                    conference.setConferenceType(EMConferenceManager.EMConferenceType.from(subObj.optInt("type")));
                    conference.setMemberNum(subObj.optInt("memTotal"));
                    JSONArray array = subObj.optJSONArray("admins");
                    String[] admins = new String[array.length()];
                    for (int i = 0; i < array.length(); i++) {
                        admins[i] = array.getString(i);
                    }
                    conference.setAdmins(admins);
                    if (callback != null) {
                        callback.onSuccess(conference);
                    }
                } catch (JSONException e) {
                    e.printStackTrace();
                    if (callback != null) {
                        callback.onError(EMError.GENERAL_ERROR, e.getMessage());
                    }
                }
            }
        });
    }

    /**
     * \~chinese
     * 创建并加入会议
     *
     * @param password 会议密码
     * @param callback 结果回调
     *
     * \~english
     * Create  and join conference
     * @param password Conference password
     * @param callback result callback
     */
    public void createAndJoinConference(final EMConferenceType type, final String password, final EMValueCallBack<EMConference> callback) {
        createAndJoinConference(type, password, null, callback);
    }

    /**
     * \~chinese
     * 创建并加入会议
     *
     * @param password 会议密码
     * @param param 加入会议时 publish 自己本地数据参数
     * @param callback 结果回调
     *
     * \~english
     * Create  and join conference
     * @param password Conference password
     * @param param join conference publish local stream param
     * @param callback result callback
     *
     * @Deprecated Use {@link #createAndJoinConference(EMConferenceType, String, EMValueCallBack)} instead
     * and publish stream use {@link #publish(EMStreamParam, EMValueCallBack)} in {@param callback#onSuccess()} callback.
     */
    @Deprecated
    public void createAndJoinConference(final EMConferenceType type, final String password, final EMStreamParam param,
            final EMValueCallBack<EMConference> callback) {
        EMLog.d(TAG, "createAndJoinConference");

        if (TextUtils.isEmpty(accessToken) || TextUtils.isEmpty(appKey) || TextUtils.isEmpty(username)) {
            EMLog.e(TAG, "AccessToken AppKey or Username is not set");
            if (callback != null) {
                callback.onError(EMError.CALL_CONFERENCE_CREATE_FAILED, "If you want to use this feature without im account login, " +
                        "please setting the necessary parameters through the EMClient.getInstance().conferenceManager.set(String accessToken, String appKey, String username)" +
                        " method.");
            }
            return;
        }

        executorService.execute(new Runnable() {
            @Override public void run() {
                EMAError error = new EMAError();
                Map<String, Object> params = new HashMap<>();
                params.put("uid", EasyUtils.getMediaRequestUid(appKey, username));
                params.put("token", accessToken);
                params.put("confrType", type.code);
                params.put("password", password);
                addCommonParams(params);
                String body = convertMapToJSONObject(params).toString();
                String result = emaCallManager.requestMediaFromServer(0, body, error);
                if (error.errCode() != EMError.EM_NO_ERROR) {
                    EMLog.d(TAG, "requestMediaFromServer failed " + error.errCode() + ", " + error.errMsg());
                    if (callback != null) {
                        callback.onError(error.errCode(), error.errMsg());
                    }
                    return;
                }
                final EMConference conference = new EMConference();
                try {
                    JSONObject object = new JSONObject(result);
                    String ticket = object.optString("ticket");
                    conference.setConferenceId(object.optString("confrId"));
                    conference.setPassword(password);
                    conference.setConferenceType(type);
                    EMLog.d(TAG, "Join conference");

                    role.update(EMConferenceRole.Admin, object.optString("roleToken"));

                    mediaSession = mediaManager.newSessionWithTicket(ticket, "{'extension':'creator'}", sessionDelegate);
                    mediaManager.setSession(mediaSession, EMClient.getInstance().getCurrentUser());
                    mediaManager.join(mediaSession, configWrap(param), new EMediaEntities.EMediaIdBlockType() {
                        @Override public void onDone(Object uid, EMediaEntities.EMediaError error) {
                            if (error != null) {
                                EMLog.d(TAG, "Join conference failed code=" + error.code + ", desc=" + error.errorDescription);
                                if (callback != null) {
                                    callback.onError(errorMap(error.code.errorcode), error.errorDescription);
                                }
                            } else {
                                EMLog.d(TAG, "Join conference success!");
                                if (type == EMConferenceType.LiveStream) {
                                    conference.setConferenceRole(EMConferenceRole.from(mediaSession.role));
                                } else {
                                    conference.setConferenceRole(EMConferenceRole.Admin);
                                }
                                conference.setPubStreamId((String) uid, EMConferenceStream.StreamType.NORMAL);

                                if (callback != null) {
                                    callback.onSuccess(conference);
                                }
                            }
                        }
                    });
                } catch (JSONException e) {
                    e.printStackTrace();
                    if (callback != null) {
                        callback.onError(EMError.CALL_CONFERENCE_CREATE_FAILED, e.getMessage());
                    }
                }
            }
        });
    }
    /**
     * \~chinese
     * 加入会议
     *
     * @param confId 会议 id
     * @param password 会议密码
     * @param callback 结果回调
     *
     * \~english
     * Join conference
     * @param confId conference id
     * @param password Conference password
     * @param callback result callback
     */
    public void joinConference(final String confId, final String password, final EMValueCallBack<EMConference> callback) {
        joinConference(confId, password, null, callback);
    }

    /**
     * \~chinese
     * 加入会议
     *
     * @param confId 会议 id
     * @param password 会议密码
     * @param param 加入会议时 publish 自己本地数据参数
     * @param callback 结果回调
     *
     * \~english
     * Join conference
     * @param confId conference id
     * @param password Conference password
     * @param param join conference publish local stream param
     * @param callback result callback
     *
     * @Deprecated Use {@link #joinConference(String, String, EMValueCallBack)} instead
     * and publish stream use {@link #publish(EMStreamParam, EMValueCallBack)} in {@param callback#onSuccess()} callback.
     */
    @Deprecated
    public void joinConference(final String confId, final String password, final EMStreamParam param,
            final EMValueCallBack<EMConference> callback) {
        EMLog.d(TAG, "joinConference");

        if (TextUtils.isEmpty(accessToken) || TextUtils.isEmpty(appKey) || TextUtils.isEmpty(username)) {
            EMLog.e(TAG, "AccessToken AppKey or Username is not set");
            if (callback != null) {
                callback.onError(EMError.CALL_CONFERENCE_CREATE_FAILED, "If you want to use this feature without im account login, " +
                        "please setting the necessary parameters through the EMClient.getInstance().conferenceManager.set(String accessToken, String appKey, String username)" +
                        " method.");
            }
            return;
        }

        executorService.execute(new Runnable() {
            @Override
            public void run() {
                EMAError error = new EMAError();
                Map<String, Object> params = new HashMap<>();
                params.put("uid", EasyUtils.getMediaRequestUid(appKey, username));
                params.put("token", accessToken);
                params.put("confrId", confId);
                params.put("password", password);
                addCommonParams(params);
                String body = convertMapToJSONObject(params).toString();
                String result = emaCallManager.requestMediaFromServer(0, body, error);
                if (error.errCode() != EMError.EM_NO_ERROR) {
                    EMLog.d(TAG, "requestMediaFromServer failed " + error.errCode() + ", " + error.errMsg());
                    if (callback != null) {
                        callback.onError(error.errCode(), error.errMsg());
                    }
                    return;
                }

                final EMConference conference = new EMConference();
                conference.setConferenceId(confId);
                conference.setPassword(password);
                try {
                    JSONObject object =new JSONObject(result);
                    String ticket = object.optString("ticket");
                    conference.setConferenceType(EMConferenceType.from(object.optInt("type")));
                    mediaSession = mediaManager.newSessionWithTicket(ticket, "{'extension':'member'}", sessionDelegate);
                    mediaManager.setSession(mediaSession, EMClient.getInstance().getCurrentUser());
                    mediaManager.join(mediaSession, configWrap(param), new EMediaEntities.EMediaIdBlockType() {
                        @Override public void onDone(Object uid, EMediaEntities.EMediaError error) {
                            if (error != null) {
                                EMLog.d(TAG, "Join conference failed code=" + error.code + ", desc=" + error.errorDescription);
                                if (callback != null) {
                                    callback.onError(errorMap(error.code.errorcode), error.errorDescription);
                                }
                            } else {
                                EMLog.d(TAG, "Join conference success!");
                                if (conference.getConferenceType() == EMConferenceType.LiveStream) {
                                    conference.setConferenceRole(EMConferenceRole.from(mediaSession.role));
                                } else {
                                    conference.setConferenceRole(EMConferenceRole.Admin);
                                }
                                conference.setPubStreamId((String) uid, EMConferenceStream.StreamType.NORMAL);

                                if (callback != null) {
                                    callback.onSuccess(conference);
                                }
                            }
                        }
                    });
                } catch (JSONException e) {
                    e.printStackTrace();
                    if (callback != null) {
                        callback.onError(EMError.CALL_CONFERENCE_CREATE_FAILED, e.getMessage());
                    }
                }
            }
        });
    }

    /**
     * \~chinese
     * 通过 Ticket 加入会议
     *
     * @param ticket 加入会议需要的凭证
     * @param param 加入会议所需参数 {@link EMStreamParam}
     *
     * \~english
     * Join the conference via Ticket
     * @param ticket Join the required ticket for the conference
     * @param param Join the required param for the conference {@link EMStreamParam}
     */
    public void joinConferenceWithTicket(final String ticket, final EMStreamParam param, final EMValueCallBack<String> callback){
        EMLog.d(TAG, "joinConferenceWithTicket");
        executorService.execute(new Runnable() {
            @Override public void run() {
                try {
                    EMLog.d(TAG, "Join conference");
                    mediaSession = mediaManager.newSessionWithTicket(ticket, "{'extension':'member'}", sessionDelegate);
                    mediaManager.setSession(mediaSession, EMClient.getInstance().getCurrentUser());
                    mediaManager.join(mediaSession, configWrap(param), new EMediaEntities.EMediaIdBlockType() {
                        @Override public void onDone(Object uid, EMediaEntities.EMediaError error) {
                            if (error != null) {
                                EMLog.d(TAG, "Join conference with ticket failed code=" + error.code + ", desc=" + error.errorDescription);
                                if (callback != null) {
                                    callback.onError(errorMap(error.code.errorcode), error.errorDescription);
                                }
                            } else {
                                EMLog.d(TAG, "Join conference with ticket success!");
                                if (callback != null) {
                                    callback.onSuccess((String) uid);
                                }
                            }
                        }
                    });
                } catch (JSONException e) {
                    e.printStackTrace();
                    if (callback != null) {
                        callback.onError(EMError.CALL_CONFERENCE_CREATE_FAILED, e.getMessage());
                    }
                }
            }
        });
    }

    /**
     * \~chinese
     * 邀请其他人加入会议
     *
     * @param confId 会议 id
     * @param password 会议密码
     * @param username 被邀请者名字
     * @param extension 邀请他人加入的扩展信息
     * @param callback 结果回调
     *
     * \~english
     * Invite other people join in conference
     * @param confId conference id
     * @param password conference password
     * @param username The name of the invitee
     * @param extension Invitation extended information
     * @param callback result callback
     */
    public void inviteUserToJoinConference(final String confId, final String password, final String username,
            final String extension, final EMValueCallBack callback) {
        executorService.execute(new Runnable() {
            @Override public void run() {
                EMLog.d(TAG, "inviteUserToJoinConference");
                EMAError error = new EMAError();
                emaCallManager.inviteUserToJoinConference(confId, password, username, extension, error);
                if (error.errCode() != EMError.EM_NO_ERROR) {
                    EMLog.d(TAG, "inviteUserToJoinConference failed " + error.errCode() + ", " + error.errMsg());
                    callback.onError(error.errCode(), error.errMsg());
                    return;
                } else {
                    callback.onSuccess(null);
                }
            }
        });
    }

    /**
     * \~chinese
     * 管理员改变用户角色。
     *
     * @param confId   会议id
     * @param member   {@link EMConferenceMember},目前使用memberName进行的操作
     * @param toRole   目标角色，{@link EMConferenceRole}
     * @param callback 结果回调
     *
     * \~english
     * Admin grant a user's role.
     *
     * @param confId   Conference id
     * @param member   {@link EMConferenceMember}
     * @param toRole   Target role，{@link EMConferenceRole}
     * @param callback Result callback
     */
    public void grantRole(final String confId, final EMConferenceMember member, final EMConferenceRole toRole,
                          final EMValueCallBack<String> callback) {
        if (this.role.getRole() != EMConferenceRole.Admin) return;

        executorService.execute(new Runnable() {
            @Override
            public void run() {
                EMLog.d(TAG, "grantRole, " + member.memberName);
                EMAError error = new EMAError();
                Map<String, Object> params = new HashMap<>();
                params.put("uids", new Object[] {member.memberName});
                params.put("roleToken", EMConferenceManager.this.role.getRoleToken());
                params.put("role", toRole.code);
                addCommonParams(params);
                String body = convertMapToJSONObject(params).toString();
                String result = emaCallManager.requestMediaFromServer(1, body, error);
                if (error.errCode() != EMError.EM_NO_ERROR) {
                    EMLog.d(TAG, "grantRole failed " + error.errCode() + ", " + error.errMsg());
                    if (callback != null) {
                        callback.onError(error.errCode(), error.errMsg());
                    }
                } else {
                    EMLog.d(TAG, "grantRole success!");
                    if (callback != null) {
                        callback.onSuccess(result);
                    }
                }
            }
        });
    }

    protected void addCommonParams(Map<String, Object> params) {
        Map<String, Object> terminalMap = new HashMap<>();
        terminalMap.put("os", "Android");
        terminalMap.put("version", EMClient.VERSION);
        params.put("terminal", terminalMap);
    }

    /**
     * \~chinese
     * 销毁会议
     *
     * \~english
     * destroy conference
     *
     * @param callback
     */
    public void destroyConference(final EMValueCallBack callback) {
        EMLog.d(TAG, "destroyConference");

        if (this.role.getRole() != EMConferenceRole.Admin) return;

        executorService.execute(new Runnable() {
            @Override
            public void run() {
                EMAError error = new EMAError();
                Map<String, Object> params = new HashMap<>();
                params.put("roleToken", EMConferenceManager.this.role.getRoleToken());
                String body = convertMapToJSONObject(params).toString();
                String result = emaCallManager.requestMediaFromServer(3, body, error);
                if (error.errCode() != EMError.EM_NO_ERROR) {
                    EMLog.d(TAG, "destroyConference failed " + error.errCode() + ", " + error.errMsg());
                    if (callback != null) {
                        callback.onError(error.errCode(), error.errMsg());
                    }
                } else {
                    EMLog.d(TAG, "destroyConference success!");
                    if (callback != null) {
                        callback.onSuccess(null);
                    }
                }
            }
        });
    }

    /**
     * \~chinese
     * 退出会议
     *
     * \~english
     * Exit conference
     */
    public void exitConference(final EMValueCallBack callback) {
        EMLog.d(TAG, "Exit conference - async");
        mediaManager.exit(mediaSession, new EMediaEntities.EMediaIdBlockType() {
            @Override public void onDone(Object uid, EMediaEntities.EMediaError error) {
                if (error != null) {
                    EMLog.d(TAG, "Exit conference failed code=" + error.code + ", desc=" + error.errorDescription);
                    if (callback != null) {
                        callback.onError(errorMap(error.code.errorcode), error.errorDescription);
                    }
                } else {
                    EMLog.d(TAG, "Exit conference success");
                    if (callback != null) {
                        callback.onSuccess(null);
                    }
                }
                memberList.clear();
                subscribedStreamMap.clear();
                availableStreamMap.clear();
            }
        });
    }

    /**
     * \~chinese
     * 本地推流
     *
     * @param param 推送本地流时配置信息
     * @param callback 结果回调
     *
     * \~english
     * Publish local stream
     * @param param publish local stream config
     * @param callback result callback
     */
    public void publish(EMStreamParam param, final EMValueCallBack<String> callback) {
        EMLog.d(TAG, "Publish local stream");
        mediaManager.publish(mediaSession, configWrap(param), new EMediaEntities.EMediaIdBlockType() {
            @Override public void onDone(Object uid, EMediaEntities.EMediaError error) {
                if (error != null) {
                    EMLog.d(TAG, "Publish failed code=" + error.code + ", desc=" + error.errorDescription);
                    if (callback != null) {
                        callback.onError(errorMap(error.code.errorcode), error.errorDescription);
                    }
                } else {
                    EMLog.d(TAG, "Publish success Stream id - " + uid);
                    if (callback != null) {
                        callback.onSuccess((String) uid);
                    }
                }
            }
        });
    }

    /**
     * \~chinese
     * 取消本地推流
     *
     * @param pubStreamId 本地数据流 id
     *
     * \~english
     * UNPublish local stream
     * @param pubStreamId local stream id
     */
    public void unpublish(String pubStreamId, final EMValueCallBack<String> callback) {
        EMLog.d(TAG, "UNPublish local stream - async");
        mediaManager.unpublish(mediaSession,pubStreamId, new EMediaEntities.EMediaIdBlockType() {
            @Override public void onDone(Object uid, EMediaEntities.EMediaError error) {
                if (error != null) {
                    EMLog.d(TAG, "Unpublish failed code=" + error.code + ", desc=" + error.errorDescription);
                    if (callback != null) {
                        callback.onError(errorMap(error.code.errorcode), error.errorDescription);
                    }
                } else {
                    EMLog.d(TAG, "Unpublish success Stream id - " + uid);
                    if (callback != null) {
                        callback.onSuccess((String) uid);
                    }
                }
            }
        });
    }

    /**
     * \~chinese
     * 订阅成员推送流数据
     *
     * @param stream 当前操作的流
     * @param surfaceView 用来显示订阅的流画面的控件
     * @param callback 结果回调
     *
     * \~english
     * Subscribe member publish stream
     * @param stream current stream
     * @param surfaceView Displays the remote image controls
     * @param callback result callback
     */
    public void subscribe(final EMConferenceStream stream, EMCallSurfaceView surfaceView, final EMValueCallBack<String> callback) {
        EMLog.d(TAG, "Subscribe stream - async " + stream.toString());
        VideoViewRenderer renderer = surfaceView != null ? surfaceView.getRenderer() : null;
        mediaManager.subscribe(mediaSession, stream.getStreamId(), renderer, new EMediaEntities.EMediaIdBlockType() {
            @Override public void onDone(Object uid, EMediaEntities.EMediaError error) {
                if (error != null) {
                    EMLog.d(TAG, "Subscribe failed code=" + error.code + ", desc=" + error.errorDescription);
                    if (callback != null) {
                        callback.onError(errorMap(error.code.errorcode), error.errorDescription);
                    }
                } else {
                    EMLog.d(TAG, "Subscribe success Stream id - " + uid);
                    subscribedStreamMap.put(stream.getStreamId(), stream);
                    availableStreamMap.remove(stream.getStreamId());
                    if (callback != null) {
                        callback.onSuccess((String) uid);
                    }
                }
            }
        });
    }

    /**
     * \~chinese
     * 更新订阅成员推送流数据
     *
     * @param stream 当前操作的流
     * @param surfaceView 用来显示订阅的流画面的控件
     * @param callback 结果回调
     *
     * \~english
     * Update subscribe member publish stream
     * @param stream current stream
     * @param surfaceView Displays the remote image controls
     * @param callback result callback
     */
    public void updateSubscribe(final EMConferenceStream stream, EMCallSurfaceView surfaceView,final EMValueCallBack<String> callback) {
        EMLog.d(TAG, "Update subscribe stream - async " + stream.toString());
        VideoViewRenderer renderer = surfaceView != null ? surfaceView.getRenderer() : null;
        mediaManager.updateSubscribe(mediaSession, stream.getStreamId(), renderer, new EMediaEntities.EMediaIdBlockType() {
            @Override public void onDone(Object uid, EMediaEntities.EMediaError error) {
                if (error != null) {
                    EMLog.d(TAG, "Update subscribe failed code=" + error.code + ", desc=" + error.errorDescription);
                    if (callback != null) {
                        callback.onError(errorMap(error.code.errorcode), error.errorDescription);
                    }
                } else {
                    EMLog.d(TAG, "Update subscribe success Stream id - " + uid);
                    subscribedStreamMap.put(stream.getStreamId(), stream);
                    availableStreamMap.remove(stream.getStreamId());
                    if (callback != null) {
                        callback.onSuccess((String) uid);
                    }
                }
            }
        });
    }

    /**
     * \~chinese
     * 取消订阅成员推送流数据
     *
     * @param stream 当前流
     * @param callback 结果回调接口
     *
     * \~english
     * Unsubscribe member publish stream
     * @param stream current stream
     * @param callback result callback
     */
    public void unsubscribe(final EMConferenceStream stream, final EMValueCallBack<String> callback) {
        EMLog.d(TAG, "UNSubscribe stream - async " + stream.toString());
        mediaManager.unsubscribe(mediaSession, stream.getStreamId(), new EMediaEntities.EMediaIdBlockType() {
            @Override public void onDone(Object uid, EMediaEntities.EMediaError error) {
                if (error != null) {
                    EMLog.d(TAG, "Unsubscribe failed code=" + error.code + ", desc=" + error.errorDescription);
                    if (callback != null) {
                        callback.onError(errorMap(error.code.errorcode), error.errorDescription);
                    }
                } else {
                    EMLog.d(TAG, "Unsubscribe success Stream id - " + uid);
                    availableStreamMap.put(stream.getStreamId(), stream);
                    subscribedStreamMap.remove(stream.getStreamId());
                    if (callback != null) {
                        callback.onSuccess((String) uid);
                    }
                }
            }
        });
    }

    /**
     * \~chinese
     * 外部输入视频数据方法，此方法主要是为分享桌面回调使用
     * @param bitmap bitmap (格式 - ARGB_8888) 数据
     *
     * \~english
     * Input external video data
     * @param bitmap bitmap (format - ARGB_8888) input
     *
     */
    public void inputExternalVideoData(Bitmap bitmap) {
        mediaManager.inputExternalVideoData(bitmap);
    }

    /**
     * \~chinese
     * 外部输入数据方法
     * @param data 视频数据流，需要是 YUV (格式 - YUV420SP) 数据
     * @param width 视频数据帧宽
     * @param height 视频数据帧高
     * @param rotation 旋转角度
     *
     * \~english
     * Input external view data
     * @param data Video data YUV (format - YUV420SP) data
     * @param width Video frame width
     * @param height Video frame height
     * @param rotation Video frame rotation
     *
     */
    public void inputExternalVideoData(byte[] data, int width, int height, int rotation) {
        mediaManager.inputExternalVideoData(data, width, height, rotation);
    }

    /**
     * \~chinese
     * 开启正在说话监听器
     * @param interval {@link EMConferenceListener#onSpeakers(List)} 回调间隔
     *
     * \~english
     * Start speaking monitor
     * @param interval {@link EMConferenceListener#onSpeakers(List)} interval
     */
    public void startMonitorSpeaker(int interval) {
        mediaManager.setAudioTalkerInterval(interval, new EMediaEntities.EMediaIdBlockType() {
            @Override
            public void onDone(Object uid, EMediaEntities.EMediaError error) {
                EMLog.d(TAG, "error: " + error.code + " desc:" + error.errorDescription);
            }
        });
    }

    /**
     * \~chinese
     * 停止正在说话监听器
     *
     * \~english
     * Stop speaker monitor
     */
    public void stopMonitorSpeaker(){
        mediaManager.stopAudioTalker();
    }


    /**
     * \~chinese
     * 设置会议模式
     *
     * \~english
     * config conference mode
     */
    public void setConferenceMode(EMConferenceListener.ConferenceMode mode) {
        conferenceMode = mode;
        setSubscribeAudioMixEnable();
    }

    /**
     * 通知 rtc 底层是否订阅 mix 流
     */
    private void setSubscribeAudioMixEnable() {
        if (conferenceMode == EMConferenceListener.ConferenceMode.LARGE) {
            mediaManager.setSubscribeAudioMixEnabled(false);
        }else{
            mediaManager.setSubscribeAudioMixEnabled(true);
        }
    }

    /**
     * \~chinese
     * 设置显示自己本地预览画面控件
     *
     * @param localView 显示本地图像的控件
     *
     * \~english
     * Set local surface view
     * @param localView Displays the local image controls
     */
    public void setLocalSurfaceView(EMCallSurfaceView localView) {
        VideoViewRenderer localRender = null;
        if (localView != null) {
            localRender = localView.getRenderer();
        }
        mediaManager.setLocalPreviewView(localRender);
    }

    /**
     * \~chinese
     * 更新显示本地画面控件
     *
     * @param localView 显示本地图像的控件
     *
     * \~english
     * Update local surface view
     * @param localView Displays the local image controls
     */
    public void updateLocalSurfaceView(EMCallSurfaceView localView) {
        mediaManager.setVideoViews(null, localView == null ? null : localView.getRenderer(), null, true);
    }

    /**
     * \~chinese
     * 更新显示远端画面控件
     *
     * @param streamId 当前控件显示的流 id
     * @param remoteView 显示远端图像控件
     *
     * \~english
     * Update remote surface view
     * @param streamId current stream id
     * @param remoteView Displays the remote image controls
     */
    public void updateRemoteSurfaceView(String streamId, EMCallSurfaceView remoteView) {
        mediaManager.setVideoViews(streamId, null, remoteView == null ? null : remoteView.getRenderer(), false);
    }

    /**
     *  \~chinese
     *  更新视频最大码率
     *
     *  @param maxKbps 最大码率
     *
     *  \~english
     *  Update video maximum bit rate
     *
     *  @param maxKbps Maximum bit rate
     */
    public void updateVideoMaxKbps(int maxKbps) {
        mediaManager.updateVideoMaxKbps(maxKbps);
    }

    /**
     * \~chinese
     * 获取当前摄像头 id， 0 表示后置摄像头，1 表示前置摄像头
     *
     * \~english
     * get current camera id, 0 back, 1 front
     */
    public int getCameraId() {
        return mediaManager.getCameraFacing();
    }

    /**
     * \~chinese
     * 切换摄像头
     *
     * \~english
     * Switch camera
     */
    public void switchCamera() {
        mediaManager.switchCamera();
    }

    /**
     * \~chinese
     * 关闭视频传输
     *
     * \~english
     * Close video transfer
     */
    public void closeVideoTransfer() {
        mediaManager.setVideoEnabled(false);
    }

    /**
     * \~chinese
     * 打开视频传输
     *
     * \~english
     * Open video transfer
     */
    public void openVideoTransfer() {
        mediaManager.setVideoEnabled(true);
    }

    /**
     * \~chinese
     * 关闭语音传输
     *
     * \~english
     * Close voice transfer
     */
    public void closeVoiceTransfer() {
        mediaManager.setMuteEnabled(true);
    }

    /**
     * \~chinese
     * 打开语音传输
     *
     * \~english
     * Open voice transfer
     */
    public void openVoiceTransfer() {
        mediaManager.setMuteEnabled(false);
    }

    /**
     * \~chinese
     * 启用统计
     *
     * @param enable 是否启用统计
     *
     * \~english
     * enable statistics
     * @params enable enable statistics
     */
    public void enableStatistics(boolean enable) {
        if (mediaSession != null) {
            mediaManager.enableStatistics(mediaSession, enable);
        }else{
            EMLog.e(TAG, "Conference no start");
        }
    }

    /**
     * \~chinese
     * 获取当前会议成员
     *
     * \~english
     * Get conference member list
     */
    public List<EMConferenceMember> getConferenceMemberList() {
        return memberList;
    }

    /**
     * \~chinese
     * 获取当前会议可订阅 Stream
     *
     * \~english
     * Get subscribable stream map
     */
    public Map<String, EMConferenceStream> getAvailableStreamMap() {
        return availableStreamMap;
    }

    /**
     * \~chinese
     * 获取当前会议已订阅 Stream
     *
     * \~english
     * get subscribed stream list
     */
    public Map<String, EMConferenceStream> getSubscribedStreamMap() {
        return subscribedStreamMap;
    }

    /**
     * \~chinese
     * 设置本地视频view镜像，可在通话过程中动态设置
     * @param mirror
     *
     * \~english
     * Set local video view mirror, can be set during a video call.
     * @param mirror
     */
    public void setLocalVideoViewMirror(@EMMirror.MIRROR int mirror) {
        RtcConnection.MIRROR rtcMirror = EMMirror.convert(mirror);
        mediaManager.setLocalVideoViewMirror(rtcMirror);
    }

    /**
     * 多人会议监听器
     */
    EMediaSession.EMediaSessionDelegate sessionDelegate = new EMediaSession.EMediaSessionDelegate() {
        @Override public void joinMember(EMediaSession mediaSession, EMediaEntities.EMediaMember mediaMember) {
            EMLog.d(TAG, "onMemberJoined() memberName: " + mediaMember.memberName + ", extension: " + mediaMember.extension);
            EMConferenceMember containMember = memberContains(mediaMember.memberName);
            if (null != containMember) {
                memberList.remove(containMember);
            }

            EMConferenceMember member = EMConferenceMember.from(mediaMember);
            memberList.add(member);

            synchronized (conferenceListeners) {
                for (EMConferenceListener listener : conferenceListeners.subList(0, conferenceListeners.size())) {
                    listener.onMemberJoined(member);
                }
            }
        }

        @Override public void exitMember(EMediaSession mediaSession, EMediaEntities.EMediaMember mediaMember) {
            EMLog.d(TAG, "onMemberExited() memberName: " + mediaMember.memberName + ", extension: " + mediaMember.extension);
            EMConferenceMember containMember = memberContains(mediaMember.memberName);
            if (null != containMember) {
                memberList.remove(containMember);
            }

            synchronized (conferenceListeners) {
                for (EMConferenceListener listener : conferenceListeners.subList(0, conferenceListeners.size())) {
                    listener.onMemberExited(EMConferenceMember.from(mediaMember));
                }
            }
        }

        @Override public void addStream(EMediaSession mediaSession, EMediaStream mediaStream) {
            EMLog.d(TAG, "onStreamAdded() memberName: "
                    + mediaStream.memberName + ", extension: "
                    + mediaStream.extension + ", streamId: "
                    + mediaStream.streamId + ", streamName: "
                    + mediaStream.streamName + ", streamType: "
                    + mediaStream.streamType + ", audioOff: "
                    + mediaStream.audioOff + ", videoOff: "
                    + mediaStream.videoOff);
            EMConferenceStream stream = new EMConferenceStream();
            stream.init(mediaStream);
            availableStreamMap.put(stream.getStreamId(), stream);
            synchronized (conferenceListeners) {
                for (EMConferenceListener listener : conferenceListeners.subList(0, conferenceListeners.size())) {
                    listener.onStreamAdded(stream);
                }
            }
        }

        @Override public void removeStream(EMediaSession mediaSession, EMediaStream mediaStream) {
            EMLog.d(TAG, "onStreamRemoved() memberName: "
                    + mediaStream.memberName + ", extension: "
                    + mediaStream.extension + ", streamId: "
                    + mediaStream.streamId + ", streamName: "
                    + mediaStream.streamName + ", streamType: "
                    + mediaStream.streamType + ", audioOff: "
                    + mediaStream.audioOff + ", videoOff: "
                    + mediaStream.videoOff);
            EMConferenceStream stream = null;
            if (availableStreamMap.containsKey(mediaStream.streamId)) {
                stream = availableStreamMap.get(mediaStream.streamId);
                availableStreamMap.remove(mediaStream.streamId);
            } else if (subscribedStreamMap.containsKey(mediaStream.streamId)) {
                stream = subscribedStreamMap.get(mediaStream.streamId);
                subscribedStreamMap.remove(mediaStream.streamId);
            } else {
                stream = new EMConferenceStream();
            }
            stream.init(mediaStream);
            mediaManager.unsubscribe(mediaSession, mediaStream.streamId, null);
            synchronized (conferenceListeners) {
                for (EMConferenceListener listener : conferenceListeners.subList(0, conferenceListeners.size())) {
                    listener.onStreamRemoved(stream);
                }
            }
        }

        @Override public void updateStream(EMediaSession mediaSession, EMediaStream mediaStream) {
            EMLog.d(TAG, "onStreamUpdate() memberName: "
                    + mediaStream.memberName + ", extension: "
                    + mediaStream.extension + ", streamId: "
                    + mediaStream.streamId + ", streamName: "
                    + mediaStream.streamName + ", streamType: "
                    + mediaStream.streamType + ", audioOff: "
                    + mediaStream.audioOff + ", videoOff: "
                    + mediaStream.videoOff);
            EMConferenceStream stream = null;
            if (availableStreamMap.containsKey(mediaStream.streamId)) {
                stream = availableStreamMap.get(mediaStream.streamId);
            } else if (subscribedStreamMap.containsKey(mediaStream.streamId)) {
                stream = subscribedStreamMap.get(mediaStream.streamId);
            } else {
                stream = new EMConferenceStream();
            }
            stream.init(mediaStream);
            synchronized (conferenceListeners) {
                for (EMConferenceListener listener : conferenceListeners.subList(0, conferenceListeners.size())) {
                    listener.onStreamUpdate(stream);
                }
            }
        }

        @Override public void passiveCloseReason(EMediaSession mediaSession, int code, String reason) {
            EMLog.d(TAG, "onPassiveLeave() code: " + code + ", reason: " + reason);
            synchronized (conferenceListeners) {
                for (EMConferenceListener listener : conferenceListeners.subList(0, conferenceListeners.size())) {
                    listener.onPassiveLeave(code, reason);
                }
            }
        }

        @Override public void notice(EMediaSession mediaSession, EMediaDefines.EMediaNoticeCode mediaNoticeCode, String arg1, String arg2,
                Object arg3) {
            EMLog.d(TAG, "Notice code: " + mediaNoticeCode + ", arg1=" + arg1 + ", arg2=" + arg2 + ", arg3=" + arg3);
            synchronized (conferenceListeners) {
                for (EMConferenceListener listener : conferenceListeners.subList(0, conferenceListeners.size())) {
                    EMConferenceListener.ConferenceState state = stateMap(mediaNoticeCode.noticeCode);
                    if (state == EMConferenceListener.ConferenceState.STATE_PUBLISH_SETUP) {
                        listener.onStreamSetup(arg1);
                    } else if (state == EMConferenceListener.ConferenceState.STATE_SUBSCRIBE_SETUP) {
                        listener.onStreamSetup(arg1);
                    } else if (state == EMConferenceListener.ConferenceState.STATE_STATISTICS) {
                        EMLog.d(TAG, "Notice state: " + state + ", arg1=" + arg1 + ", arg2=" + arg2 + ", arg3=" + arg3);
                        listener.onStreamStatistics(new EMStreamStatistics(arg2, (RtcConnection.RtcStatistics) arg3));
                    } else if (state == EMConferenceListener.ConferenceState.STATE_CUSTOM_MSG
                            || state == EMConferenceListener.ConferenceState.STATE_P2P_PEER_EXIT
                            || state == EMConferenceListener.ConferenceState.STATE_TAKE_CAMERA_PICTURE) {
                    } else if (state == EMConferenceListener.ConferenceState.STATE_AUDIO_TALKERS) {
                        Map<String, EMediaTalker> talkers = (Map<String, EMediaTalker>) arg3;
                        List<String> speakers = new ArrayList<>(talkers.keySet());
                        listener.onSpeakers(speakers);
                    } else {
                        listener.onConferenceState(state);
                    }
                }
            }
        }

        @Override
        public void changeRole(EMediaSession session) {
            int role = session.role;
            EMLog.i(TAG, "updateRole: " + role);
            synchronized (conferenceListeners) {
                for (EMConferenceListener listener : conferenceListeners.subList(0, conferenceListeners.size())) {
                    listener.onRoleChanged(EMConferenceRole.from(role));
                }
            }
        }
    };

    /**
     * 包装 publish 配置信息
     * @param param 用户提供的配置参数
     * @return 包装成 RTC 底层需要的参数
     */
    private EMediaPublishConfiguration configWrap(EMStreamParam param) {
        EMediaPublishConfiguration config = null;
        if (param == null) {
            return config;
        }
        if (param.streamType == EMConferenceStream.StreamType.NORMAL) {
            if (conferenceMode == EMConferenceListener.ConferenceMode.LARGE) {
                config = EMediaPublishConfiguration.initAudioMixConfig();
            } else {
                config = EMediaPublishConfiguration.initNormalConfig();
            }
        }else{
            config = EMediaPublishConfiguration.initDesktopConfig();
        }
        config.setExtension(param.extension);
        config.setMute(param.audioOff);
        config.setVideoOff(param.videoOff);
        config.setVwidth(param.videoWidth);
        config.setVheight(param.videoHeight);
        config.setPubView(param.shareView);
        config.setUseBackCamera(param.useBackCamera);
        config.setMaxVideoKbps(param.maxVideoKbps);
        config.setMaxAudioKbps(param.maxAudioKbps);
        config.setExternalVideoSource(param.usingExternalSource);
        {
            RtcConnection.enableFixedVideoResolution(param.enableFixedVideoResolution);
            RtcConnection.setMinVideoKbps(param.minVideoKbps);
            RtcConnection.setAudioSampleRate(param.audioSampleRate);
        }
        return config;
    }

    private static JSONObject convertMapToJSONObject(Map<String, Object> map) {
        JSONObject obj = new JSONObject();
        Set<Map.Entry<String, Object>> entries = map.entrySet();
        for (Map.Entry<String, Object> entry : entries) {
            Object result;
            Object value = entry.getValue();
            if (value instanceof Map) { // is a JSONObject
                result = convertMapToJSONObject((Map<String, Object>) value);
            } else if (value instanceof List) { // is a JSONArray
                result = new JSONArray();
                for (Object item : (List) value) {
                    ((JSONArray)result).put(item);
                }
            } else if (value instanceof Object[]) {
                result = new JSONArray();
                for (Object item : (Object[]) value) {
                    ((JSONArray)result).put(item);
                }
            } else { // is common value
                result = value;
            }
            try {
                obj.put(entry.getKey(), result);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
        return obj;
    }

    private EMConferenceMember memberContains(String memberName) {
        for (EMConferenceMember item : memberList) {
            if (item.memberName.equals(memberName)) {
                return item;
            }
        }
        return null;
    }

    /**
     * 错误码映射
     *
     * @param code rtc 错误码
     * @return im 错误码
     */
    private int errorMap(int code) {
        int error = 0;
        switch (code) {
            case 0:
                error = EMError.EM_NO_ERROR;
                break;
            case -1:
                error = EMError.GENERAL_ERROR;
                break;
            case -102:
                error = EMError.CALL_INVALID_PARAMS;
                break;
            case -106:
                error = EMError.CALL_CONFERENCE_CANCEL;
                break;
            case -108:
                error = EMError.NETWORK_ERROR;
                break;
            case -109:
                error = EMError.CALL_CONNECTION_ERROR;
                break;
            case -112:
                error = EMError.CALL_CONNECTION_TIMEOUT;
                break;
            case -113:
                error = EMError.CALL_JOIN_TIMEOUT;
                break;
            case -122:
                error = EMError.CALL_ALREADY_JOIN;
                break;
            case -123:
                error = EMError.CALL_ALREADY_PUB;
                break;
            case -124:
                error = EMError.CALL_ALREADY_SUB;
                break;
            case -142:
                error = EMError.CALL_NO_SESSION;
                break;
            case -143:
                error = EMError.CALL_NO_PUBLISH;
                break;
            case -144:
                error = EMError.CALL_NO_SUBSCRIBE;
                break;
            case -145:
                error = EMError.CALL_NO_STREAM;
                break;
            case -404:
                error = EMError.CALL_CONNECTION_ERROR;
                break;
            case -410:
                error = EMError.CALL_OTHER_DEVICE;
                break;
            case -411:
                error = EMError.CALL_CONFERENCE_DISMISS;
                break;
            case -500:
                error = EMError.CALL_TICKET_INVALID;
                break;
            case -502:
                error = EMError.CALL_TICKET_EXPIRED;
                break;
            case -504:
                error = EMError.CALL_SESSION_EXPIRED;
                break;
            case -506:
                error = EMError.CALL_CONFERENCE_NO_EXIST;
                break;
            default:
                error = EMError.GENERAL_ERROR;
                break;
        }
        return error;
    }

    /**
     * 状态映射
     *
     * @param state rtc 返回状态
     * @return im 状态
     */
    private EMConferenceListener.ConferenceState stateMap(int state) {
        EMConferenceListener.ConferenceState conferenceState;
        switch (state) {
            case 0:
                conferenceState = EMConferenceListener.ConferenceState.STATE_NORMAL;
                break;
            case 100:
                conferenceState = EMConferenceListener.ConferenceState.STATE_STATISTICS;
                break;
            case 120:
                conferenceState = EMConferenceListener.ConferenceState.STATE_DISCONNECTION;
                break;
            case 121:
                conferenceState = EMConferenceListener.ConferenceState.STATE_RECONNECTION;
                break;
            case 122:
                conferenceState = EMConferenceListener.ConferenceState.STATE_POOR_QUALITY;
                break;
            case 123:
                conferenceState = EMConferenceListener.ConferenceState.STATE_PUBLISH_SETUP;
                break;
            case 124:
                conferenceState = EMConferenceListener.ConferenceState.STATE_SUBSCRIBE_SETUP;
                break;
            case 125:
                conferenceState = EMConferenceListener.ConferenceState.STATE_TAKE_CAMERA_PICTURE;
                break;
            case 126:
                conferenceState = EMConferenceListener.ConferenceState.STATE_CUSTOM_MSG;
                break;
            case 131:
                conferenceState = EMConferenceListener.ConferenceState.STATE_AUDIO_TALKERS;
                break;
            case 181:
                conferenceState = EMConferenceListener.ConferenceState.STATE_P2P_PEER_EXIT;
                break;
            case 201:
                conferenceState = EMConferenceListener.ConferenceState.STATE_OPEN_CAMERA_FAIL;
                break;
            case 202:
                conferenceState = EMConferenceListener.ConferenceState.STATE_OPEN_MIC_FAIL;
                break;
            default:
                conferenceState = EMConferenceListener.ConferenceState.STATE_NORMAL;
                break;
        }
        return conferenceState;
    }
}
