/*
 *  * EaseMob CONFIDENTIAL
 * __________________
 * Copyright (C) 2017 EaseMob Technologies. All rights reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat.adapter;

import com.hyphenate.EMError;
import com.hyphenate.chat.EMCursorResult;
import com.hyphenate.chat.EMGroupInfo;

import java.util.List;
import java.util.Map;

public class EMAGroupManager extends EMABase {

    /**
      * \brief Add a listener to group manager.
      *
      * @param  A group manager listener.
      * @return NA
      */
    public void addListener(EMAGroupManagerListener listener) {
    	nativeAddListener(listener);
    }
    native void nativeAddListener(EMAGroupManagerListener listener);
    
    /**
      * \brief Remove a listener.
      *
      * @param  A group manager listener.
      * @return NA
      */
    public void removeListener(EMAGroupManagerListener listener) {
    	nativeRemoveListener(listener);
	}
    native void nativeRemoveListener(EMAGroupManagerListener listener);
    
    /**
      * \brief Remove all the listeners.
      *
      * @param  NA
      * @return NA
      */
    public void clearListeners() {
        nativeClearListeners();
    }
    native void nativeClearListeners();
    
    /**
      * \brief Get groups for login user from memory.
      *
      * @param  EMAError used for output.
      * @return All user joined groups in memory.
      */
    public List<EMAGroup> allMyGroups(EMAError error) {
    	return nativeAllMyGroups(error);
    }
    native List<EMAGroup> nativeAllMyGroups(EMAError error);
    
    /**
      * \brief Fetch all groups for login user from server.
      *
      * Note: Groups in memory will be updated.
      * @param  error used for output.
      * @return All user joined groups.
      */
    public List<EMAGroup> fetchAllMyGroups(EMAError error) {
    	return nativeFetchAllMyGroups(error);
    }
    native List<EMAGroup> nativeFetchAllMyGroups(EMAError error);

    /**
     * \brief Fetch my groups by page for login user from server.
     *
     * Note: Groups in memory will be updated.
     * @param pageIndex
     * @param pageSize
     * @param  error used for output.
     * @return All user joined groups.
     */
    public List<EMAGroup> fetchAllMyGroupsWithPage(int pageIndex, int pageSize, EMAError error) {
        return nativeFetchAllMyGroupsWithPage(pageIndex, pageSize, error);
    }
    native List<EMAGroup> nativeFetchAllMyGroupsWithPage(int pageIndex, int pageSize, EMAError error);


    /**
      * \brief Fetch app's public groups.
      *
      * Note: User can input empty string as cursor at the first time.
      * @param  Page's cursor.
      * @param  Page's size.
      * @param  EMAError used for output.
      * @return Wrapper of next page's cursor and current page result.
      */
    public EMCursorResult<EMGroupInfo> fetchPublicGroupsWithCursor(
        String cursor,
        int pageSize,
        EMAError error) {
    	return nativeFetchPublicGroupsWithCursor(
            cursor,
            pageSize,
            error);
	}
    native EMCursorResult<EMGroupInfo> nativeFetchPublicGroupsWithCursor(
            String cursor,
            int pageSize,
            EMAError error);
   
    /**
      * \brief Create a new group.
      *
      * Note: Login user will be the owner of created .
      * @param  Group's subject.
      * @param  Group's description.
      * @param  Welcome message that will be sent to invited user.
      * @param  Group's setting.
      * @param  Group's members.
      * @param  EMAError used for output.
      * @return The created group.
      */
    public EMAGroup createGroup(
        String subject,
        String description,
        String welcomeMessage,
        EMAGroupSetting setting,
        List<String> members,
        boolean inviteNeedConfirm,
        EMAError error) {
    return nativeCreateGroup(
            subject,
            description,
            welcomeMessage,
            setting,
            members,
            inviteNeedConfirm,
            error);
    }
    native EMAGroup nativeCreateGroup(
            String subject,
            String description,
            String welcomeMessage,
            EMAGroupSetting setting,
            List<String> members,
            boolean inviteNeedConfirm,
            EMAError error);
    
    /**
      * \brief Join a public group.
      *
      * Note: The group's style must be PUBLIC_JOIN_OPEN, or will return error.
      * @param  Group's ID.
      * @param  EMAError used for output.
      * @return The joined group.
      */
    public EMAGroup joinPublicGroup(
        String groupId,
        EMAError error
        ) {
    	return nativeJoinPublicGroup(groupId,error);
    }
    native EMAGroup nativeJoinPublicGroup(
            String groupId,
            EMAError error
            );
    
    /**
      * \brief Apply to join a public group, need owner's approval.
      *
      * Note: The group's style must be PUBLIC_JOIN_APPROVAL, or will return error.
      * @param  groupId Group's ID.
      * @param  nickName Nick name in the group.
      * @param  message Apply message, that will be sent to group owner.
      * @param  error EMAError used for output.
      * @return The group that try to join.
      */
    public EMAGroup applyJoinPublicGroup(
        String groupId,
        String nickName,
        String message,
        EMAError error) {
    	return nativeApplyJoinPublicGroup(
            groupId,
            nickName,
            message,
            error
            );
    }
    native EMAGroup nativeApplyJoinPublicGroup(
            String groupId,
            String nickName,
            String message,
            EMAError error
            );
    
    /**
      * \brief Leave a group.
      *
      * Note: Group's owner can't leave the group.
      * @param  groupId Group's ID.
      * @param  error EMAError used for output.
      * @return The leaved group.
      */
    public void leaveGroup(
        String groupId,
        EMAError error
        ) {
    	nativeLeaveGroup(
            groupId,
            error
            );
    }
    native void nativeLeaveGroup(
            String groupId,
            EMAError error
            );
    
    /**
      * \brief Destroy a group.
      *
      * Note: Only group's owner can destroy the group.
      * @param  groupId Group's ID.
      * @param  error EMAError used for output.
      * @return The destroyed group.
      */
    public void destroyGroup(
        String groupId,
        EMAError error
        ) {
    	nativeDestroyGroup(
             groupId,
             error
            );
    }
    native void nativeDestroyGroup(
            String groupId,
            EMAError error
            );
    
    /**
      * \brief Add members to a group.
      *
      * Note: Maybe user don't have the priviledge, it depends on group's setting.
      * @param  groupId Group's ID.
      * @param  members Invited users.
      * @param  welcomeMessage Welcome message that will be sent to invited user.
      * @param  error EMAError used for output.
      * @return The group that added members.
      */
    public EMAGroup addGroupMembers(
        String groupId,
        List<String> members,
        String welcomeMessage,
        EMAError error
        ) {
    	return nativeAddGroupMembers(
            groupId,
            members,
            welcomeMessage,
            error
            );
    }
    native EMAGroup nativeAddGroupMembers(
            String groupId,
            List<String> members,
            String welcomeMessage,
            EMAError error
            );
    
    /**
      * \brief Remove members from a group.
      *
      * Note: Only group's owner can remove members.
      * @param  groupId Group's ID.
      * @param  members Removed users.
      * @param  error EMAError used for output.
      * @return The group that removed members.
      */
    public EMAGroup removeGroupMembers(
        String groupId,
        List<String> members,
        EMAError error
        ) {
    	return nativeRemoveGroupMembers(
            groupId,
            members,
            error
            );
    }
    native EMAGroup nativeRemoveGroupMembers(
            String groupId,
            List<String> members,
            EMAError error
            );
    
    /**
      * \brief Block group's members, the blocked user can't send message in the group.
      *
      * Note: Only group's owner can block other members.
      * @param  groupId Group's ID.
      * @param  members Blocked users.
      * @param  error EMAError used for output.
      * @param  reason The reason that why block the members.
      * @return The group that blocked members.
      */
    public EMAGroup blockGroupMembers(
        String groupId,
        List<String> members,
        EMAError error,
        String reason
        ) {
    	return nativeBlockGroupMembers(
            groupId,
            members,
            error,
            reason
            );
    }
    native EMAGroup nativeBlockGroupMembers(
            String groupId,
            List<String> members,
            EMAError error,
            String reason
            );
    
    /**
      * \brief Unblock group's members.
      *
      * Note: Only group's owner can unblock other members.
      * @param  groupId Group's ID.
      * @param  members Unblocked users.
      * @param  error EMAError used for output.
      * @return The group that unblocked members.
      */
    public EMAGroup unblockGroupMembers(
        String groupId,
        List<String> members,
        EMAError error
        ) {
    return nativeUnblockGroupMembers(
            groupId,
            members,
            error
            );
    }
    native EMAGroup nativeUnblockGroupMembers(
            String groupId,
            List<String> members,
            EMAError error
            );
    
    /**
      * \brief Change group's subject.
      *
      * Note: Only group's owner can change group's subject.
      * @param  groupId Group's ID.
      * @param  newSubject The new group subject.
      * @param  error EMAError used for output.
      * @return The group that with new subject.
      */
    public EMAGroup changeGroupSubject(
        String groupId,
        String newSubject,
        EMAError error
        ) {
    	return nativeChangeGroupSubject(
            groupId,
            newSubject,
            error
            );
    }
    native EMAGroup nativeChangeGroupSubject(
            String groupId,
            String newSubject,
            EMAError error
            );
    
    /**
      * \brief Change group's description.
      *
      * Note: Only group's owner can change group's description.
      * @param  groupId Group's ID.
      * @param  newDescription The new group description.
      * @param  error EMAError used for output.
      * @return The group that with new description.
      */
    public EMAGroup changeGroupDescription(
        String groupId,
        String newDescription,
        EMAError error
        ) {
    	return nativeChangeGroupDescription(
            groupId,
            newDescription,
            error
            );
    }
    native EMAGroup nativeChangeGroupDescription(
            String groupId,
            String newDescription,
            EMAError error
            );
    
    /**
      * \brief Get group's specification.
      *
      * @param  groupId Group's ID.
      * @param  error EMAError used for output.
      * @param  fetchMembers Whether get group's members.
      * @return The group that update it's specification.
      */
    public EMAGroup fetchGroupSpecification(
        String groupId,
        EMAError error,
        boolean fetchMembers
        ) {
    	return nativeFetchGroupSpecification(
            groupId,
            error,
            fetchMembers
            );
    }
    native EMAGroup nativeFetchGroupSpecification(
            String groupId,
            EMAError error,
            boolean fetchMembers
            );
    
    /**
      * \brief Search for a public group.
      *
      * @param  groupId Group's ID.
      * @param  error EMAError used for output.
      * @return The search result.
      */
    public EMAGroup searchPublicGroup(
        String groupId,
        EMAError error
        ) {
    	return nativeSearchPublicGroup(
            groupId,
            error
            );
    }
    native EMAGroup nativeSearchPublicGroup(
            String groupId,
            EMAError error
            );
    
    /**
      * \brief Block group message.
      *
      * Note: Owner can't block the group message.
      * @param  groupId Group's ID.
      * @param  error EMAError used for output.
      * @return The group that blocked message.
      */
    public EMAGroup blockGroupMessage(
        String groupId,
        EMAError error
        ) {
    	return nativeBlockGroupMessage(
            groupId,
            error
            );
    }
    native EMAGroup nativeBlockGroupMessage(
            String groupId,
            EMAError error
            );
    
    /**
      * \brief Unblock group message.
      *
      * @param  groupId Group's ID.
      * @param  error EMAError used for output.
      * @return The group that unclocked message.
      */
    public EMAGroup unblockGroupMessage(
        String groupId,
        EMAError error
        ) {
    	return nativeUnblockGroupMessage(
            groupId,
            error
            );
    }
    native EMAGroup nativeUnblockGroupMessage(
            String groupId,
            EMAError error
            );
    
    /**
      * \brief Accept user's join application.
      *
      * Note: Only group's owner can approval someone's application.
      * @param  groupId Group's ID.
      * @param  user The user that send the application.
      * @param  error EMAError used for output.
      * @return NA.
      */
    public void acceptJoinGroupApplication(
        String groupId,
        String user,
        EMAError error) {
    	nativeAcceptJoinGroupApplication(
            groupId,
            user,
            error);
    }
    native void nativeAcceptJoinGroupApplication(
            String groupId,
            String user,
            EMAError error);
    
    /**
      * \brief Reject user's join application.
      *
      * Note: Only group's owner can reject someone's application.
      * @param  groupId Group's ID.
      * @param  user The user that send the application.
      * @param  reason The reject reason.
      * @param  error EMAError used for output.
      * @return NA.
      */
    public void declineJoinGroupApplication(
        String groupId,
        String user,
        String reason,
        EMAError error) {
    nativeDeclineJoinGroupApplication(
            groupId,
            user,
            reason,
            error);
    }
    native void nativeDeclineJoinGroupApplication(
            String groupId,
            String user,
            String reason,
            EMAError error);
    
    public void loadAllMyGroupsFromDB() {
        nativeLoadAllMyGroupsFromDB();
    }
    native void nativeLoadAllMyGroupsFromDB();
    
    

    /**
      * \brief accept group's invitation.
      *
      * @param  groupId Group's ID.
      * @param  inviter Inviter.
      * @param  error EMError used for output.
      * @return The group user has accepted.
      */
    public EMAGroup acceptInvitationFromGroup(
        String groupId,
        String inviter,
        EMAError error) {
        return nativeacceptInvitationFromGroup(groupId, inviter, error);
    }
    native EMAGroup nativeacceptInvitationFromGroup(
            String groupId,
            String inviter,
            EMAError error);
    
    /**
      * \brief decline group's invitation.
      *
      * @param  groupId Group's ID.
      * @param  inviter Inviter.
      * @param  reason The decline reason.
      * @param  error EMError used for output.
      * @return NA.
      */
    public void declineInvitationFromGroup(
        String groupId,
        String inviter,
        String reason,
        EMAError error) {
        nativedeclineInvitationFromGroup(groupId, inviter, reason, error);
    }
    native void nativedeclineInvitationFromGroup(
            String groupId,
            String inviter,
            String reason,
            EMAError error);

    // ============================= group_reform new add api begin
    public EMCursorResult<String> fetchGroupMembers(
            String groupId,
            String cursor,
            int pageSize,
            EMAError error) {
        return nativeFetchGroupMembers(groupId, cursor, pageSize, error);
    }
    native EMCursorResult<String> nativeFetchGroupMembers(String groupId, String cursor, int pageSize, EMAError error);

    public EMAGroup transferGroupOwner(
            String groupId,
            String newOwner,
            EMAError error) {
        return nativeTransferGroupOwner(groupId, newOwner, error);
    }
    native EMAGroup nativeTransferGroupOwner(String groupId, String newOwner, EMAError error);

    public EMAGroup addGroupAdmin(
            String groupId,
            String admin,
            EMAError error) {
        return nativeAddGroupAdmin(groupId, admin, error);
    }
    native EMAGroup nativeAddGroupAdmin(String groupId, String admin, EMAError error);


    public EMAGroup removeGroupAdmin(
            String groupId,
            String admin,
            EMAError error) {
        return nativeRemoveGroupAdmin(groupId, admin, error);

    }
    native EMAGroup nativeRemoveGroupAdmin(
            String groupId,
            String admin,
            EMAError error);

    public EMAGroup muteGroupMembers(
            String groupId,
            List<String> muteMembers,
            long duration,
            EMAError error) {
        return nativeMuteGroupMembers(groupId, muteMembers, duration, error);
    }
    native EMAGroup nativeMuteGroupMembers(String groupId, List<String> muteMembers, long duration, EMAError error);

    public EMAGroup unMuteGroupMembers(
            String groupId,
            List<String> members,
            EMAError error) {
        return nativeUnMuteGroupMembers(groupId, members, error);
    }
    native EMAGroup nativeUnMuteGroupMembers(String groupId, List<String> members, EMAError error);

    public Map<String, Long> fetchGroupMutes(
            String groupId,
            int pageNum,
            int pageSize,
            EMAError error) {
        return nativeFetchGroupMutes(groupId, pageNum, pageSize, error);
    }
    native Map<String, Long> nativeFetchGroupMutes(String groupId, int pageNum, int pageSize, EMAError error);

    public List<String> fetchGroupBlackList(
            String groupId,
            int pageNum,
            int pageSize,
            EMAError error) {
        return nativeFetchGroupBlackList(groupId, pageNum, pageSize, error);
    }
    native List<String> nativeFetchGroupBlackList(String groupId, int pageNum, int pageSize, EMAError error);

    public EMAGroup updateGroupExtension(String groupId, String extension, EMAError error) {
        return nativeUpdateGroupExtension(groupId, extension, error);
    }
    native EMAGroup nativeUpdateGroupExtension(String groupId, String extension, EMAError error);

    // ============================= group_reform new add api end

    public String fetchGroupAnnouncement(String groupId, EMAError error){
        return nativeFetchGroupAnnouncement(groupId, error);
    }
    native String nativeFetchGroupAnnouncement(String groupId, EMAError error);

    public void updateGroupAnnouncement(String groupId, String announcement, EMAError error){
        nativeUpdateGroupAnnouncement(groupId, announcement, error);
    }
    native void nativeUpdateGroupAnnouncement(String groupId, String announcement, EMAError error);

    public EMAMucShareFile uploadGroupShareFile(String groupId, String filePath, EMACallback callback,
                         EMAError error){
        return nativeUploadGroupShareFile(groupId, filePath, callback, error);
    }
    native EMAMucShareFile nativeUploadGroupShareFile(String groupId, String filePath, EMACallback callback,
                                           EMAError error);

    public List<EMAMucShareFile> fetchGroupShareFiles(String groupId, int pageNum, int pageSize, EMAError error){
        return nativeFetchGroupShareFiles(groupId, pageNum, pageSize, error);
    }
    native List<EMAMucShareFile> nativeFetchGroupShareFiles(String groupId, int pageNum, int pageSize, EMAError error);

    public void deleteGroupShareFile(String groupId, String fileId, EMAError error){
        nativeDeleteGroupShareFile(groupId, fileId, error);
    }
    native void nativeDeleteGroupShareFile(String groupId, String fileId, EMAError error);


    public void downloadGroupShareFile(String groupId, String fileId, String savePath, EMACallback callback, EMAError error){
        nativeDownloadGroupShareFile(groupId, fileId, savePath, callback, error);
    }

    native void nativeDownloadGroupShareFile(String groupId, String fileId, String savePath, EMACallback callback, EMAError error);


    public EMAGroupManager() {
    }
    
    public EMAGroupManager(EMAGroupManager old) {
    	nativeInit(old);
    }
    native void nativeInit(EMAGroupManager groupManager);
}
