/*
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */
package internal.org.apache.http.entity.mime;

import internal.org.apache.http.entity.mime.content.ContentBody;

public class FormBodyPart {
	private final String name;
	  private final Header header;
	  private final ContentBody body;

	  public FormBodyPart(String name, ContentBody body)
	  {
	    if (name == null) {
	      throw new IllegalArgumentException("Name may not be null");
	    }
	    if (body == null) {
	      throw new IllegalArgumentException("Body may not be null");
	    }
	    this.name = name;
	    this.body = body;
	    this.header = new Header();

	    generateContentDisp(body);
	    generateContentType(body);
	    generateTransferEncoding(body);
	  }

	  public String getName() {
	    return this.name;
	  }

	  public ContentBody getBody() {
	    return this.body;
	  }

	  public Header getHeader() {
	    return this.header;
	  }

	  public void addField(String name, String value) {
	    if (name == null) {
	      throw new IllegalArgumentException("Field name may not be null");
	    }
	    this.header.addField(new MinimalField(name, value));
	  }

	  protected void generateContentDisp(ContentBody body) {
	    StringBuilder buffer = new StringBuilder();
	    buffer.append("form-data; name=\"");
	    buffer.append(getName());
	    buffer.append("\"");
	    if (body.getFilename() != null) {
	      buffer.append("; filename=\"");
	      buffer.append(body.getFilename());
	      buffer.append("\"");
	    }
	    addField("Content-Disposition", buffer.toString());
	  }

	  protected void generateContentType(ContentBody body) {
	    StringBuilder buffer = new StringBuilder();
	    buffer.append(body.getMimeType());
	    if (body.getCharset() != null) {
	      buffer.append("; charset=");
	      buffer.append(body.getCharset());
	    }
	    addField("Content-Type", buffer.toString());
	  }

	  protected void generateTransferEncoding(ContentBody body) {
	    addField("Content-Transfer-Encoding", body.getTransferEncoding());
	  }
}
