/************************************************************
 *  * EaseMob CONFIDENTIAL 
 * __________________ 
 * Copyright (C) 2013-2014 EaseMob Technologies. All rights reserved. 
 *  
 * NOTICE: All information contained herein is, and remains 
 * the property of EaseMob Technologies.
 * Dissemination of this information or reproduction of this material 
 * is strictly forbidden unless prior written permission is obtained
 * from EaseMob Technologies.
 */
package com.hyphenate.chat;

import com.hyphenate.EMCallBack;
import com.hyphenate.EMError;
import com.hyphenate.EMGroupChangeListener;
import com.hyphenate.EMValueCallBack;
import com.hyphenate.chat.adapter.EMAError;
import com.hyphenate.chat.adapter.EMAGroup;
import com.hyphenate.chat.adapter.EMAGroup.EMGroupLeaveReason;
import com.hyphenate.chat.adapter.EMAGroupManager;
import com.hyphenate.chat.adapter.EMAGroupManagerListener;
import com.hyphenate.chat.adapter.EMAGroupSetting;
import com.hyphenate.exceptions.HyphenateException;
import com.hyphenate.util.EMLog;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * \~chinese
 * 群组管理, 用于管理群组的创建，删除，成员管理等操作
 * 
 * \~english
 * group manager is used to manage all the groups related operations, create, delete, group and invite members, block memebers, etc..
 */
public class EMGroupManager {
	public enum EMGroupStyle{
        /** 
         * \~chinese 
         * 私有群组，创建完成后，只允许Owner邀请用户加入 
         *
         * \~english 
         * Private groups, after created, only allowing owner invite users to join
         */
        EMGroupStylePrivateOnlyOwnerInvite,   

        /** 
         * \~chinese 
         * 私有群组，创建完成后，只允许Owner和群成员邀请用户加入 
         *
         * \~english
         * Private groups, after created, only allowing owner and members invite users to join
         */
        EMGroupStylePrivateMemberCanInvite,

        /**
         * \~chinese 
         * 公开群组，创建完成后，只允许Owner邀请用户加入; 非群成员用户需发送入群申请，Owner同意后才能入组
         *
         * \~english
         * Public groups, after created, only allowing owner invite users to join; 
         * Non-group members into the group must send an application to the owner after the Owner agree
         */
        EMGroupStylePublicJoinNeedApproval,

        /**
         * \~chinese 
         * 公开群组，创建完成后，允许非群组成员加入，不需要Owner同意 
         *
         * \~english
         * Public groups, after created, to allow non-members to join the group, does not require owner agrees
         */
        EMGroupStylePublicOpenJoin
	}
	
	public static class EMGroupOptions {
        public int maxUsers = 200;
        public EMGroupStyle style = EMGroupStyle.EMGroupStylePrivateOnlyOwnerInvite;
		public boolean inviteNeedConfirm = false;
		public String extField;
	}
	
    EMAGroupManager emaObject;
    
	private static String TAG = "group";
	
	List<EMGroupChangeListener> groupChangeListeners;
	
	EMAGroupManagerListener listenerImpl = new EMAGroupManagerListener() {

	    @Override
	    public void onReceiveInviteFromGroup(String groupId, String inviter, String inviteMessage) {
	        synchronized (groupChangeListeners) {
		        try {
			        for (EMGroupChangeListener listener : groupChangeListeners) {
				        listener.onInvitationReceived(groupId, /*groupName*/ "", inviter, inviteMessage);
			        }
		        } catch (Exception e) {
			        e.printStackTrace();
		        }
	        }
	    }

	    /**
	     * \brief Callback user when the user accept to join the group.
	     *
	     * @param  invitee The group that invite the user.
	     * @return NA
	     */
	    @Override
	    public void onReceiveInviteAcceptionFromGroup(EMAGroup group, String invitee) {
	        synchronized (groupChangeListeners) {
		        try {
		            for (EMGroupChangeListener listener : groupChangeListeners) {
		                listener.onInvitationAccepted(group.groupId(), invitee, "");
		            }
		        } catch (Exception e) {
			        e.printStackTrace();
		        }
	        }
	    }

	    /**
	     * \brief Callback user when the user decline to join the group.
	     *
	     * @param  invitee The group that invite the user.
	     * @param  reason User's decline reason.
	     * @return NA
	     */
	    @Override
	    public void onReceiveInviteDeclineFromGroup(EMAGroup group, String invitee, String reason) {
	        synchronized (groupChangeListeners) {
		        try {
		            for (EMGroupChangeListener listener : groupChangeListeners) {
		                listener.onInvitationDeclined(group.groupId(), invitee, "");
		            }
		        } catch (Exception e) {
			        e.printStackTrace();
		        }
	        }
	    }

	    @Override
	    public void onAutoAcceptInvitationFromGroup(EMAGroup group, String inviter, String inviteMessage) {
	        synchronized (groupChangeListeners) {
		        try {
		            for (EMGroupChangeListener listener : groupChangeListeners) {
		                listener.onAutoAcceptInvitationFromGroup(group.groupId(), inviter, inviteMessage);
		            }
		        } catch (Exception e) {
			        e.printStackTrace();
		        }
	        }
	    }

	    @Override
	    public void onLeaveGroup(EMAGroup group, int reason) {
            EMClient.getInstance().chatManager().caches.remove(group.groupId());
	        synchronized (groupChangeListeners) {
		        try {
		            for (EMGroupChangeListener listener : groupChangeListeners) {
		                if (reason == EMGroupLeaveReason.BE_KICKED.ordinal()) {
		                    listener.onUserRemoved(group.groupId(), group.groupSubject());
		                } else {
		                    listener.onGroupDestroyed(group.groupId(), group.groupSubject());
		                }
		            }
		        } catch (Exception e) {
			        e.printStackTrace();
		        }
	        }
	    }

	    @Override
	    public void onReceiveJoinGroupApplication(EMAGroup group, String from, String message) {
            synchronized (groupChangeListeners) {
	            try {
	                for (EMGroupChangeListener listener : groupChangeListeners) {
	                    listener.onRequestToJoinReceived(group.groupId(), group.groupSubject(), from, message);
	                }
	            } catch (Exception e) {
		            e.printStackTrace();
	            }
            }
	    }

	    @Override
	    public void onReceiveAcceptionFromGroup(EMAGroup group) {
	        synchronized (groupChangeListeners) {
		        try {
		            for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onRequestToJoinAccepted(group.groupId(), group.groupSubject(), group.getOwner());
		            }
		        } catch (Exception e) {
			        e.printStackTrace();
		        }
	        }
	    }

	    @Override
	    public void onReceiveRejectionFromGroup(String groupId, String reason) {
	        EMGroup group = EMGroupManager.this.getGroup(groupId);
	        String groupName = group == null ? "" : group.groupSubject(); 
	        String decliner = group == null ? "" : group.getOwner();

	        synchronized (groupChangeListeners) {
		        try {
		            for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onRequestToJoinDeclined(groupId, groupName, decliner, reason);
		            }
		        } catch (Exception e) {
			        e.printStackTrace();
		        }
	        }
	    }

		@Override
		public void onUpdateMyGroupList(List<EMAGroup> groups) {
			
		}

		@Override
		public void onAddMutesFromGroup(EMAGroup group, final List<String> muteMembers, long muteExpire) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onMuteListAdded(group.groupId(), muteMembers, muteExpire);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onRemoveMutesFromGroup(EMAGroup group, final List<String> banPostList) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onMuteListRemoved(group.groupId(), banPostList);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onAddAdminFromGroup(EMAGroup group, String admin) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onAdminAdded(group.groupId(), admin);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onRemoveAdminFromGroup(EMAGroup group, String admin) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onAdminRemoved(group.groupId(), admin);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}

		@Override
		public void onAssignOwnerFromGroup(EMAGroup group, String newOwner, String oldOwner) {
			synchronized (groupChangeListeners) {
				try {
					for (EMGroupChangeListener listener : groupChangeListeners) {
						listener.onOwnerChanged(group.groupId(), newOwner, oldOwner);
					}
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}
		
	    @Override
	    public void onMemberJoined(EMAGroup group, String member) {
	        synchronized (groupChangeListeners) {
                try {
                    for (EMGroupChangeListener listener : groupChangeListeners) {
                        listener.onMemberJoined(group.groupId(), member);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
	    }
	    
	    @Override
	    public void onMemberExited(EMAGroup group, String member) {
	        synchronized (groupChangeListeners) {
                try {
                    for (EMGroupChangeListener listener : groupChangeListeners) {
                        listener.onMemberExited(group.groupId(), member);
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
	    }
	};
	
	EMClient mClient;
    EMGroupManager(EMClient client, EMAGroupManager groupManager) {
    	emaObject = groupManager;
    	mClient = client;
		groupChangeListeners = Collections.synchronizedList(new ArrayList<EMGroupChangeListener>());
		emaObject.addListener(listenerImpl);
    }

	/**
	 * \~chinese
	 * 获取当前(内存)用户的所有群组
	 * @return 群组列表
	 * 
	 * \~english
	 * get all groups from cache
	 * @return group List
	 */
	public List<EMGroup> getAllGroups() {
		EMAError error = new EMAError();
		List<EMAGroup> groups = emaObject.allMyGroups(error);
		List<EMGroup> result = new ArrayList<EMGroup>();
		for (EMAGroup group : groups) {
			result.add(new EMGroup(group));
		}
		return Collections.unmodifiableList(result);
	}

	/**
	 * \~chinese
	 * 根据群组ID，获得群组对象
	 * @param groupId
	 *            群组的ID
	 * @return 群组对象，如果群组不存在，返回null
	 * 
	 * \~english
	 * get local group from cache by group id
	 * @param groupId
	 * @return instance of EMGroup, return null if group is not exist
	 */
	public EMGroup getGroup(String groupId) {
		EMAError error = new EMAError();
		List<EMAGroup> groups = emaObject.allMyGroups(error);
		for (EMAGroup group : groups) {
			if (group.groupId().equals(groupId)) {
				return new EMGroup(group);
			}
		}
		return null;
	}
	
	/**
	 * \~chinese
	 * 在IM服务器创建一个群组
	 * @param groupName     群组的名字
	 * @param allMembers    群成员数组,不需要群主id
	 * @param reason        邀请群成员加入的信息
	 * @param option        群的设置
	 * 
	 * \~english
	 * create a group on IM server
	 * @param groupName     group name
	 * @param allMembers    array of all members, no need include the owner of this group
	 * @param reason        message used to invite the group members
	 * @param option        options for this group
	 */
	public EMGroup createGroup(String groupName, String desc, String[] allMembers, String reason, EMGroupOptions option) throws HyphenateException{
		int style = EMAGroupSetting.EMAGroupStyle_PRIVATE_OWNER_INVITE;

		switch(option.style){
		case EMGroupStylePrivateOnlyOwnerInvite:
			style = EMAGroupSetting.EMAGroupStyle_PRIVATE_OWNER_INVITE;
			break;
			
		case EMGroupStylePrivateMemberCanInvite:
			style = EMAGroupSetting.EMAGroupStyle_PRIVATE_MEMBER_INVITE;
			break;
			
		case EMGroupStylePublicJoinNeedApproval:
			style = EMAGroupSetting.EMAGroupStyle_PUBLIC_JOIN_APPROVAL;
			break;
			
		case EMGroupStylePublicOpenJoin:
			style = EMAGroupSetting.EMAGroupStyle_PUBLIC_JOIN_OPEN;
			break;
		}
		
		return createGroup(style,groupName,desc,allMembers,option.maxUsers,reason, option.inviteNeedConfirm, option.extField);
	}
	
      /**
     * \~chinese
     * 在IM服务器创建一个群组
     * @param groupName     群组的名字
     * @param allMembers    群成员数组,不需要群主id
     * @param reason        邀请群成员加入的信息
     * @param option        群的设置
     * 
     * \~english
     * create a group on IM server
     * @param groupName     group name
     * @param allMembers    array of all members, no need include the owner of this group
     * @param reason        message used to invite the group members
     * @param option        options for this group
     */
    public void asyncCreateGroup(final String groupName, final String desc, final String[] allMembers, final String reason, final EMGroupOptions option, final EMValueCallBack<EMGroup> callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    EMGroup group = createGroup(groupName, desc, allMembers, reason, option);
                    callback.onSuccess(group);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

    /**
     * \~chinese
     * 同步加载所有群组
     * 
     * \~english
	 * load all local groups synchronously
	 */
	public synchronized void loadAllGroups() {
	    emaObject.loadAllMyGroupsFromDB();
	}
	
	/**
	 * \~chinese
	 * 解散群组 （只有群组的创建者才能调用此函数来删除）
	 * @param groupId
	 * @throws HyphenateException
	 * 
	 * \~english
	 * dismiss the group and only the group owner can do it.
	 * @param groupId
	 * @throws HyphenateException
	 */
	public void destroyGroup(final String groupId) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.destroyGroup(groupId, error);
		EMClient.getInstance().chatManager().caches.remove(groupId);
		handleError(error);
	}

    /**
     * \~chinese
     * 解散群组 （只有群组的创建者才能调用此函数来删除）
     * @param groupId
     * @throws HyphenateException
     * 
     * \~english
     * dismiss the group and only the group owner can do it.
     * @param groupId
     * @throws HyphenateException
     */
    public void asyncDestroyGroup(final String groupId, final EMCallBack callback){
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    destroyGroup(groupId);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }
 
	/**
	 * \~chinese
	 * 向群组中添加新的成员
	 * @param groupId
	 *            群组的ID
	 * @param newmembers
	 *            待添加的新成员用户名
	 * @throws HyphenateException
	 * 
	 * \~english
	 * add users to the group and only the group owner can do it.
	 * @param groupId
	 * @param newmembers
	 *            user id to be added
	 * @throws HyphenateException
	 */
	public void addUsersToGroup(String groupId, String[] newmembers) throws HyphenateException {
		EMAError error = new EMAError();
		List<String> newmembers2 = new ArrayList<String>();
		Collections.addAll(newmembers2, newmembers);
		emaObject.addGroupMembers(groupId, newmembers2, "welcome", error);
		handleError(error);
	}
	
	/**
     * \~chinese
     * 向群组中添加新的成员
     * @param groupId
     *            群组的ID
     * @param newmembers
     *            待添加的新成员用户名
     * @throws HyphenateException
     * 
     * \~english
     * add users to the group and only the group owner can do it.
     * @param groupId
     * @param newmembers
     *            user id to be added
     * @throws HyphenateException
     */
    public void asyncAddUsersToGroup(final String groupId, final String[] newmembers, final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    addUsersToGroup(groupId, newmembers);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }
    
	/**
	 * \~chinese
	 * 从群组中删除成员
	 * @param groupId
	 *            群组的ID
	 * @param username
	 *            待删除成员的用户名
	 * @throws HyphenateException
	 * 
	 * \~english
	 * remove the member from the group and only the group owner can do it.
     * @param groupId
     * @param username   user id to be removed
	 * @throws HyphenateException
	 */
	public void removeUserFromGroup(final String groupId, final String username) throws HyphenateException {
		List<String> members = new ArrayList<String>();
		EMAError error = new EMAError();
		members.add(username);
		emaObject.removeGroupMembers(groupId, members, error);
		handleError(error);
		emaObject.fetchGroupSpecification(groupId, error, true);
		handleError(error);
	}

    /**
     * \~chinese
     * 从群组中删除成员
     * @param groupId
     *            群组的ID
     * @param username
     *            待删除成员的用户名
     * @throws HyphenateException
     * 
     * \~english
     * remove the member from the group and only the group owner can do it.
     * @param groupId
     * @param username   user id to be removed
     * @throws HyphenateException
     */
    public void asyncRemoveUserFromGroup(final String groupId, final String username, final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    removeUserFromGroup(groupId, username);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }
    
	/**
	 * \~chinese
	 * 当前登录用户退出群组
	 * @param groupId
	 * @throws HyphenateException
	 * 
	 * \~english
	 * the member exit the group
	 * @param groupId
	 * @throws HyphenateException
	 */
	public void leaveGroup(String groupId) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.leaveGroup(groupId, error);
		EMClient.getInstance().chatManager().caches.remove(groupId);
		handleError(error);
	}
	
	/**
     * \~chinese
     * 当前登录用户退出群组
     * @param groupId
     * @throws HyphenateException
     * 
     * \~english
     * the member exit the group
     * @param groupId
     * @throws HyphenateException
     */
    public void asyncLeaveGroup(final String groupId, final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    leaveGroup(groupId);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }
    
	/**
	 * \~chinese
	 * 从服务器获取群组的详细信息
	 * @param groupId
	 * @return EMGroup 更新的群组对象
	 * 
	 * \~english
	 * get information of group from server.
	 * @param groupId
	 * @return  instance of EMGroup
	 */
	public EMGroup getGroupFromServer(String groupId) throws HyphenateException {
	    if(groupId == null || groupId.isEmpty())
	        throw new HyphenateException(EMError.GROUP_INVALID_ID, "group id is null or empty");
		EMAError error = new EMAError();
		EMAGroup group = emaObject.fetchGroupSpecification(groupId, error, false);
		handleError(error);
		return new EMGroup(group);
	}

	/**
	 * \~chinese
	 * 从服务器获取群组的详细信息。
	 * fetchMembers为true的时候，获取群成员，如需获取更多成员，使用 {@link #fetchGroupMembers(String, String, int)}。
	 * 只有群成员才能获取群组的成员列表。
	 * @param fetchMembers 是否获取成员列表，默认取200人的成员列表
	 * @return EMGroup 更新的群组对象
	 *
	 * \~english
	 * fetch group specification
	 * @param fetchMembers whether need fetch group members, if need fetch members, default action fetch 200 members.
	 * @return instance of EMGroup
	 */
	public EMGroup getGroupFromServer(String groupId, boolean fetchMembers) throws HyphenateException {
		if(groupId == null || groupId.isEmpty())
			throw new HyphenateException(EMError.GROUP_INVALID_ID, "group id is null or empty");
		EMAError error = new EMAError();
		EMAGroup group = emaObject.fetchGroupSpecification(groupId, error, fetchMembers);
		handleError(error);
		return new EMGroup(group);
	}
	
	/**
     * \~chinese
     * 从服务器获取群组的详细信息和最新的成员列表
     * 
     * @param groupId
     * @return EMGroup 更新的群组对象
     * 
     * \~english
     * get information of group from server.
     * @param groupId
     * @return  instance of EMGroup
     */
    public void asyncGetGroupFromServer(final String groupId, final EMValueCallBack<EMGroup> callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    EMGroup group = getGroupFromServer(groupId);
                    callback.onSuccess(group);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

	/**
	 * \~chinese
	 * 从服务器端获取当前用户的所有群组 （此操作只返回群组列表，并不获取群组的所有成员信息，如果要更新某个群组包括成员的全部信息，需要再调用
	 * {@link #getGroupFromServer(String groupId)}
	 * @return 群组列表
	 * @throws HyphenateException
	 * 
	 * \~english
	 * this api will get groups from remote server and update local groups, this method only return the list of group,
	 * not include member list in group, use {@link #getGroupFromServer(String groupId)} if you want it.
	 * @return array of group list 
	 * @throws HyphenateException
	 */
	synchronized List<EMGroup> getGroupsFromServer() throws HyphenateException {
		EMAError error = new EMAError();
		List<EMAGroup> groups = emaObject.fetchAllMyGroups(error);
		handleError(error);
		List<EMGroup> result = new ArrayList<EMGroup>();
		for (EMAGroup group : groups) {
			result.add(new EMGroup(group));
		}
		return result;
	}
	
	/**
     * \~chinese
     * 从服务器端获取当前用户的所有群组 （此操作只返回群组列表，并不获取群组的所有成员信息，如果要更新某个群组包括成员的全部信息，需要再调用
     * {@link #getGroupFromServer(String groupId)}
     * @return 群组列表
     * @throws HyphenateException
     * 
     * \~english
     * this api will get groups from remote server and update local groups, this method only return the list of group,
     * not include member list in group, use {@link #getGroupFromServer(String groupId)} if you want it.
     * @return array of group list 
     * @throws HyphenateException
     */
    void asyncGetGroupsFromServer(final EMValueCallBack<List<EMGroup>> callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    List<EMGroup> groups = getGroupsFromServer();
                    callback.onSuccess(groups);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }
	
	/**
	 * \~chiese
     * 从服务器端获取当前用户的所有群组 （此操作只返回群组列表，并不获取群组的所有成员信息，如果要更新某个群组包括成员的全部信息，需要再调用
     * {@link #getGroupFromServer(String groupId)}
     * @return 群组列表
     * @throws HyphenateException
     * 
     * \~english
     * fetch all joined groups from server, only return the group list, not include member list in group, use
     * {@link #getGroupFromServer(String groupId)} if you want it
     * @return group list
     * @throws HyphenateException
     */
	public synchronized List<EMGroup> getJoinedGroupsFromServer() throws HyphenateException{
	    return getGroupsFromServer();
	}
	
	/**
     * \~chiese
     * 从服务器端获取当前用户的所有群组 （此操作只返回群组列表，并不获取群组的所有成员信息，如果要更新某个群组包括成员的全部信息，需要再调用
     * {@link #getGroupFromServer(String groupId)}
     * @return 群组列表
     * @throws HyphenateException
     * 
     * \~english
     * fetch all joined groups from server, only return the group list, not include member list in group, use
     * {@link #getGroupFromServer(String groupId)} if you want it
     * @return group list
     * @throws HyphenateException
     */
    public void asyncGetJoinedGroupsFromServer(final EMValueCallBack<List<EMGroup>> callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    List<EMGroup> groups = getJoinedGroupsFromServer();
                    callback.onSuccess(groups);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }
    
	/**
	 * \~chinese
	 * 从服务器获取公开群组
	 * @param pageSize 
	 *             获取多少条
	 * @param cursor 
	 *             用于分批获取数据的cursor,首次获取数据时传null即可
	 * @return 
	 *             包含用于下次获取数据的cursor以及群组list的对象
	 * @throws HyphenateException
	 * 
	 * \~english
	 * get all public groups from server
	 * @param pageSize  the number of groups
	 * @param cursor    used to get group page by page, use null if first call
	 * @return cursor used to get next page and the group list
	 * @throws HyphenateException
	 */
	public EMCursorResult<EMGroupInfo> getPublicGroupsFromServer(int pageSize, String cursor) throws HyphenateException {
		EMAError error = new EMAError();
		EMCursorResult<EMGroupInfo> result = emaObject.fetchPublicGroupsWithCursor(cursor, pageSize, error);
		handleError(error);
		return result;
	}

	 /**
     * \~chinese
     * 从服务器获取公开群组
     * @param pageSize 
     *             获取多少条
     * @param cursor 
     *             用于分批获取数据的cursor,首次获取数据时传null即可
     * @return 
     *             包含用于下次获取数据的cursor以及群组list的对象
     * @throws HyphenateException
     * 
     * \~english
     * get all public groups from server
     * @param pageSize  the number of groups
     * @param cursor    used to get group page by page, use null if first call
     * @return cursor used to get next page and the group list
     * @throws HyphenateException
     */
    public void asyncGetPublicGroupsFromServer(final int pageSize, final String cursor, final EMValueCallBack<EMCursorResult<EMGroupInfo>> callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    EMCursorResult<EMGroupInfo> result = getPublicGroupsFromServer(pageSize, cursor);
                    callback.onSuccess(result);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }
    
	/**
	 * \~chinese
	 * 当前登录用户加入公开群(如果是自由加入的公开群，直接进入群组；需要验证的公开群需要等待群主同意)
	 * @param groupId
	 * 
	 * \~english
	 * join the group by the group id
	 * @param groupId
	 */
	public void joinGroup(String groupId) throws HyphenateException {
		EMAError error = new EMAError();
		EMAGroup _group = emaObject.fetchGroupSpecification(groupId, error, false);
		handleError(error);
		if (_group.groupSetting() == null) {
			throw new HyphenateException(); 
		}
		if (_group.groupSetting().style() == EMAGroupSetting.EMAGroupStyle_PUBLIC_JOIN_OPEN) {
			emaObject.joinPublicGroup(groupId, error);
	        handleError(error);
			return;
		}
		if (_group.groupSetting().style() == EMAGroupSetting.EMAGroupStyle_PUBLIC_JOIN_APPROVAL) {
			emaObject.applyJoinPublicGroup(groupId, mClient.getCurrentUser(), "hello", error);
	        handleError(error);
			return;
		}
	}
	
	/**
     * \~chinese
     * 当前登录用户加入公开群(如果是自由加入的公开群，直接进入群组；需要验证的公开群需要等待群主同意)
     * @param groupId
     * 
     * \~english
     * join the group by the group id
     * @param groupId
     */
    public void asyncJoinGroup(final String groupId, final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    joinGroup(groupId);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }
    
	/**
	 * \~chinese
	 * 改变群组的名称(群主才能调用此方法)	 
	 * @param groupId 需要改变名称的群组的id
	 * @param changedGroupName 改变后的群组名称
	 * @throws HyphenateException 
	 * 
	 * \~english
	 * change the group name and only group owner can do it.
	 * @param groupId group id to be change name
	 * @param changedGroupName new name
	 * @throws HyphenateException 
	 */
	public void changeGroupName(String groupId, String changedGroupName) throws HyphenateException{
		EMAError error = new EMAError();
		emaObject.changeGroupSubject(groupId, changedGroupName, error);
		handleError(error);
	}
	
	 /**
     * \~chinese
     * 改变群组的名称(群主才能调用此方法)    
     * @param groupId 需要改变名称的群组的id
     * @param changedGroupName 改变后的群组名称
     * @throws HyphenateException 
     * 
     * \~english
     * change the group name and only group owner can do it.
     * @param groupId group id to be change name
     * @param changedGroupName new name
     * @throws HyphenateException 
     */
    public void asyncChangeGroupName(final String groupId, final String changedGroupName, final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    changeGroupName(groupId, changedGroupName);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

	/**
	 * \~chinese
	 * 修改群描述(群主才能调用此方法)
	 * @param groupId 群id
	 * @param changedDescription 改变后的群描述
	 * @throws HyphenateException
	 *
	 * \~english
	 * change the group description and only group owner can do it.
	 * @param groupId group id
	 * @param changedDescription changed group description
	 * @throws HyphenateException
	 */
	public void changeGroupDescription(String groupId, String changedDescription)
			throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.changeGroupDescription(groupId, changedDescription, error);
		handleError(error);
	}

	/**
	 * \~chinese
	 * 异步修改群描述(群主才能调用此方法)
	 * @param groupId 群id
	 * @param changedDescription 改变后的群描述
	 * @param callBack
	 * @throws HyphenateException
	 *
	 * \~english
	 * async change the group description and only group owner can do it.
	 * @param groupId group id
	 * @param changedDescription changed group description
	 * @param callBack
	 * @throws HyphenateException
	 */
	public void asyncChangeGroupDescription(final String groupId, final String changedDescription, final EMCallBack callBack){
		EMClient.getInstance().execute(new Runnable() {
			@Override public void run() {
				try {
					changeGroupDescription(groupId, changedDescription);
					callBack.onSuccess();
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}


	/**
	 * \~chinese
	 * 接受加入群的邀请
	 * @param groupId
	 * @param inviter
	 * @return 用户已经同意邀请的群组对象
	 * 
	 * \~english
     * accept group's invitation.
     *
     * @param  groupId Group's ID.
     * @param  inviter Inviter.
     * @return The group user has accepted.
     */
    public EMGroup acceptInvitation(String groupId, String inviter) throws HyphenateException {
        EMAError error = new EMAError();
        EMAGroup group = emaObject.acceptInvitationFromGroup(groupId, inviter == null ? "" : inviter, error);
        handleError(error);
        return new EMGroup(group);
    }
    
    /**
     * \~chinese
     * 接受加入群的邀请
     * @param groupId
     * @param inviter
     * @return 用户已经同意邀请的群组对象
     * 
     * \~english
     * accept group's invitation.
     *
     * @param groupId Group's ID.
     * @param inviter Inviter.
     * @return The group user has accepted.
     */
    public void asyncAcceptInvitation(final String groupId, final String inviter, final EMValueCallBack<EMGroup> callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    EMGroup group = acceptInvitation(groupId, inviter);
                    callback.onSuccess(group);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }
    

    /**
	 * \~chinese
	 * 拒绝加入群的邀请
	 * @param groupId
	 * @param inviter
	 * @param reason, 拒绝的原因
	 * @return 用户已经同意邀请的群组对象
	 * 
	 * \~english
     * decline group's invitation.
     *
     * @param groupId Group's ID.
     * @param inviter Inviter.
     * @param reason,  message for decline
     * @return The group user has accepted.
     */
    public void declineInvitation(
            String groupId,
            String inviter,
            String reason) throws HyphenateException {
        EMAError error = new EMAError();
        emaObject.declineInvitationFromGroup(groupId, inviter == null ? "" : inviter, reason == null ? "" : reason, error);
        handleError(error);
    }
    
    
    /**
     * \~chinese
     * 拒绝加入群的邀请
     * @param groupId
     * @param inviter
     * @param reason, 拒绝的原因
     * @return 用户已经同意邀请的群组对象
     * 
     * \~english
     * decline group's invitation.
     *
     * @param groupId Group's ID.
     * @param inviter Inviter.
     * @param reason,  message for decline
     * @return The group user has accepted.
     */
    public void asyncDeclineInvitation(
            final String groupId,
            final String inviter,
            final String reason, final EMCallBack callback) {
        
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    declineInvitation(groupId, inviter, reason);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }
    
    
	/**
	 * \~chinese
	 * 同意加群申请
	 * @param username 申请人username
	 * @param groupId  要申请加入的群id
	 * @throws HyphenateException
	 * 
	 * \~english
	 * accept the application of the user to join this group
	 * @param username application user
	 * @param groupId  group id for application
	 * @throws HyphenateException
	 */
	public void acceptApplication(String username, String groupId) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.acceptJoinGroupApplication(groupId, username, error);
		handleError(error);
	}
	
	 /**
     * \~chinese
     * 同意加群申请
     * @param username 申请人username
     * @param groupId  要申请加入的群id
     * @throws HyphenateException
     * 
     * \~english
     * accept the application of the user to join this group
     * @param username application user
     * @param groupId  group id for application
     * @throws HyphenateException
     */
    public void asyncAcceptApplication(final String username, final String groupId, final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    acceptApplication(username, groupId);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }
    
	/**
	 * \~chinese
	 * 拒绝加群申请
	 * @param username 申请人username
	 * @param groupId  要申请加入的群id
	 * @param reason   拒绝的理由
	 * @throws HyphenateException
	 * 
	 * \~english
	 * accept the application of the user to join this group
	 * @param username application user
	 * @param groupId  group id for application
	 * @param reason   message for decline
	 * @throws HyphenateException
	 */
	public void declineApplication(String username, String groupId, String reason) throws HyphenateException{
		EMAError error = new EMAError();
		emaObject.declineJoinGroupApplication(groupId, username, reason, error);
		handleError(error);
	}


    /**
     * \~chinese
     * 拒绝加群申请
     * @param username 申请人username
     * @param groupId  要申请加入的群id
     * @param reason   拒绝的理由
     * @throws HyphenateException
     * 
     * \~english
     * accept the application of the user to join this group
     * @param username application user
     * @param groupId  group id for application
     * @param reason   message for decline
     * @throws HyphenateException
     */
    public void asyncDeclineApplication(final String username, final String groupId, final String reason, final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    declineApplication(username, groupId, reason);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

	/**
	 * \~chinese
	 * 群成员邀请用户加入群组 （如果群组设置成可以成员邀请，现在的群组成员可以邀请其他用户加入）
	 * @param groupId
	 *            群组Id
	 * @param beInvitedUsernames
	 *            被邀请人username的数组
	 * @param reason
	 * @throws HyphenateException
	 * 
	 * \~english
	 * invite other users to join the group if the group has been set to allow member to invite
	 * @param groupId
	 * @param beInvitedUsernames   array of users to be invited
	 * @param reason               message for invitation
	 * @throws HyphenateException
	 */
	public void inviteUser(String groupId, String[] beInvitedUsernames, String reason) throws HyphenateException {
		EMAError error = new EMAError();
		List<String> members = new ArrayList<String>();
		Collections.addAll(members, beInvitedUsernames);
		emaObject.addGroupMembers(groupId, members, reason, error);
		handleError(error);
	}

	/**
     * \~chinese
     * 群成员邀请用户加入群组 （如果群组设置成可以成员邀请，现在的群组成员可以邀请其他用户加入）
     * @param groupId
     *            群组Id
     * @param beInvitedUsernames
     *            被邀请人username的数组
     * @param reason
     * @throws HyphenateException
     * 
     * \~english
     * invite other users to join the group if the group has been set to allow member to invite
     * @param groupId
     * @param beInvitedUsernames   array of users to be invited
     * @param reason               message for invitation
     * @throws HyphenateException
     */
    public void asyncInviteUser(final String groupId, final String[] beInvitedUsernames, final String reason, final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    inviteUser(groupId, beInvitedUsernames, reason);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }
    
	/**
	 * \~chinese
	 * 申请加入某个群（用于加入需要验证的公开群）
	 * @param groupId
	 * @param reason    申请加入的原因
	 * 
	 * \~english
	 * apply to join the group
	 * @param groupId
	 * @param reason    message for join application
	 * @throws HyphenateException
	 */
	public void applyJoinToGroup(String groupId, String reason) throws HyphenateException {
		String userName = mClient.getCurrentUser();
		EMAError error = new EMAError();
		emaObject.applyJoinPublicGroup(groupId, userName, reason, error);
		handleError(error);
	}
	
	/**
     * \~chinese
     * 申请加入某个群（用于加入需要验证的公开群）
     * @param groupId
     * @param reason    申请加入的原因
     * 
     * \~english
     * apply to join the group
     * @param groupId
     * @param reason    message for join application
     * @throws HyphenateException
     */
    public void asyncApplyJoinToGroup(final String groupId, final String reason, final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    applyJoinToGroup(groupId, reason);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }
	
	/**
	 * \~chinese
	 * 屏蔽群消息，（还是群里面的成员，但不再接收群消息
	 * @param groupId
	 * @throws HyphenateException
	 *
	 *\~english
	 * set to disable receiving the group messages
	 * @param groupId
	 * @throws HyphenateException
	 */
	public void blockGroupMessage(String groupId) throws HyphenateException {
		EMAError error = new EMAError();
		emaObject.blockGroupMessage(groupId, error);
		handleError(error);
	}
	

    /**
     * \~chinese
     * 屏蔽群消息，（还是群里面的成员，但不再接收群消息
     * @param groupId
     * @throws HyphenateException
     *
     *\~english
     * set to disable receiving the group messages
     * @param groupId
     * @throws HyphenateException
     */
    public void asyncBlockGroupMessage(final String groupId, final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    blockGroupMessage(groupId);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }
    
	/**
	 * \~chinese
	 * 取消屏蔽群消息
	 * @param groupId
	 * @throws HyphenateException
	 *
	 * \~english
	 * set to enable to receive the group message again
	 * @param groupId
	 * @throws HyphenateException
	 */
	public void unblockGroupMessage(String groupId) throws HyphenateException {
		EMLog.d(TAG, "try to unblock group msg:" + groupId);
		EMAError error = new EMAError();
		emaObject.unblockGroupMessage(groupId, error);
		handleError(error);
	}
	
    /**
     * \~chinese
     * 取消屏蔽群消息
     * @param groupId
     * @throws HyphenateException
     *
     * \~english
     * set to enable to receive the group message again
     * @param groupId
     * @throws HyphenateException
     */
    public void asyncUnblockGroupMessage(final String groupId, final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    unblockGroupMessage(groupId);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }
    
	/**
	 * \~chinese
	 * 将用户加到群组的黑名单，被加入黑名单的用户无法加入群，无法收发此群的消息
	 * （只有群主才能设置群的黑名单）。
	 * 3.3.0群组改造和之前版本的区别：被加入到黑名单的用户会被先移出群，再加入到黑名单中，之前版本没有移出群的动作。
	 * @param groupId 群组的id
	 * @param username 待屏蔽的用户名
	 * @exception HyphenateException 出错会抛出
	 * 
	 *\~english
	 * set to block member who will be not allowed to receive any group messages
	 * and only the group owner can do it.
	 * @param groupId
g	 * @param username user to be blocked
	 * @exception HyphenateException
	 */
	public void blockUser(String groupId, String username) throws HyphenateException {
		EMLog.d(TAG, "block user for groupid:" + groupId + " username:" + username);
		EMAError error = new EMAError();
		String reason = "";
		List<String> members = new ArrayList<String>();
		members.add(username);
		emaObject.blockGroupMembers(groupId, members, error, reason);
		handleError(error);
	}
	

    /**
     * \~chinese
     * 将用户加到群组的黑名单，被加入黑名单的用户无法加入群，无法收发此群的消息
     * （只有群主才能设置群的黑名单）
     * @param groupId 群组的id
     * @param username 待屏蔽的用户名
     * @exception HyphenateException 出错会抛出
     * 
     *\~english
     * set to block member who will be not allowed to receive any group messages
     * and only the group owner can do it.
     * @param groupId
     * @param username user to be blocked
     * @exception HyphenateException
     */
    public void asyncBlockUser(final String groupId, final String username, final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    blockUser(groupId, username);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }
    
	/**
	 * \~chinese
	 * 将用户从群组的黑名单移除
	 * @param groupId 群组的id
	 * @param username 待解除屏蔽的 用户名
	 * 
	 * \~english
	 * remove the blocked user from the group balcklist
	 * @param groupId
	 * @param username
	 */
	public void unblockUser(String groupId, String username) throws HyphenateException{
		EMLog.d(TAG, "unblock user groupid:" + groupId + " username:" + username);
		EMAError error = new EMAError();
		List<String> members = new ArrayList<String>();
		members.add(username);
		emaObject.unblockGroupMembers(groupId, members, error);
		handleError(error);
	}
		

    /**
     * \~chinese
     * 将用户从群组的黑名单移除
     * @param groupId 群组的id
     * @param username 待解除屏蔽的 用户名
     * 
     * \~english
     * remove the blocked user from the group balcklist
     * @param groupId
     * @param username
     */
    public void asyncUnblockUser(final String groupId, final String username, final EMCallBack callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    unblockUser(groupId, username);
                    callback.onSuccess();
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }


	/**
	 * \~chinese
	 * 获取群组的黑名单用户列表, 默认最多取200个成员
	 * （只有群主才能调用此函数）
	 * return List<String>
	 * @throws HyphenateException 获取失败
	 *
	 * \~english
	 * fetch black list from server.
	 * Need owner's authority to access, max returned count is 200
	 * @return List<String>
	 * @throws HyphenateException
	 */
	public List<String> getBlockedUsers(String groupId) throws HyphenateException{
		return getBlockedUsers(groupId, 0, 200);
	}

	/**
	 * \~chinese
	 * 获取群组的黑名单用户列表
	 * （只有群主才能调用此函数）
	 * return List<String> 
	 * @throws HyphenateException 获取失败
	 * 
	 * \~english
	 * Get black list of group from server,
	 * need owner's authority to access.
	 * @return List<String> 
	 * @throws HyphenateException
	 */
	public List<String> getBlockedUsers(String groupId, int pageIndex, int pageSize) throws HyphenateException{
		EMLog.d(TAG, "get blocked users for group:" + groupId);
		EMAError error = new EMAError();
		List<String> members = emaObject.fetchGroupBlackList(groupId, pageIndex, pageSize, error);
		handleError(error);
		return members;
	}

	/**
	 * \~chinese
	 * 获取群组的黑名单用户列表, 默认最多取200个成员
	 * （只有群主才能调用此函数）
	 * return List<String>
	 * @throws HyphenateException 获取失败
	 *
	 * \~english
	 * Get black list from server,
	 * need owner's authority to access., max fetched block users count is 200
	 * @return List<String>
	 * @throws HyphenateException
	 */
	public void asyncGetBlockedUsers(final String groupId, final EMValueCallBack<List<String>> callback) {
		asyncGetBlockedUsers(groupId, 0, 200, callback);
	}

    /**
     * \~chinese
     * 获取群组的黑名单用户列表
     * （只有群主才能调用此函数）
     * return List<String> 
     * @throws HyphenateException 获取失败
     * 
     * \~english
     * Get black list of group, need owner's authority to access.
     * @return List<String> 
     * @throws HyphenateException
     */
    public void asyncGetBlockedUsers(final String groupId, final int pageIndex, final int pageSize, final EMValueCallBack<List<String>> callback) {
        EMClient.getInstance().execute(new Runnable() {
            
            @Override
            public void run() {
                try {
                    List<String> users = getBlockedUsers(groupId, pageIndex, pageSize);
                    callback.onSuccess(users);
                } catch (HyphenateException e) {
                    callback.onError(e.getErrorCode(), e.getDescription());
                }
            }
        });
    }

	/**
	 * \~chinese
	 * 注册群变动事件监听
	 * @param listener
	 * @see EMGroupChangeListener
	 * 
	 * \~english
	 * register the group change listener
	 * @param listener
	 * @see EMGroupChangeListener
	 */
	public void addGroupChangeListener(EMGroupChangeListener listener) {
		EMLog.d(TAG, "add group change listener:" + listener.getClass().getName());
		if (!groupChangeListeners.contains(listener)) {
		    groupChangeListeners.add(listener);
		}
	}

	/**
	 * \~chinese
	 * 移除群组变化监听器
	 * @param listener
	 * 
	 * \~english
	 * remove group change listener
	 * @param listener
	 */
	public void removeGroupChangeListener(EMGroupChangeListener listener) {
		EMLog.d(TAG, "remove group change listener:" + listener.getClass().getName());
		groupChangeListeners.remove(listener);
	}
	
	private EMGroup createGroup(int style, String groupName, String desc, String[] allMembers, int maxUsers, String reason, boolean inviteNeedConfirm, String extension) throws HyphenateException {
		EMAGroupSetting setting = new EMAGroupSetting(style, maxUsers, inviteNeedConfirm, extension);
		List<String> newMembers = new ArrayList<String>();
		Collections.addAll(newMembers, allMembers);
		EMAError error = new EMAError();
		EMAGroup group = emaObject.createGroup(groupName, desc, reason, setting, newMembers, inviteNeedConfirm, error);
	    handleError(error);
		return new EMGroup(group);
	}

	// ============================= group_reform new add api begin
	/**
	 * \~chinese
	 * 获取群组成员列表
	 * @param groupId
	 * @param cursor
	 * @param pageSize
	 * @return
	 * @throws HyphenateException
	 *
	 * \~english
	 * Get group's member list for certain range.
	 * @param groupId
	 * @param cursor
	 * @param pageSize
	 * @return
	 * @throws HyphenateException
     */
	public EMCursorResult<String> fetchGroupMembers(String groupId, String cursor, int pageSize) throws HyphenateException {
		EMAError error = new EMAError();
		EMCursorResult<String> result = emaObject.fetchGroupMembers(groupId, cursor, pageSize, error);
		handleError(error);
		return result;
	}

	/**
	 * \~chinese
	 * 获取群组成员列表, 异步方法
	 * @param groupId
	 * @param cursor
	 * @param pageSize
	 * @return
	 * @throws HyphenateException
	 *
	 * \~english
	 * Get group's member list, asynchronous method
	 * @param groupId
	 * @param cursor
	 * @param pageSize
	 * @return
	 * @throws HyphenateException
	 */
	public void asyncFetchGroupMembers(final String groupId, final String cursor, final int pageSize, final EMValueCallBack<EMCursorResult<String>> callback) {
		EMClient.getInstance().execute(new Runnable() {

           @Override
           public void run() {
               try {
	               EMCursorResult<String> result = fetchGroupMembers(groupId, cursor, pageSize);
	               callback.onSuccess(result);
               } catch (HyphenateException e) {
                   callback.onError(e.getErrorCode(), e.getDescription());
               }
           }
       });
	}

	/**
	 * \~chinese
	 * 群组所有权给他人
	 * @param groupId
	 * @param newOwner
	 * @return
	 * @throws HyphenateException
	 *
	 * \~english
	 * Transfer group ownership to others, need owner's authority to access.
	 * @param groupId
	 * @param newOwner
	 * @return
	 * @throws HyphenateException
     */
	public EMGroup changeOwner(String groupId, String newOwner) throws HyphenateException {
		EMAError error = new EMAError();
		EMAGroup group = emaObject.transferGroupOwner(groupId, newOwner, error);
		handleError(error);
		return new EMGroup(group);
	}

	/**
	 * \~chinese
	 * 群组所有权给他人
	 * @param groupId
	 * @param newOwner
	 * @return
	 * @throws HyphenateException
	 *
	 * \~english
	 * Transfer group ownership to others, need owner's authority to access.
	 * @param groupId
	 * @param newOwner
	 * @return
	 * @throws HyphenateException
	 */
	public void asyncChangeOwner(final String groupId, final String newOwner, final EMValueCallBack<EMGroup> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					EMGroup group = changeOwner(groupId, newOwner);
					callBack.onSuccess(group);
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
 	 * \~chinese
	 * 增加群组管理员，需要owner权限
	 * @param groupId
	 * @param admin
	 * @return
	 * @throws HyphenateException
	 *
	 * \~english
	 * add group administrator, need owner's authority to access.
     * @param groupId
	 * @param admin
	 * @return
	 * @throws HyphenateException
     */
	public EMGroup addGroupAdmin(final String groupId, final String admin) throws HyphenateException {
		EMAError error = new EMAError();
		EMAGroup group = emaObject.addGroupAdmin(groupId, admin, error);
		handleError(error);
		return new EMGroup(group);
	}

	/**
	 * \~chinese
	 * 增加群组管理员，需要owner权限
	 * @param groupId
	 * @param admin
	 * @return
	 * @throws HyphenateException
	 *
	 * \~english
	 * add group administrator, need owner's authority to access.
	 * @param groupId
	 * @param admin
	 * @return
	 * @throws HyphenateException
	 */
	public void asyncAddGroupAdmin(final String groupId, final String admin, final EMValueCallBack<EMGroup> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(addGroupAdmin(groupId, admin));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 删除群组管理员，需要owner权限
	 * @param groupId
	 * @param admin
	 * @return
	 * @throws HyphenateException
	 *
	 * \~english
	 * remove group administrator, need owner's authority to access.
	 * @param groupId
	 * @param admin
	 * @return
	 * @throws HyphenateException
     */
	public EMGroup removeGroupAdmin(String groupId, String admin) throws HyphenateException {
		EMAError error = new EMAError();
		EMAGroup group = emaObject.removeGroupAdmin(groupId, admin, error);
		handleError(error);
		return new EMGroup(group);
	}

	/**
	 * \~chinese
	 * 删除群组管理员，需要owner权限
	 * @param groupId
	 * @param admin
	 * @return
	 * @throws HyphenateException
	 *
	 * \~english
	 * remove group administrator, need owner's authority to access.
	 * @param groupId
	 * @param admin
	 * @return
	 * @throws HyphenateException
	 */
	public void asyncRemoveGroupAdmin(final String groupId, final String admin, final EMValueCallBack<EMGroup> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(removeGroupAdmin(groupId, admin));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 禁止某些群组成员发言, 需要群组拥有者或者管理员权限
	 * @param groupId
	 * @param muteMembers 禁言的用户列表
	 * @param duration 禁言的时间，单位是毫秒
	 * @return
	 * @throws HyphenateException
	 *
	 * \~english
	 * mute group members, banning post message for some time, need owner or administrator's authority to access.
	 * @param groupId
	 * @param muteMembers mute member list
	 * @param duration mute duration
	 * @return
	 * @throws HyphenateException
     */
	public EMGroup muteGroupMembers(String groupId, List<String> muteMembers, long duration) throws HyphenateException {
		EMAError error = new EMAError();
		EMAGroup group = emaObject.muteGroupMembers(groupId, muteMembers, duration, error);
		handleError(error);
		return new EMGroup(group);
	}

	/**
	 * \~chinese
	 * 禁止某些群组成员发言, 需要群组拥有者或者管理员权限
	 * @param groupId
	 * @param muteMembers   需要禁言的群组成员
	 * @param duration 禁言持续时间，单位是毫秒
	 * @return
	 * @throws HyphenateException
	 *
	 * \~english
	 * mute group members, banning post message for some time, need owner or administrator's authority to access.
	 * @param groupId
	 * @param muteMembers mute group members
	 * @param duration mute duration, in milli-seconds
	 * @return
	 * @throws HyphenateException
	 */
	public void aysncMuteGroupMembers(final String groupId, final List<String> muteMembers, final long duration, final EMValueCallBack<EMGroup> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(muteGroupMembers(groupId, muteMembers, duration));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinse
	 * 解除禁言, 需要群组拥有者或者管理员权限
	 * @param groupId
	 * @param members
	 * @return
	 * @throws HyphenateException
	 *
	 * \~english
	 * Unmute group members, need owner or administrator's authority to access.
     */
	public EMGroup unMuteGroupMembers(String groupId, List<String> members) throws HyphenateException {
		EMAError error = new EMAError();
		EMAGroup group = emaObject.unMuteGroupMembers(groupId, members, error);
		handleError(error);
		return new EMGroup(group);
	}

	/**
	 * \~chinse
	 * 取消禁言，需要群组拥有者或者管理员权限
	 * @param groupId
	 * @param members
	 * @return
	 * @throws HyphenateException
	 *
	 * \~english
	 * Unmute group members, need owner or administrator's authority to access.
	 * @param groupId
	 * @param members
	 * @param callBack
	 * @return
	 * @throws HyphenateException
	 */
	public void asyncUnMuteGroupMembers(final String groupId, final List<String> members, final EMValueCallBack<EMGroup> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(unMuteGroupMembers(groupId, members));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 获取群组的禁言列表，需要群组拥有者或者管理员权限
	 * @param groupId
	 * @param pageNum
	 * @param pageSize
	 * @return Map.entry.key 是禁言的成员id，Map.entry.value是禁言动作存在的时间，单位是毫秒。
	 * @throws HyphenateException
	 *
	 * \~english
	 * fetch mute list, which contains mute members and mute time, need owner or administrator's authority to access.
	 * @param groupId
	 * @param pageNum
	 * @param pageSize
	 * @return Map.entry.key is username of mute action, Map.entry.value is expired time of banning post action, in milli-seconds
     */
	public Map<String, Long> fetchGroupMuteList(String groupId, int pageNum, int pageSize) throws HyphenateException {
		EMAError error = new EMAError();
		Map<String, Long> muteMembers = emaObject.fetchGroupMutes(groupId, pageNum, pageSize, error);
		handleError(error);
		return muteMembers;
	}

	/**
	 * \~chinese
	 * 获取群组的禁言列表，需要群组拥有者或者管理员权限
	 * @param groupId
	 * @param pageNum
	 * @param pageSize
	 * @param callBack callback function that receive the executed result.
	 *                 Map.entry.key is username of mute action, Map.entry.value is expired time of banning post action.
	 * @throws HyphenateException
	 *
	 * \~english
	 * fetch mute list, which contains mute members and mute time, need owner or administrator's authority to access.
	 * @param groupId
	 * @param pageNum
	 * @param pageSize
	 * @param callBack callback function that receive the executed result,
	 *                 Map.entry.key is username of mute action, Map.entry.value is expired time of banning post action, in milli-seconds
	 *
	 */
	public void asyncFetchGroupMuteList(final String groupId, final int pageNum, final int pageSize, final EMValueCallBack<Map<String, Long>> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchGroupMuteList(groupId, pageNum, pageSize));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	/**
	 * \~chinese
	 * 从服务器获分页获取群组黑名单，方法适用于成员数量较大的群组，需要群组拥有者或者管理员权限
	 * @param groupId       群组id
	 * @param pageNum       分页号
	 * @param pageSize      分页大小
	 * @return
	 * @throws HyphenateException
	 *
	 * \~english
	 * fetch black list from server, need owner or administrator's authority to access.
	 * @param groupId       group id
	 * @param pageNum       page number
	 * @param pageSize      page size
	 * @return
	 * @throws HyphenateException
     */
	public List<String> fetchGroupBlackList(String groupId, int pageNum, int pageSize) throws HyphenateException {
		EMAError error = new EMAError();
		List<String> muteMembers = emaObject.fetchGroupBlackList(groupId, pageNum, pageSize, error);
		handleError(error);
		return muteMembers;
	}

	/**
	 * \~chinese
	 * 从服务器获分页获取群组黑名单，方法适用于成员数量较大的群组，需要群组拥有者或者管理员权限
	 * @param groupId       群组id
	 * @param pageNum       分页号
	 * @param pageSize      分页大小
	 * @param callBack      异步函数回调方法
	 * @return
	 * @throws HyphenateException
	 *
	 * \~english
	 * fetch black list from server, need owner or administrator's authority to access.
	 * @param groupId       group id
	 * @param pageNum       page number
	 * @param pageSize      page size
	 * @param callBack      asynchronous callback method
	 * @return
	 * @throws HyphenateException
	 */
	public void asyncFetchGroupBlackList(final String groupId, final int pageNum, final int pageSize, final EMValueCallBack<List<String>> callBack) {
		EMClient.getInstance().execute(new Runnable() {

			@Override
			public void run() {
				try {
					callBack.onSuccess(fetchGroupBlackList(groupId, pageNum, pageSize));
				} catch (HyphenateException e) {
					callBack.onError(e.getErrorCode(), e.getDescription());
				}
			}
		});
	}

	// ============================= group_reform new add api end
	
	void onLogout() {		
	}
	
	private void handleError(EMAError error)  throws HyphenateException {
	    if (error.errCode() != EMAError.EM_NO_ERROR) {
	        throw new HyphenateException(error);
	    }
	}
}
