package com.hippoagent.utils;

import android.app.Activity;
import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.Intent;
import android.media.AudioManager;
import android.net.Uri;
import android.os.Environment;
import android.os.SystemClock;
import androidx.core.content.ContextCompat;
import android.text.Spannable;
import android.text.TextUtils;
import android.text.style.ForegroundColorSpan;
import android.text.style.RelativeSizeSpan;
import android.widget.TextView;
import android.widget.Toast;

import com.hippoagent.BuildConfig;
import com.hippoagent.HippoConfig;
import com.hippoagent.R;
import com.hippoagent.datastructure.FuguAppConstant;
import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.GoogleApiAvailability;
import com.hippoagent.langs.Restring;

import java.io.File;
import java.text.DecimalFormat;

/**
 * Created by gurmail on 25/04/18.
 *
 * @author gurmail
 */

public class Utils implements FuguAppConstant {

    public final static boolean isValidEmail(CharSequence target) {
        if (target == null) {
            return false;
        } else {
            return android.util.Patterns.EMAIL_ADDRESS.matcher(target).matches();
        }
    }

    public final static boolean isValidPhoneNumber(String phoneNo) {
        if(!TextUtils.isEmpty(phoneNo) && phoneNo.length() >7 && phoneNo.length()<14 && checkIfOnlyDigits(phoneNo)) {
            return isPhoneValid(phoneNo);
        } else {
            return false;
        }
    }

    private static boolean checkIfOnlyDigits(String strTocheck) {
        //Pattern regex = "[0-9+]+";
        return strTocheck.matches("[0-9+]+");
    }


    public static boolean isPhoneValid(CharSequence phone) {
        if(phone.length()<9)
            return false;
        return android.util.Patterns.PHONE.matcher(phone).matches();
    }

    /**
     * Method to assign Strings safely
     *
     * @param assignable
     * @param alternative
     * @return
     */
    public static String assign(String assignable, String alternative) {

        return assignable == null ?
                (alternative == null ? "" : alternative) :
                (assignable.equals("null") ? assign(alternative) : assignable);
    }

    /**
     * Method to assign strings Safely
     *
     * @param assignable
     * @return
     */
    public static String assign(String assignable) {

        return assignable == null || assignable.equalsIgnoreCase("[]") ? "" : assignable;
    }

    /**
     * Method to redirect the User to the page from
     * where a new version of App could be downloaded.
     */
    public static void redirectUserToUrl(Activity activity, String webUrl) {

        if (webUrl == null)
            return;

        if (webUrl.isEmpty())
            return;

        boolean isRecognizable = webUrl.startsWith("http://") || webUrl.startsWith("https://");
        webUrl = (isRecognizable ? "" : "http://") + webUrl;

        try {
            Intent openWebUrl = new Intent(Intent.ACTION_VIEW);
            openWebUrl.setData(Uri.parse(webUrl));
            activity.startActivity(openWebUrl);
            //activity.overridePendingTransition(R.anim.right_in, R.anim.left_out);
        } catch (ActivityNotFoundException e) {
            //snackBar(activity, activity.getString(R.string.no_app_found_to_open_the_link), Codes.SnackBarType.ERROR);

            e.printStackTrace();
        }
    }

    private static final int    MULTIPLE_CLICK_THRESHOLD       = 2000; //in milli seconds
    // variable to track event time
    private static long mLastClickTime = 0;

    /**
     * Method to prevent multiple clicks
     */
    public static boolean preventMultipleClicks() {
        if (SystemClock.elapsedRealtime() - mLastClickTime < MULTIPLE_CLICK_THRESHOLD) {
            return false;
        }
        mLastClickTime = SystemClock.elapsedRealtime();
        return true;
    }

    private static final String AppName = HippoConfig.getInstance().getAttributes().getAppName();
    public static String getDirectoryPath(String folder) {
        return Environment.getExternalStorageDirectory()
                + File.separator + AppName + File.separator + folder;
    }

    private static final int NOT_FOUND = -1;

    /**
     * The extension separator character.
     * @since 1.4
     */
    public static final char EXTENSION_SEPARATOR = '.';

    /**
     * The Unix separator character.
     */
    private static final char UNIX_SEPARATOR = '/';

    /**
     * The Windows separator character.
     */
    private static final char WINDOWS_SEPARATOR = '\\';


    /**
     * Gets the extension of a filename.
     * <p>
     * This method returns the textual part of the filename after the last dot.
     * There must be no directory separator after the dot.
     * <pre>
     * foo.txt      --&gt; "txt"
     * a/b/c.jpg    --&gt; "jpg"
     * a/b.txt/c    --&gt; ""
     * a/b/c        --&gt; ""
     * </pre>
     * <p>
     * The output will be the same irrespective of the machine that the code is running on.
     *
     * @param filename the filename to retrieve the extension of.
     * @return the extension of the file or an empty string if none exists or {@code null}
     * if the filename is {@code null}.
     */
    public static String getExtension(final String filename) {
        if (filename == null) {
            return null;
        }
        final int index = indexOfExtension(filename);
        if (index == NOT_FOUND) {
            return "";
        } else {
            return filename.substring(index + 1);
        }
    }

    /**
     * Returns the index of the last extension separator character, which is a dot.
     * <p>
     * This method also checks that there is no directory separator after the last dot. To do this it uses
     * {@link #indexOfLastSeparator(String)} which will handle a file in either Unix or Windows format.
     * </p>
     * <p>
     * The output will be the same irrespective of the machine that the code is running on.
     * </p>
     *
     * @param filename
     *            the filename to find the last extension separator in, null returns -1
     * @return the index of the last extension separator character, or -1 if there is no such character
     */
    public static int indexOfExtension(final String filename) {
        if (filename == null) {
            return NOT_FOUND;
        }
        final int extensionPos = filename.lastIndexOf(EXTENSION_SEPARATOR);
        final int lastSeparator = indexOfLastSeparator(filename);
        return lastSeparator > extensionPos ? NOT_FOUND : extensionPos;
    }

    /**
     * Returns the index of the last directory separator character.
     * <p>
     * This method will handle a file in either Unix or Windows format.
     * The position of the last forward or backslash is returned.
     * <p>
     * The output will be the same irrespective of the machine that the code is running on.
     *
     * @param filename  the filename to find the last path separator in, null returns -1
     * @return the index of the last separator character, or -1 if there
     * is no such character
     */
    public static int indexOfLastSeparator(final String filename) {
        if (filename == null) {
            return NOT_FOUND;
        }
        final int lastUnixPos = filename.lastIndexOf(UNIX_SEPARATOR);
        final int lastWindowsPos = filename.lastIndexOf(WINDOWS_SEPARATOR);
        return Math.max(lastUnixPos, lastWindowsPos);
    }

    public static String humanReadableByteCount(long bytes, boolean si) {
        int unit = si ? 1024 : 1024;
        if (bytes < unit) return bytes + " B";
        int exp = (int) (Math.log(bytes) / Math.log(unit));
        String pre = (si ? "kMGTPE" : "KMGTPE").charAt(exp-1) + (si ? "" : "i");
        return String.format("%.2f %sB", bytes / Math.pow(unit, exp), pre);
    }

    public static String humanReadableSize(long bytes, boolean si) {
        int unit = si ? 1024 : 1024;
        if (bytes < unit) return bytes + " B";
        int exp = (int) (Math.log(bytes) / Math.log(unit));
        String pre = (si ? "kMGTPE" : "KMGTPE").charAt(exp-1) + (si ? "" : "i");
        return String.format("%.0f %sB", bytes / Math.pow(unit, exp), pre);
    }


    public static String getOrCreateDirectoryPath(String folderName) {
        String path = Environment.getExternalStorageDirectory()
                + File.separator + AppName + File.separator + folderName;
        File folder = new File(path);
        if(!folder.exists())
            folder.mkdirs();

        return path;
    }

    public static String setImageFullPath(String fileName) {
        String destinationDirectoryPath = Utils.getOrCreateDirectoryPath(FOLDER_TYPE.get(DocumentType.IMAGE.type));
        return destinationDirectoryPath + File.separator + fileName;
    }

    public static String getFileName(String filename, String muid) {
        if(!filename.contains(muid)) {
            String ext = Utils.getExtension(filename);
            String name = Utils.extractFileNameWithoutSuffix(filename);
            filename = name + "_" + muid + "." + ext;
        }
        return filename;
    }

    /**
     * Extract the file name in a URL
     * /storage/emulated/legacy/Download/sample.pptx = sample
     *
     * @param url String of a URL
     * @return the file name of URL without suffix
     */
    public static String extractFileNameWithoutSuffix(String url) {
        try {
            return url.substring(url.lastIndexOf("/") + 1, url.lastIndexOf("."));
        } catch (StringIndexOutOfBoundsException e) {
            try {
                return url.substring(url.lastIndexOf("/") + 1);
            } catch (Exception e1) {
                e1.printStackTrace();
            }
            e.printStackTrace();
            return "";
        }
    }

    public static String getMuid(String name) {
        try {
            return name.substring(name.lastIndexOf("_") + 1, name.lastIndexOf("."));
        } catch (StringIndexOutOfBoundsException e) {
            e.printStackTrace();
            return "";
        }
    }

    /**
     * Extract the file name in a URL
     * /storage/emulated/legacy/Download/sample.pptx = sample.pptx
     *
     * @param url String of a URL
     * @return the file name of URL with suffix
     */
    public static String extractFileNameWithSuffix(String url) {
        return url.substring(url.lastIndexOf("/") + 1);
    }

    public static void setColor(Context context, TextView view, String text) {
        String subtext = "First Name";
        String fulltext = text.replace("{{{full_name}}}", subtext);
        view.setText(fulltext, TextView.BufferType.SPANNABLE);
        Spannable str = (Spannable) view.getText();
        int i = fulltext.indexOf(subtext);
        if(i<0)
            return;
        str.setSpan(new ForegroundColorSpan(ContextCompat.getColor(context, R.color.colorPrimary)), i, i + subtext.length(), Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);
        str.setSpan(new RelativeSizeSpan(1.1f), i, i + subtext.length(), Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);
    }

    public static int dp2px(Context context, float dipValue) {
        final float scale = context.getResources().getDisplayMetrics().density;
        return (int) (dipValue * scale + 0.5f);
    }

    /**
     * @return true if google play service present & updated
     * false if not presented or not updated
     */
    private static int REQ_CODE_PLAY_SERVICES_RESOLUTION = 16061;

    public static boolean checkPlayServices(Context context) {
        GoogleApiAvailability apiAvailability = GoogleApiAvailability.getInstance();
        int resultCode = apiAvailability.isGooglePlayServicesAvailable(context);
        if (resultCode != ConnectionResult.SUCCESS) {
            if (apiAvailability.isUserResolvableError(resultCode)) {
                apiAvailability.getErrorDialog((Activity) context, resultCode, REQ_CODE_PLAY_SERVICES_RESOLUTION)
                        .show();
            } else {
                Toast.makeText(context, Restring.getString(context, R.string.error_device_not_supported), Toast.LENGTH_SHORT).show();
            }
            return false;
        }
        return true;
    }

    public static AudioManager getAudioManager(Context context) {
        return (AudioManager)context.getSystemService(Context.AUDIO_SERVICE);
    }

    private static DecimalFormat decimalFormatMoney;
    public static DecimalFormat getMoneyDecimalFormat(){
        if(decimalFormatMoney == null){
            decimalFormatMoney = new DecimalFormat("#.##");
        }
        return decimalFormatMoney;
    }

}
