package com.hippoagent.helper

import android.os.Handler
import com.hippoagent.Config
import com.hippoagent.HippoApplication
import com.hippoagent.callback.OnCloseListener
import com.hippoagent.callback.OnInitializedListener
import com.hippoagent.eventbus.BusProvider
import faye.FuguAgentFayeClient
import faye.FuguAgentFayeClientListener
import faye.FuguAgentMetaMessage
import org.json.JSONException
import org.json.JSONObject
import android.os.HandlerThread
import android.text.TextUtils
import com.hippoagent.HippoConfig
import com.hippoagent.confcall.OngoingCallService
import java.util.*

/**
 * Created by gurmail on 2020-04-18.
 * @author gurmail
 */
object ConnectionManager: OnCloseListener, OnInitializedListener {

    override fun onClose() {
        stopFayeClient()
    }

    override fun onInitialized() {
        initFayeConnection()
    }

    private var fayeClient: FuguAgentFayeClient? = null
    private var meta = FuguAgentMetaMessage()
    private val mChannels: HashSet<String> = HashSet()
    private val mPendingChannels: HashSet<String> = HashSet()
    private var connecting = false
    private var isDelaySubscribe = false

    private val RECONNECTION_TIME = 5000
    private var retryCount = 0
    private var stopConnection = false

    private val NOT_CONNECTED = 0
    private val CONNECTED_TO_INTERNET = 1
    private val CONNECTED_TO_INTERNET_VIA_WIFI = 2
    private var NETWORK_STATUS = 1

    private var pongCount = 0
    private const val MAX_COUNT = 20
    private const val MAX_RETRY_ATTEMPTS = 50

    fun initFayeConnection() {
        fayeClient = getFayeClient()
        forceStop = false
        if(fayeClient != null && !isConnected() && !connecting) {
            fayeClient!!.connectServer()
            connecting = true
        }

        setListener()
    }

    private fun getFayeClient(): FuguAgentFayeClient? {
        if(fayeClient == null) {
            val jsonExt = JSONObject()
            try {
                if (HippoApplication.getInstance().userData != null) {
                    jsonExt.put("user_id", HippoApplication.getInstance().userData.userId)
                    jsonExt.put("device_type", 1)
                    jsonExt.put("source", 1)
                } else {
                    connecting = false
                    return null
                }
            } catch (e: JSONException) {
                e.printStackTrace()
            }

            meta.setAllExt(jsonExt.toString())
            fayeClient = FuguAgentFayeClient(getFayeUrl(), meta)
        }
        return fayeClient
    }

    private fun reconnectConnection() {
        initFayeConnection()
    }

    private fun getFayeUrl(): String {
        return Config.getFayeServerUrl() + "faye"
    }

    private fun setListener() {
        if(fayeClient != null) {
            fayeClient!!.setmManagerListener(object: FuguAgentFayeClientListener {

                override fun onConnectedServer(fc: FuguAgentFayeClient?) {
                    stopConnection = true
                    connecting = true
                    retryCount = 0
                    if(!isDelaySubscribe)
                        subscribePendingChannels()
                    else
                        subscribeDelayPendingChannels()
                    BusProvider.getInstance().post(FayeMessage(BusEvents.CONNECTED_SERVER.toString(), "", ""))
                    stopAutoAttemptConnection()
                }

                override fun onDisconnectedServer(fc: FuguAgentFayeClient?) {
                    connecting = false
                    stopConnection = false
                    if(!forceStop) {
                        saveSubsribedChannels()
                        BusProvider.getInstance().post(FayeMessage(BusEvents.DISCONNECTED_SERVER.toString(), "", ""))
                        attemptAutoConnection()
                    }
                }

                override fun onReceivedMessage(fc: FuguAgentFayeClient?, msg: String?, channel: String?) {
                    stopConnection = true
                    connecting = true
                    retryCount = 0
                    ParseMessage.receivedMessage(msg, channel)
                }

                override fun onPongReceived() {
                    stopConnection = true
                    connecting = true
                    retryCount = 0
                    subscribePendingChannels()
                    BusProvider.getInstance().post(FayeMessage(BusEvents.PONG_RECEIVED.toString(), "", ""))
                    pongCount()
                }

                override fun onWebSocketError() {
                    connecting = false
                    pongCount = 0
                    BusProvider.getInstance().post(FayeMessage(BusEvents.WEBSOCKET_ERROR.toString(), "", ""))
                }

                override fun onErrorReceived(fc: FuguAgentFayeClient?, msg: String?, channel: String?) {
                    connecting = true
                    BusProvider.getInstance().post(FayeMessage(BusEvents.ERROR_RECEIVED.toString(), channel, msg))
                }

                override fun onNotConnected() {
                    connecting = false
                    pongCount = 0
                    BusProvider.getInstance().post(FayeMessage(BusEvents.NOT_CONNECTED.toString(), "", ""))
                }

            })
        }
    }

    public fun isConnected(): Boolean {
        return fayeClient != null && fayeClient!!.isConnectedServer && fayeClient!!.isOpened
    }

    public fun subScribeChannel(channel: String) {
        if(isConnected()) {
            if(!mChannels.contains(channel)) {
                mChannels.add(channel)
                fayeClient!!.subscribeChannel(channel)
            } else {
                // already subscribed channel
            }
        } else {
            // pending channels for subscriptions
            mPendingChannels.add(channel)
            reconnectConnection()
        }
    }

    public fun subscribeOnDelay(channel: String) {
        if(isConnected()) {
            if(!mChannels.contains(channel)) {
                mChannels.add(channel)
                fayeClient!!.subscribeChannel(channel)
            } else {
                mChannels.add(channel)
                fayeClient!!.subscribeChannel(channel)
                // already subscribed channel
            }
        } else {
            isDelaySubscribe = true
            mPendingChannels.add(channel)
            reconnectConnection()
        }
    }

    public fun unsubScribeChannel(channel: String) {
        mChannels.remove(channel)
        fayeClient!!.unsubscribeChannel(channel)
    }

    public fun publish(channel: String, jsonObject: JSONObject) {
        publish(channel, jsonObject, false)
    }
    public fun publish(channel: String, jsonObject: JSONObject, canStoreLocally: Boolean) {
        if(isConnected()) {
            try {
                val string = HippoConfig.getInstance().currentLanguage
                if(!TextUtils.isEmpty(string)) {
                    jsonObject.put("lang", string)
                }
            } catch (e: Exception) {
            }
            fayeClient!!.publish(channel, jsonObject)
        } else {
            mPendingChannels.add(channel)
            if(canStoreLocally) {

            }
            //CoroutineScope
            reconnectConnection()
        }
    }

    private fun pongCount() {
        pongCount += 1
        if(pongCount>MAX_COUNT && !OngoingCallService.NotificationServiceState.isConferenceConnected) {
            if(HippoApplication.getInstance() == null ||
                    !ConnectionUtils.isAppRunning(HippoApplication.getInstance().applicationContext)) {
                stopFayeClient()
            } else {
                pongCount = 0
            }
        }
    }

    var forceStop: Boolean = false
    private fun stopFayeClient() {
        if(fayeClient != null && !OngoingCallService.NotificationServiceState.isConferenceConnected) {
            val thread = HandlerThread("TerminateThread")
            thread.start()
            Handler(thread.looper).post(Runnable {
                if(fayeClient != null) {
                    mPendingChannels.clear()
                    mChannels.clear()
                    if(isConnected()) {
                        forceStop = true
                        fayeClient!!.disconnectServer()
                    }
                    pongCount = 0
                    fayeClient = null
                    connecting = false
                }
            })
        }
    }

    private fun subscribePendingChannels() {
        synchronized(this) {
            if(mPendingChannels.size>0) {
                val tempChannel = mPendingChannels
                for (channel in tempChannel) {
                    fayeClient!!.subscribeChannel(channel)
                }
                mPendingChannels.removeAll(tempChannel)
            }
        }
    }

    private fun subscribeDelayPendingChannels() {
        Handler().postDelayed({
            isDelaySubscribe = false
            if(mPendingChannels.size>0) {
                val tempChannel = mPendingChannels
                for (channel in tempChannel) {
                    fayeClient!!.subscribeChannel(channel)
                }
                mPendingChannels.removeAll(tempChannel)
            }
        }, 2000)
    }

    public fun changeStatus(status: Int) {
        NETWORK_STATUS = status
        when (status) {
            NOT_CONNECTED -> {
                saveSubsribedChannels()
                stopAutoAttemptConnection()
                retryCount = 0
            }
            CONNECTED_TO_INTERNET, CONNECTED_TO_INTERNET_VIA_WIFI -> {
                initFayeConnection()
            }
        }
    }

    @Synchronized
    private fun saveSubsribedChannels() {
        connecting = false
        if(mChannels.size>0) {
            mPendingChannels.addAll(mChannels)
            mChannels.clear()
        }
    }


    private fun attemptAutoConnection() {
        if(NETWORK_STATUS != NOT_CONNECTED) {
            if(retryCount < MAX_RETRY_ATTEMPTS && !stopConnection) {
                retryCount += 1
                try {
                    Timer().schedule(object : TimerTask() {
                        override fun run() {
                            try {
                                if(!stopConnection)
                                    initFayeConnection()
                            } catch (e: Exception) {

                            }
                        }
                    }, RECONNECTION_TIME.toLong())
                } catch (e: Exception) {
                }
            }
        }
    }

    private fun stopAutoAttemptConnection() {
        //handler.removeCallbacks(runnable)
    }
}