package com.hippoagent.dialogs;

import android.app.Activity;
import android.app.Dialog;

import androidx.fragment.app.Fragment;
import androidx.core.content.ContextCompat;
import android.text.SpannableString;
import android.text.TextPaint;
import android.text.method.LinkMovementMethod;
import android.text.style.ClickableSpan;
import android.view.View;
import android.view.Window;
import android.view.WindowManager;
import android.widget.Button;
import android.widget.TextView;

import com.hippoagent.R;
import com.hippoagent.model.onBoard.UserConfig;
import com.hippoagent.utils.Log;
import com.hippoagent.utils.Pref1;
import com.hippoagent.utils.Utils;
import com.google.gson.Gson;


/**
 * Created by gurmail on 02/05/18.
 *
 * @author gurmail
 */
public class ConfirmationDialog {


    private static final String TAG = ConfirmationDialog.class.getSimpleName();

    private Dialog optionsDialog;

    /*
     * The context which would listen to the Events
     * of the AlertDialog (must be an implementation of
     * Listener)
     */
    private Listener listener;

    /*
     * The activity on which the AlertDialog would be displayed
     */
    private Activity activity;


    // The title for the Options Dialog
    private String title;
    // The message to be conveyed to the User
    private String message;
    // The text for Ok button
    private String positiveButton;
    // The text for Cancel button
    private String negativeButton;

    /**
     * Method to showOn a multi-event Dialog
     */
    private ConfirmationDialog init() {

        try {

            optionsDialog = new Dialog(activity, android.R.style.Theme_Translucent_NoTitleBar);
            optionsDialog.setContentView(R.layout.dialog_confirmation);

            Window dialogWindow = optionsDialog.getWindow();
            WindowManager.LayoutParams layoutParams = dialogWindow.getAttributes();
            layoutParams.dimAmount = 0.6f;

            dialogWindow.getAttributes().windowAnimations = R.style.CustomDialog;

            dialogWindow.addFlags(WindowManager.LayoutParams.FLAG_DIM_BEHIND);
            optionsDialog.setCancelable(false);
            optionsDialog.setCanceledOnTouchOutside(false);

            TextView tvTitle = (TextView) optionsDialog.findViewById(R.id.tvTitle);
            TextView tvMessage = (TextView) optionsDialog.findViewById(R.id.tvMessage);

            tvTitle.setText(title);
            setTocSpannableText(activity, tvMessage);
            //tvMessage.setText(message);

            Button btnConfirm = (Button) optionsDialog.findViewById(R.id.btnAction);
            Button btnCancel = (Button) optionsDialog.findViewById(R.id.btnNegative);


            btnConfirm.setText(positiveButton);
            btnCancel.setText(negativeButton);

            btnConfirm.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View view) {

                    optionsDialog.dismiss();

                    if (listener != null)
                        listener.performPositiveAction();
                }
            });

            btnCancel.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View view) {

                    optionsDialog.dismiss();

                    if (listener != null)
                        listener.performNegativeAction();
                }
            });

        } catch (Exception e) {
            e.printStackTrace();
            Log.e(TAG, "init: " + e.getMessage());
        }

        return this;
    }

    /**
     * Method to init the initialized dialog
     */
    public void show() {

        // Check if activity lives
        if (activity != null)
            // Check if dialog contains data
            if (optionsDialog != null)
                // Show the Dialog
                optionsDialog.show();
    }

    /**
     * Listener to listen to the events when a main_button
     * on the Options Dialog was pressed
     * <p/>
     * Developer: Rishabh
     * Dated: 19/05/15.
     */
    public interface Listener {

        /**
         * Override this method to perform operations
         * after OK button was pressed
         *
         */
        void performPositiveAction();


        /**
         * Override this method to perform operations
         * after CANCEL button was pressed
         *
         */
        void performNegativeAction();
    }

    /**
     * Class to build the OptionsDialog
     */
    public static class Builder {

        private ConfirmationDialog optionsDialog = new ConfirmationDialog();

        /**
         * Constructor to initialize the OptionsDialog
         * (for an Activity)
         *
         * @param activity
         */
        public Builder(Activity activity) {

            optionsDialog.activity = activity;

            if (activity instanceof Listener)
                optionsDialog.listener = (Listener) activity;
        }

        /**
         * Constructor to initialize the OptionsDialog
         * (for a Fragment)
         *
         * @param fragment
         */
        public Builder(Fragment fragment) {

            optionsDialog.activity = fragment.getActivity();

            if (fragment instanceof Listener)
                optionsDialog.listener = (Listener) fragment;
        }

        /**
         * Method to set a custom listener to listen
         * the events from OptionsDialog
         *
         * @param listener
         * @return
         */
        public Builder listener(Listener listener) {

            optionsDialog.listener = listener;
            return this;
        }

        /**
         * Method to set title to the Title
         *
         * @param title
         * @return
         */
        public Builder title(String title) {

            optionsDialog.title = title;
            return this;
        }

        /**
         * Method to set message to the Dialog
         *
         * @param message
         * @return
         */
        public Builder message(String message) {

            optionsDialog.message = message;
            return this;
        }

        /**
         * Method to set text to the Positive Button
         *
         * @param buttonText
         * @return
         */
        public Builder positiveButton(String buttonText) {

            optionsDialog.positiveButton = buttonText;
            return this;
        }

        /**
         * Method to set Text to the Negative Button
         *
         * @param cancel
         * @return
         */
        public Builder negativeButton(String cancel) {

            optionsDialog.negativeButton = cancel;
            return this;
        }

        /**
         * Method to init the Options Dialog
         *
         * @return
         */
        public ConfirmationDialog build() {

            optionsDialog.title = getString(R.string.tnc_title);
            optionsDialog.message = getString(R.string.tnc_message);
            optionsDialog.positiveButton = getString(R.string.accept);
            optionsDialog.negativeButton = getString(R.string.decline);

            return optionsDialog.init();
        }

        /**
         * Method to retrieve the String resource
         *
         * @param resourceId
         * @return
         */
        private String getString(int resourceId) {
            return optionsDialog.activity.getString(resourceId);
        }
    }

    /**
     * set spannable text
     */
    private void setTocSpannableText(final Activity activity, TextView textView) {
        String spanText = activity.getResources().getString(R.string.terms_of_service);
        String privacyText = activity.getResources().getString(R.string.privacy_policy);
        String fullText = activity.getResources().getString(R.string.tnc_message);
        SpannableString spannableString = new SpannableString(activity.getResources().getString(R.string.tnc_message));
        ClickableSpan terms = new ClickableSpan() {
            @Override
            public void onClick(final View widget) {
                Utils.redirectUserToUrl(activity, getLink(activity, 1));
            }

            @Override
            public void updateDrawState(final TextPaint ds) {
                // super.updateDrawState(ds);
                ds.setUnderlineText(false);
                ds.setColor(ContextCompat.getColor(activity, R.color.colorPrimaryDark));
            }
        };
        ClickableSpan privacy = new ClickableSpan() {
            @Override
            public void onClick(final View widget) {
                Utils.redirectUserToUrl(activity, getLink(activity, 2));
            }

            @Override
            public void updateDrawState(final TextPaint ds) {
                // super.updateDrawState(ds);
                ds.setUnderlineText(false);
                ds.setColor(ContextCompat.getColor(activity, R.color.colorPrimaryDark));
            }
        };

        spannableString.setSpan(terms, fullText.indexOf(spanText), (fullText.indexOf(spanText)+spanText.length()), 0);
        spannableString.setSpan(privacy, fullText.indexOf(privacyText), (fullText.indexOf(privacyText)+privacyText.length()), 0);

        textView.setMovementMethod(LinkMovementMethod.getInstance());
        textView.setText(spannableString, TextView.BufferType.SPANNABLE);
        textView.setSelected(true);
    }

    private String conditionLink = "";
    private String policyLink = "";

    private String getLink(Activity activity, int type) {
        try {
            UserConfig userConfig = new Gson().fromJson(Pref1.with(activity).getString("userConfig", ""), UserConfig.class);
            conditionLink = userConfig.getData().getTermsAndCondition();
            policyLink = userConfig.getData().getPrivacyPolicy();
        } catch (Exception e) {
            e.printStackTrace();
            conditionLink = activity.getString(R.string.terms_of_service_link);
            policyLink = activity.getString(R.string.privacy_policy_link);
        }
        if(type == 1) {
            return conditionLink;
        } else {
            return policyLink;
        }
    }
}
