/*
 * Decompiled with CFR 0.152.
 */
package com.helger.base.pool;

import com.helger.annotation.Nonnegative;
import com.helger.annotation.concurrent.GuardedBy;
import com.helger.annotation.concurrent.ThreadSafe;
import com.helger.base.concurrent.SimpleLock;
import com.helger.base.enforce.ValueEnforcer;
import com.helger.base.log.ConditionalLogger;
import com.helger.base.log.IHasConditionalLogger;
import com.helger.base.pool.IMutableObjectPool;
import com.helger.base.pool.IObjectPoolFactory;
import com.helger.base.reflection.GenericReflection;
import com.helger.base.state.ESuccess;
import com.helger.base.tostring.ToStringGenerator;
import jakarta.annotation.Nonnull;
import jakarta.annotation.Nullable;
import java.util.Arrays;
import java.util.concurrent.Semaphore;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public final class ObjectPool<DATATYPE>
implements IMutableObjectPool<DATATYPE>,
IHasConditionalLogger {
    private static final Logger LOGGER = LoggerFactory.getLogger(ObjectPool.class);
    private static final ConditionalLogger CONDLOG = new ConditionalLogger(LOGGER);
    private final SimpleLock m_aLock = new SimpleLock();
    private final IObjectPoolFactory<DATATYPE> m_aFactory;
    private final Semaphore m_aAvailable;
    @GuardedBy(value="m_aLock")
    private final Object[] m_aItems;
    @GuardedBy(value="m_aLock")
    private final boolean[] m_aUsed;

    public static boolean isSilentMode() {
        return CONDLOG.isDisabled();
    }

    public static boolean setSilentMode(boolean bl) {
        return !CONDLOG.setEnabled(!bl);
    }

    public ObjectPool(@Nonnegative int n, @Nonnull Supplier<? extends DATATYPE> supplier) {
        this(n, IObjectPoolFactory.wrap(supplier));
    }

    public ObjectPool(@Nonnegative int n, @Nonnull IObjectPoolFactory<DATATYPE> iObjectPoolFactory) {
        ValueEnforcer.isGT0(n, "ItemCount");
        ValueEnforcer.notNull(iObjectPoolFactory, "Factory");
        this.m_aAvailable = new Semaphore(n);
        this.m_aItems = new Object[n];
        this.m_aUsed = new boolean[n];
        Arrays.fill(this.m_aUsed, 0, n, false);
        this.m_aFactory = iObjectPoolFactory;
    }

    @Nonnegative
    public int getPoolSize() {
        return this.m_aItems.length;
    }

    @Nonnegative
    public int getBorrowedObjectCount() {
        return this.m_aItems.length - this.m_aAvailable.availablePermits();
    }

    public void clearUnusedItems() {
        this.m_aLock.lock();
        try {
            for (int i = 0; i < this.m_aItems.length; ++i) {
                if (this.m_aUsed[i]) continue;
                this.m_aItems[i] = null;
            }
        }
        finally {
            this.m_aLock.unlock();
        }
    }

    @Override
    @Nullable
    public DATATYPE borrowObject() {
        try {
            this.m_aAvailable.acquire();
        }
        catch (InterruptedException interruptedException) {
            CONDLOG.error("ObjectPool interrupted", (Exception)interruptedException);
            Thread.currentThread().interrupt();
            return null;
        }
        this.m_aLock.lock();
        try {
            for (int i = 0; i < this.m_aItems.length; ++i) {
                Object object;
                if (this.m_aUsed[i]) continue;
                int n = i;
                if (this.m_aItems[i] == null) {
                    CONDLOG.debug(() -> "ObjectPool creates a new object for index " + n);
                    object = this.m_aFactory.create();
                    this.m_aItems[i] = object;
                    if (object == null) {
                        throw new IllegalStateException("The factory returned a null object [1]!");
                    }
                } else {
                    CONDLOG.debug(() -> "ObjectPool reuses object for index " + n);
                    object = GenericReflection.uncheckedCast(this.m_aItems[i]);
                    if (this.m_aFactory.activate(object).isFailure()) {
                        CONDLOG.info(() -> "ObjectPool failed to activate object for index " + n);
                        object = this.m_aFactory.create();
                        this.m_aItems[i] = object;
                        if (object == null) {
                            throw new IllegalStateException("The factory returned a null object [2]!");
                        }
                    } else {
                        CONDLOG.debug(() -> "ObjectPool successfully activated object for index " + n);
                    }
                }
                this.m_aUsed[i] = true;
                Object object2 = object;
                return (DATATYPE)object2;
            }
            throw new IllegalStateException("Should never be reached - ObjectPool exceeds its limit. Looks like a programming error.");
        }
        finally {
            this.m_aLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    @Nonnull
    public ESuccess returnObject(@Nonnull DATATYPE DATATYPE) {
        this.m_aLock.lock();
        try {
            for (int i = 0; i < this.m_aItems.length; ++i) {
                if (!this.m_aUsed[i] || DATATYPE != this.m_aItems[i]) continue;
                int n = i;
                CONDLOG.debug(() -> "ObjectPool passivates object for index " + n);
                this.m_aFactory.passivate(DATATYPE);
                this.m_aUsed[i] = false;
                this.m_aAvailable.release();
                ESuccess eSuccess = ESuccess.SUCCESS;
                return eSuccess;
            }
            CONDLOG.error(() -> "Object " + String.valueOf(DATATYPE) + " is not pooled!");
            ESuccess eSuccess = ESuccess.FAILURE;
            return eSuccess;
        }
        finally {
            this.m_aLock.unlock();
        }
    }

    public String toString() {
        return new ToStringGenerator(this).append("Factory", this.m_aFactory).getToString();
    }
}

