/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.hazelcast.shaded.org.apache.calcite.sql.fun;

import com.hazelcast.shaded.org.apache.calcite.rel.type.RelDataType;
import com.hazelcast.shaded.org.apache.calcite.sql.SqlCall;
import com.hazelcast.shaded.org.apache.calcite.sql.SqlCallBinding;
import com.hazelcast.shaded.org.apache.calcite.sql.SqlFunction;
import com.hazelcast.shaded.org.apache.calcite.sql.SqlFunctionCategory;
import com.hazelcast.shaded.org.apache.calcite.sql.SqlKind;
import com.hazelcast.shaded.org.apache.calcite.sql.SqlOperandCountRange;
import com.hazelcast.shaded.org.apache.calcite.sql.type.OperandTypes;
import com.hazelcast.shaded.org.apache.calcite.sql.type.ReturnTypes;
import com.hazelcast.shaded.org.apache.calcite.sql.type.SqlOperandCountRanges;
import com.hazelcast.shaded.org.apache.calcite.sql.type.SqlOperandTypeChecker;
import com.hazelcast.shaded.org.apache.calcite.sql.type.SqlTypeTransforms;
import com.hazelcast.shaded.org.apache.calcite.sql.type.SqlTypeUtil;
import com.hazelcast.shaded.org.apache.calcite.sql.validate.SqlValidator;

import com.hazelcast.shaded.org.checkerframework.checker.nullness.qual.Nullable;

import java.util.Locale;

import static com.hazelcast.shaded.org.apache.calcite.util.Static.RESOURCE;

/**
 * Definition of the <code>JSON_INSERT</code>, <code>JSON_REPLACE</code>
 * and <code>JSON_SET</code> JSON Modify functions.
 */
public class SqlJsonModifyFunction extends SqlFunction {
  public SqlJsonModifyFunction(String name) {
    super(name,
        SqlKind.OTHER_FUNCTION,
        ReturnTypes.cascade(ReturnTypes.VARCHAR_2000,
            SqlTypeTransforms.FORCE_NULLABLE),
        null,
        OperandTypes.ANY,
        SqlFunctionCategory.SYSTEM);
  }

  @Override public SqlOperandCountRange getOperandCountRange() {
    return SqlOperandCountRanges.from(3);
  }

  @Override protected void checkOperandCount(SqlValidator validator,
      @Nullable SqlOperandTypeChecker argType, SqlCall call) {
    assert (call.operandCount() >= 3) && (call.operandCount() % 2 == 1);
  }

  @Override public boolean checkOperandTypes(SqlCallBinding callBinding,
      boolean throwOnFailure) {
    final int count = callBinding.getOperandCount();
    for (int i = 1; i < count; i += 2) {
      RelDataType nameType = callBinding.getOperandType(i);
      if (!SqlTypeUtil.isCharacter(nameType)) {
        if (throwOnFailure) {
          throw callBinding.newError(RESOURCE.expectedCharacter());
        }
        return false;
      }
      if (nameType.isNullable()) {
        if (throwOnFailure) {
          throw callBinding.newError(
              RESOURCE.argumentMustNotBeNull(
                  callBinding.operand(i).toString()));
        }
        return false;
      }
    }
    return true;
  }

  @Override public String getSignatureTemplate(int operandsCount) {
    assert operandsCount % 2 == 1;
    StringBuilder sb = new StringBuilder();
    sb.append("{0}(");
    for (int i = 1; i < operandsCount; i++) {
      sb.append(String.format(Locale.ROOT, "{%d} ", i + 1));
    }
    sb.append("{1})");
    return sb.toString();
  }
}
