/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.hazelcast.shaded.org.apache.calcite.schema.impl;

import com.hazelcast.shaded.org.apache.calcite.adapter.enumerable.CallImplementor;
import com.hazelcast.shaded.org.apache.calcite.adapter.enumerable.NullPolicy;
import com.hazelcast.shaded.org.apache.calcite.adapter.enumerable.ReflectiveCallNotNullImplementor;
import com.hazelcast.shaded.org.apache.calcite.adapter.enumerable.RexImpTable;
import com.hazelcast.shaded.org.apache.calcite.adapter.enumerable.RexToLixTranslator;
import com.hazelcast.shaded.org.apache.calcite.linq4j.tree.Expression;
import com.hazelcast.shaded.org.apache.calcite.linq4j.tree.Expressions;
import com.hazelcast.shaded.org.apache.calcite.rel.type.RelDataType;
import com.hazelcast.shaded.org.apache.calcite.rel.type.RelDataTypeFactory;
import com.hazelcast.shaded.org.apache.calcite.rex.RexCall;
import com.hazelcast.shaded.org.apache.calcite.schema.ImplementableFunction;
import com.hazelcast.shaded.org.apache.calcite.schema.QueryableTable;
import com.hazelcast.shaded.org.apache.calcite.schema.ScannableTable;
import com.hazelcast.shaded.org.apache.calcite.schema.SchemaPlus;
import com.hazelcast.shaded.org.apache.calcite.schema.Table;
import com.hazelcast.shaded.org.apache.calcite.schema.TableFunction;
import com.hazelcast.shaded.org.apache.calcite.util.BuiltInMethod;

import com.hazelcast.shaded.org.checkerframework.checker.nullness.qual.Nullable;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.lang.reflect.Type;
import java.util.Arrays;
import java.util.List;

import static com.hazelcast.shaded.org.apache.calcite.util.Static.RESOURCE;

import static java.util.Objects.requireNonNull;

/**
 * Implementation of {@link com.hazelcast.shaded.org.apache.calcite.schema.TableFunction} based on a
 * method.
*/
public class TableFunctionImpl extends ReflectiveFunctionBase
    implements TableFunction, ImplementableFunction {
  private final CallImplementor implementor;

  /** Private constructor; use {@link #create}. */
  private TableFunctionImpl(Method method, CallImplementor implementor) {
    super(method);
    this.implementor = implementor;
  }

  /** Creates a {@link TableFunctionImpl} from a class, looking for an "eval"
   * method. Returns null if there is no such method. */
  public static @Nullable TableFunction create(Class<?> clazz) {
    return create(clazz, "eval");
  }

  /** Creates a {@link TableFunctionImpl} from a class, looking for a method
   * with a given name. Returns null if there is no such method. */
  public static @Nullable TableFunction create(Class<?> clazz, String methodName) {
    final Method method = findMethod(clazz, methodName);
    if (method == null) {
      return null;
    }
    return create(method);
  }

  /** Creates a {@link TableFunctionImpl} from a method. */
  public static @Nullable TableFunction create(final Method method) {
    if (!Modifier.isStatic(method.getModifiers())) {
      Class clazz = method.getDeclaringClass();
      if (!classHasPublicZeroArgsConstructor(clazz)) {
        throw RESOURCE.requireDefaultConstructor(clazz.getName()).ex();
      }
    }
    final Class<?> returnType = method.getReturnType();
    if (!QueryableTable.class.isAssignableFrom(returnType)
        && !ScannableTable.class.isAssignableFrom(returnType)) {
      return null;
    }
    CallImplementor implementor = createImplementor(method);
    return new TableFunctionImpl(method, implementor);
  }

  @Override public RelDataType getRowType(RelDataTypeFactory typeFactory,
      List<? extends @Nullable Object> arguments) {
    return apply(arguments).getRowType(typeFactory);
  }

  @Override public Type getElementType(List<? extends @Nullable Object> arguments) {
    final Table table = apply(arguments);
    if (table instanceof QueryableTable) {
      QueryableTable queryableTable = (QueryableTable) table;
      return queryableTable.getElementType();
    } else if (table instanceof ScannableTable) {
      return Object[].class;
    }
    throw new AssertionError("Invalid table class: " + table + " "
        + table.getClass());
  }

  @Override public CallImplementor getImplementor() {
    return implementor;
  }

  private static CallImplementor createImplementor(final Method method) {
    return RexImpTable.createImplementor(
        new ReflectiveCallNotNullImplementor(method) {
          @Override public Expression implement(RexToLixTranslator translator,
              RexCall call, List<Expression> translatedOperands) {
            Expression expr =
                super.implement(translator, call, translatedOperands);
            final Class<?> returnType = method.getReturnType();
            if (QueryableTable.class.isAssignableFrom(returnType)) {
              Expression queryable =
                  Expressions.call(
                      Expressions.convert_(expr, QueryableTable.class),
                      BuiltInMethod.QUERYABLE_TABLE_AS_QUERYABLE.method,
                      Expressions.call(translator.getRoot(),
                          BuiltInMethod.DATA_CONTEXT_GET_QUERY_PROVIDER.method),
                      Expressions.constant(null, SchemaPlus.class),
                      Expressions.constant(call.getOperator().getName(),
                          String.class));
              expr =
                  Expressions.call(queryable,
                      BuiltInMethod.QUERYABLE_AS_ENUMERABLE.method);
            } else {
              expr =
                  Expressions.call(expr,
                      BuiltInMethod.SCANNABLE_TABLE_SCAN.method,
                      translator.getRoot());
            }
            return expr;
          }
        }, NullPolicy.NONE, false);
  }

  private Table apply(List<? extends @Nullable Object> arguments) {
    try {
      Object o = null;
      if (!Modifier.isStatic(method.getModifiers())) {
        final Constructor<?> constructor =
            method.getDeclaringClass().getConstructor();
        o = constructor.newInstance();
      }
      final Object table = method.invoke(o, arguments.toArray());
      return (Table) requireNonNull(table,
          () -> "got null from " + method + " with arguments " + arguments);
    } catch (IllegalArgumentException e) {
      throw RESOURCE.illegalArgumentForTableFunctionCall(
          method.toString(),
          Arrays.toString(method.getParameterTypes()),
          arguments.toString()).ex(e);
    } catch (IllegalAccessException | InvocationTargetException
        | InstantiationException | NoSuchMethodException e) {
      throw new RuntimeException(e);
    }
  }
}
