/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.hazelcast.shaded.org.apache.calcite.rel.rules;

import com.hazelcast.shaded.org.apache.calcite.plan.RelOptRuleCall;
import com.hazelcast.shaded.org.apache.calcite.plan.RelRule;
import com.hazelcast.shaded.org.apache.calcite.rel.RelNode;
import com.hazelcast.shaded.org.apache.calcite.rel.logical.LogicalCalc;
import com.hazelcast.shaded.org.apache.calcite.rel.logical.LogicalProject;
import com.hazelcast.shaded.org.apache.calcite.rex.RexProgram;
import com.hazelcast.shaded.org.apache.calcite.tools.RelBuilderFactory;

import org.immutables.value.Value;

/**
 * Rule to convert a
 * {@link com.hazelcast.shaded.org.apache.calcite.rel.logical.LogicalProject} to a
 * {@link com.hazelcast.shaded.org.apache.calcite.rel.logical.LogicalCalc}.
 *
 * <p>The rule does not fire if the child is a
 * {@link com.hazelcast.shaded.org.apache.calcite.rel.logical.LogicalProject},
 * {@link com.hazelcast.shaded.org.apache.calcite.rel.logical.LogicalFilter} or
 * {@link com.hazelcast.shaded.org.apache.calcite.rel.logical.LogicalCalc}. If it did, then the same
 * {@link com.hazelcast.shaded.org.apache.calcite.rel.logical.LogicalCalc} would be formed via
 * several transformation paths, which is a waste of effort.
 *
 * @see FilterToCalcRule
 * @see CoreRules#PROJECT_TO_CALC
 */
@Value.Enclosing
public class ProjectToCalcRule extends RelRule<ProjectToCalcRule.Config>
    implements TransformationRule {

  /** Creates a ProjectToCalcRule. */
  protected ProjectToCalcRule(Config config) {
    super(config);
  }

  @Deprecated // to be removed before 2.0
  public ProjectToCalcRule(RelBuilderFactory relBuilderFactory) {
    this(Config.DEFAULT.withRelBuilderFactory(relBuilderFactory)
        .as(Config.class));
  }

  //~ Methods ----------------------------------------------------------------

  @Override public void onMatch(RelOptRuleCall call) {
    final LogicalProject project = call.rel(0);
    final RelNode input = project.getInput();
    final RexProgram program =
        RexProgram.create(
            input.getRowType(),
            project.getProjects(),
            null,
            project.getRowType(),
            project.getCluster().getRexBuilder());
    final LogicalCalc calc = LogicalCalc.create(input, program);
    call.transformTo(calc);
  }

  /** Rule configuration. */
  @Value.Immutable
  public interface Config extends RelRule.Config {
    Config DEFAULT = ImmutableProjectToCalcRule.Config.of()
        .withOperandSupplier(b ->
            b.operand(LogicalProject.class).anyInputs());

    @Override default ProjectToCalcRule toRule() {
      return new ProjectToCalcRule(this);
    }
  }
}
