/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to you under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.hazelcast.shaded.org.apache.calcite.adapter.enumerable;

import com.hazelcast.shaded.org.apache.calcite.plan.Convention;
import com.hazelcast.shaded.org.apache.calcite.plan.RelTraitSet;
import com.hazelcast.shaded.org.apache.calcite.rel.RelNode;
import com.hazelcast.shaded.org.apache.calcite.rel.convert.ConverterRule;
import com.hazelcast.shaded.org.apache.calcite.rel.core.RepeatUnion;
import com.hazelcast.shaded.org.apache.calcite.rel.logical.LogicalRepeatUnion;

/**
 * Rule to convert a {@link LogicalRepeatUnion} into an {@link EnumerableRepeatUnion}.
 * You may provide a custom config to convert other nodes that extend {@link RepeatUnion}.
 *
 * @see EnumerableRules#ENUMERABLE_REPEAT_UNION_RULE
 */
public class EnumerableRepeatUnionRule extends ConverterRule {
  /** Default configuration. */
  public static final Config DEFAULT_CONFIG = Config.INSTANCE
      .withConversion(LogicalRepeatUnion.class, Convention.NONE,
          EnumerableConvention.INSTANCE, "EnumerableRepeatUnionRule")
      .withRuleFactory(EnumerableRepeatUnionRule::new);

  /** Called from the Config. */
  protected EnumerableRepeatUnionRule(Config config) {
    super(config);
  }

  @Override public RelNode convert(RelNode rel) {
    RepeatUnion union = (RepeatUnion) rel;
    EnumerableConvention out = EnumerableConvention.INSTANCE;
    RelTraitSet traitSet = union.getTraitSet().replace(out);
    RelNode seedRel = union.getSeedRel();
    RelNode iterativeRel = union.getIterativeRel();

    return new EnumerableRepeatUnion(
        rel.getCluster(),
        traitSet,
        convert(seedRel, seedRel.getTraitSet().replace(out)),
        convert(iterativeRel, iterativeRel.getTraitSet().replace(out)),
        union.all,
        union.iterationLimit,
        union.getTransientTable());
  }
}
