/*
 * Decompiled with CFR 0.152.
 */
package com.hazelcast.client.nearcache;

import com.hazelcast.client.nearcache.ClientNearCacheType;
import com.hazelcast.client.spi.ClientContext;
import com.hazelcast.client.spi.EventHandler;
import com.hazelcast.config.InMemoryFormat;
import com.hazelcast.config.NearCacheConfig;
import com.hazelcast.logging.Logger;
import com.hazelcast.map.client.MapAddEntryListenerRequest;
import com.hazelcast.map.client.MapRemoveEntryListenerRequest;
import com.hazelcast.monitor.impl.NearCacheStatsImpl;
import com.hazelcast.nio.serialization.Data;
import com.hazelcast.spi.impl.PortableEntryEvent;
import com.hazelcast.util.Clock;
import com.hazelcast.util.ExceptionUtil;
import java.util.Collection;
import java.util.Comparator;
import java.util.Map;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

public class ClientNearCache<K> {
    public static final Object NULL_OBJECT = new Object();
    private static final double EVICTION_PERCENTAGE = 0.2;
    private final ClientNearCacheType cacheType;
    private final int maxSize;
    private volatile long lastCleanup;
    private final long maxIdleMillis;
    private final long timeToLiveMillis;
    private final EvictionPolicy evictionPolicy;
    private final InMemoryFormat inMemoryFormat;
    private final String mapName;
    private final ClientContext context;
    private final AtomicBoolean canCleanUp;
    private final AtomicBoolean canEvict;
    private final ConcurrentMap<K, CacheRecord<K>> cache;
    private final NearCacheStatsImpl clientNearCacheStats;
    private String registrationId;
    private final Comparator<CacheRecord<K>> comparator = new Comparator<CacheRecord<K>>(){

        @Override
        public int compare(CacheRecord<K> o1, CacheRecord<K> o2) {
            if (EvictionPolicy.LRU.equals((Object)ClientNearCache.this.evictionPolicy)) {
                return Long.valueOf(o1.lastAccessTime).compareTo(o2.lastAccessTime);
            }
            if (EvictionPolicy.LFU.equals((Object)ClientNearCache.this.evictionPolicy)) {
                return Integer.valueOf(o1.hit.get()).compareTo(o2.hit.get());
            }
            return 0;
        }
    };

    public ClientNearCache(String mapName, ClientNearCacheType cacheType, ClientContext context, NearCacheConfig nearCacheConfig) {
        this.mapName = mapName;
        this.cacheType = cacheType;
        this.context = context;
        this.maxSize = nearCacheConfig.getMaxSize();
        this.maxIdleMillis = TimeUnit.SECONDS.toMillis(nearCacheConfig.getMaxIdleSeconds());
        this.inMemoryFormat = nearCacheConfig.getInMemoryFormat();
        this.timeToLiveMillis = TimeUnit.SECONDS.toMillis(nearCacheConfig.getTimeToLiveSeconds());
        boolean invalidateOnChange = nearCacheConfig.isInvalidateOnChange();
        this.evictionPolicy = EvictionPolicy.valueOf(nearCacheConfig.getEvictionPolicy());
        this.cache = new ConcurrentHashMap<K, CacheRecord<K>>();
        this.canCleanUp = new AtomicBoolean(true);
        this.canEvict = new AtomicBoolean(true);
        this.lastCleanup = Clock.currentTimeMillis();
        this.clientNearCacheStats = new NearCacheStatsImpl();
        if (invalidateOnChange) {
            this.addInvalidateListener();
        }
    }

    private void addInvalidateListener() {
        try {
            if (this.cacheType != ClientNearCacheType.Map) {
                throw new IllegalStateException("Near cache is not available for this type of data structure");
            }
            MapAddEntryListenerRequest request = new MapAddEntryListenerRequest(this.mapName, false);
            EventHandler<PortableEntryEvent> handler = new EventHandler<PortableEntryEvent>(){

                @Override
                public void handle(PortableEntryEvent event) {
                    switch (event.getEventType()) {
                        case ADDED: 
                        case REMOVED: 
                        case UPDATED: 
                        case EVICTED: {
                            Data key = event.getKey();
                            ClientNearCache.this.cache.remove(key);
                            break;
                        }
                        case CLEAR_ALL: 
                        case EVICT_ALL: {
                            ClientNearCache.this.cache.clear();
                            break;
                        }
                        default: {
                            throw new IllegalArgumentException("Not a known event type " + (Object)((Object)event.getEventType()));
                        }
                    }
                }

                @Override
                public void beforeListenerRegister() {
                    ClientNearCache.this.cache.clear();
                }

                @Override
                public void onListenerRegister() {
                    ClientNearCache.this.cache.clear();
                }
            };
            this.registrationId = this.context.getListenerService().listen(request, null, handler);
        }
        catch (Exception e) {
            Logger.getLogger(ClientNearCache.class).severe("-----------------\n Near Cache is not initialized!!! \n-----------------", e);
        }
    }

    public void put(K key, Object object) {
        this.fireTtlCleanup();
        if (this.evictionPolicy == EvictionPolicy.NONE && this.cache.size() >= this.maxSize) {
            return;
        }
        if (this.evictionPolicy != EvictionPolicy.NONE && this.cache.size() >= this.maxSize) {
            this.fireEvictCache();
        }
        Object value = object == null ? NULL_OBJECT : (this.inMemoryFormat.equals((Object)InMemoryFormat.BINARY) ? this.context.getSerializationService().toData(object) : object);
        this.cache.put(key, new CacheRecord<K>(key, value));
    }

    private void fireEvictCache() {
        if (this.canEvict.compareAndSet(true, false)) {
            try {
                this.context.getExecutionService().execute(new Runnable(){

                    /*
                     * WARNING - Removed try catching itself - possible behaviour change.
                     */
                    @Override
                    public void run() {
                        try {
                            TreeSet records = new TreeSet(ClientNearCache.this.comparator);
                            records.addAll(ClientNearCache.this.cache.values());
                            int evictSize = (int)(0.2 * (double)ClientNearCache.this.cache.size());
                            int i = 0;
                            for (CacheRecord record : records) {
                                ClientNearCache.this.cache.remove(record.key);
                                if (++i <= evictSize) continue;
                                break;
                            }
                        }
                        finally {
                            ClientNearCache.this.canEvict.set(true);
                        }
                    }
                });
            }
            catch (RejectedExecutionException e) {
                this.canEvict.set(true);
            }
            catch (Exception e) {
                throw ExceptionUtil.rethrow(e);
            }
        }
    }

    private void fireTtlCleanup() {
        if (Clock.currentTimeMillis() < this.lastCleanup + this.timeToLiveMillis) {
            return;
        }
        if (this.canCleanUp.compareAndSet(true, false)) {
            try {
                this.context.getExecutionService().execute(new Runnable(){

                    /*
                     * WARNING - Removed try catching itself - possible behaviour change.
                     */
                    @Override
                    public void run() {
                        try {
                            ClientNearCache.this.lastCleanup = Clock.currentTimeMillis();
                            for (Map.Entry entry : ClientNearCache.this.cache.entrySet()) {
                                if (!((CacheRecord)entry.getValue()).expired()) continue;
                                ClientNearCache.this.cache.remove(entry.getKey());
                            }
                        }
                        finally {
                            ClientNearCache.this.canCleanUp.set(true);
                        }
                    }
                });
            }
            catch (RejectedExecutionException e) {
                this.canCleanUp.set(true);
            }
            catch (Exception e) {
                throw ExceptionUtil.rethrow(e);
            }
        }
    }

    public void invalidate() {
        this.cache.clear();
    }

    public void invalidate(K key) {
        this.cache.remove(key);
    }

    public void invalidate(Collection<K> keys) {
        if (keys == null || keys.isEmpty()) {
            return;
        }
        for (K key : keys) {
            this.cache.remove(key);
        }
    }

    public Object get(K key) {
        this.fireTtlCleanup();
        CacheRecord record = (CacheRecord)this.cache.get(key);
        if (record != null) {
            if (record.expired()) {
                this.cache.remove(key);
                this.clientNearCacheStats.incrementMisses();
                return null;
            }
            if (record.value.equals(NULL_OBJECT)) {
                this.clientNearCacheStats.incrementMisses();
                return NULL_OBJECT;
            }
            record.access();
            return this.inMemoryFormat.equals((Object)InMemoryFormat.BINARY) ? this.context.getSerializationService().toObject((Data)record.value) : record.value;
        }
        this.clientNearCacheStats.incrementMisses();
        return null;
    }

    public NearCacheStatsImpl getNearCacheStats() {
        return this.createNearCacheStats();
    }

    private NearCacheStatsImpl createNearCacheStats() {
        long ownedEntryCount = this.cache.values().size();
        long ownedEntryMemory = 0L;
        for (CacheRecord record : this.cache.values()) {
            ownedEntryMemory += record.getCost();
        }
        this.clientNearCacheStats.setOwnedEntryCount(ownedEntryCount);
        this.clientNearCacheStats.setOwnedEntryMemoryCost(ownedEntryMemory);
        return this.clientNearCacheStats;
    }

    public void destroy() {
        if (this.registrationId != null) {
            if (this.cacheType != ClientNearCacheType.Map) {
                throw new IllegalStateException("Near cache is not available for this type of data structure");
            }
            MapRemoveEntryListenerRequest request = new MapRemoveEntryListenerRequest(this.mapName, this.registrationId);
            this.context.getListenerService().stopListening(request, this.registrationId);
        }
        this.cache.clear();
    }

    public void clear() {
        this.cache.clear();
    }

    class CacheRecord<K> {
        final K key;
        final Object value;
        volatile long lastAccessTime;
        final long creationTime;
        final AtomicInteger hit;

        CacheRecord(K key, Object value) {
            long time;
            this.key = key;
            this.value = value;
            this.lastAccessTime = time = Clock.currentTimeMillis();
            this.creationTime = time;
            this.hit = new AtomicInteger(0);
        }

        void access() {
            this.hit.incrementAndGet();
            ClientNearCache.this.clientNearCacheStats.incrementHits();
            this.lastAccessTime = Clock.currentTimeMillis();
        }

        public long getCost() {
            if (!(this.value instanceof Data)) {
                return 0L;
            }
            if (!(this.key instanceof Data)) {
                return 0L;
            }
            return ((Data)this.key).getHeapCost() + ((Data)this.value).getHeapCost() + 16 + 4 + 12;
        }

        boolean expired() {
            long time = Clock.currentTimeMillis();
            return ClientNearCache.this.maxIdleMillis > 0L && time > this.lastAccessTime + ClientNearCache.this.maxIdleMillis || ClientNearCache.this.timeToLiveMillis > 0L && time > this.creationTime + ClientNearCache.this.timeToLiveMillis;
        }
    }

    static enum EvictionPolicy {
        NONE,
        LRU,
        LFU;

    }
}

