/*
 * Decompiled with CFR 0.152.
 */
package com.hazelcast.map.eviction;

import com.hazelcast.config.MapConfig;
import com.hazelcast.config.MaxSizeConfig;
import com.hazelcast.core.EntryEventType;
import com.hazelcast.map.MapContainer;
import com.hazelcast.map.MapEventPublisher;
import com.hazelcast.map.MapServiceContext;
import com.hazelcast.map.NearCacheProvider;
import com.hazelcast.map.PartitionContainer;
import com.hazelcast.map.RecordStore;
import com.hazelcast.map.eviction.MemoryInfoAccessor;
import com.hazelcast.map.eviction.RuntimeMemoryInfoAccessor;
import com.hazelcast.map.record.Record;
import com.hazelcast.nio.Address;
import com.hazelcast.nio.serialization.Data;
import com.hazelcast.partition.InternalPartition;
import com.hazelcast.partition.InternalPartitionService;
import com.hazelcast.spi.EventService;
import com.hazelcast.spi.NodeEngine;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

public final class EvictionOperator {
    private static final int ONE_HUNDRED_PERCENT = 100;
    private static final int EVICTION_START_THRESHOLD_PERCENTAGE = 95;
    private static final int ONE_KILOBYTE = 1024;
    private static final int ONE_MEGABYTE = 0x100000;
    private MemoryInfoAccessor memoryInfoAccessor;
    private MapServiceContext mapServiceContext;

    private EvictionOperator() {
    }

    public static EvictionOperator create(MapServiceContext mapServiceContext) {
        EvictionOperator evictionOperator = new EvictionOperator();
        RuntimeMemoryInfoAccessor memoryInfoAccessor = new RuntimeMemoryInfoAccessor();
        evictionOperator.setMemoryInfoAccessor(memoryInfoAccessor);
        evictionOperator.setMapServiceContext(mapServiceContext);
        return evictionOperator;
    }

    public static EvictionOperator create(MemoryInfoAccessor memoryInfoAccessor, MapServiceContext mapServiceContext) {
        EvictionOperator evictionOperator = new EvictionOperator();
        evictionOperator.setMemoryInfoAccessor(memoryInfoAccessor);
        evictionOperator.setMapServiceContext(mapServiceContext);
        return evictionOperator;
    }

    public void setMemoryInfoAccessor(MemoryInfoAccessor memoryInfoAccessor) {
        this.memoryInfoAccessor = memoryInfoAccessor;
    }

    public void setMapServiceContext(MapServiceContext mapServiceContext) {
        this.mapServiceContext = mapServiceContext;
    }

    public boolean checkEvictable(MapContainer mapContainer, int partitionId) {
        boolean result;
        MaxSizeConfig maxSizeConfig = mapContainer.getMapConfig().getMaxSizeConfig();
        MaxSizeConfig.MaxSizePolicy maxSizePolicy = maxSizeConfig.getMaxSizePolicy();
        switch (maxSizePolicy) {
            case PER_NODE: {
                result = this.isEvictablePerNode(mapContainer);
                break;
            }
            case PER_PARTITION: {
                result = this.isEvictablePerPartition(mapContainer, partitionId);
                break;
            }
            case USED_HEAP_PERCENTAGE: {
                result = this.isEvictableHeapPercentage(mapContainer);
                break;
            }
            case USED_HEAP_SIZE: {
                result = this.isEvictableHeapSize(mapContainer);
                break;
            }
            default: {
                throw new IllegalArgumentException("Not an appropriate max size policy [" + (Object)((Object)maxSizePolicy) + ']');
            }
        }
        return result;
    }

    public void removeEvictableRecords(RecordStore recordStore, int evictableSize, MapConfig mapConfig, boolean backup) {
        MapConfig.EvictionPolicy evictionPolicy = mapConfig.getEvictionPolicy();
        long[] criterias = this.createAndPopulateEvictionCriteriaArray(recordStore, evictionPolicy);
        if (criterias == null) {
            return;
        }
        Arrays.sort(criterias);
        int evictableBaseIndex = this.getEvictionStartIndex(criterias, evictableSize);
        long criteriaValue = criterias[evictableBaseIndex];
        int evictedRecordCounter = 0;
        Iterator<Record> iterator = recordStore.iterator();
        while (iterator.hasNext()) {
            Record record = iterator.next();
            long value = this.getEvictionCriteriaValue(record, evictionPolicy);
            if (value <= criteriaValue) {
                Data tmpKey = record.getKey();
                Object tmpValue = record.getValue();
                if (this.evictIfNotLocked(tmpKey, recordStore, backup)) {
                    ++evictedRecordCounter;
                    String mapName = mapConfig.getName();
                    if (!backup) {
                        this.interceptAndInvalidate(this.mapServiceContext, value, tmpKey, mapName);
                        this.fireEvent(tmpKey, tmpValue, mapName, this.mapServiceContext);
                    }
                }
            }
            if (evictedRecordCounter < evictableSize) continue;
            break;
        }
    }

    private long[] createAndPopulateEvictionCriteriaArray(RecordStore recordStore, MapConfig.EvictionPolicy evictionPolicy) {
        int size = recordStore.size();
        long[] criterias = null;
        int index = 0;
        Iterator<Record> iterator = recordStore.iterator();
        while (iterator.hasNext()) {
            Record record = iterator.next();
            if (criterias == null) {
                criterias = new long[size];
            }
            criterias[index] = this.getEvictionCriteriaValue(record, evictionPolicy);
            if (++index != size) continue;
            break;
        }
        if (criterias == null) {
            return null;
        }
        if (index < criterias.length) {
            for (int i = index; i < criterias.length; ++i) {
                criterias[i] = Long.MAX_VALUE;
            }
        }
        return criterias;
    }

    private int getEvictionStartIndex(long[] criterias, int evictableSize) {
        int length = criterias.length;
        int sizeToEvict = Math.min(evictableSize, length);
        int index = sizeToEvict - 1;
        return index < 0 ? 0 : index;
    }

    private void interceptAndInvalidate(MapServiceContext mapServiceContext, long value, Data tmpKey, String mapName) {
        mapServiceContext.interceptAfterRemove(mapName, value);
        NearCacheProvider nearCacheProvider = mapServiceContext.getNearCacheProvider();
        if (nearCacheProvider.isNearCacheAndInvalidationEnabled(mapName)) {
            nearCacheProvider.invalidateAllNearCaches(mapName, tmpKey);
        }
    }

    public void fireEvent(Data key, Object value, String mapName, MapServiceContext mapServiceContext) {
        if (!this.hasListener(mapName)) {
            return;
        }
        MapEventPublisher mapEventPublisher = mapServiceContext.getMapEventPublisher();
        NodeEngine nodeEngine = mapServiceContext.getNodeEngine();
        Address thisAddress = nodeEngine.getThisAddress();
        Data dataValue = mapServiceContext.toData(value);
        mapEventPublisher.publishEvent(thisAddress, mapName, EntryEventType.EVICTED, key, dataValue, null);
    }

    private boolean hasListener(String mapName) {
        String serviceName = this.mapServiceContext.serviceName();
        EventService eventService = this.mapServiceContext.getNodeEngine().getEventService();
        return eventService.hasEventRegistration(serviceName, mapName);
    }

    private boolean evictIfNotLocked(Data key, RecordStore recordStore, boolean backup) {
        if (recordStore.isLocked(key)) {
            return false;
        }
        recordStore.evict(key, backup);
        return true;
    }

    public int evictableSize(int currentPartitionSize, MapConfig mapConfig) {
        int evictableSize;
        MaxSizeConfig.MaxSizePolicy maxSizePolicy = mapConfig.getMaxSizeConfig().getMaxSizePolicy();
        int evictionPercentage = mapConfig.getEvictionPercentage();
        switch (maxSizePolicy) {
            case PER_PARTITION: {
                int maxSize = mapConfig.getMaxSizeConfig().getSize();
                int targetSizePerPartition = Double.valueOf((double)maxSize * ((double)(100 - evictionPercentage) / 100.0)).intValue();
                int diffFromTargetSize = currentPartitionSize - targetSizePerPartition;
                int prunedSize = currentPartitionSize * evictionPercentage / 100 + 1;
                evictableSize = Math.max(diffFromTargetSize, prunedSize);
                break;
            }
            case PER_NODE: {
                int maxSize = mapConfig.getMaxSizeConfig().getSize();
                int memberCount = this.mapServiceContext.getNodeEngine().getClusterService().getMembers().size();
                int maxPartitionSize = maxSize * memberCount / this.mapServiceContext.getNodeEngine().getPartitionService().getPartitionCount();
                int targetSizePerPartition = Double.valueOf((double)maxPartitionSize * ((double)(100 - evictionPercentage) / 100.0)).intValue();
                int diffFromTargetSize = currentPartitionSize - targetSizePerPartition;
                int prunedSize = currentPartitionSize * evictionPercentage / 100 + 1;
                evictableSize = Math.max(diffFromTargetSize, prunedSize);
                break;
            }
            case USED_HEAP_PERCENTAGE: 
            case USED_HEAP_SIZE: {
                evictableSize = Math.max(currentPartitionSize * evictionPercentage / 100, 1);
                break;
            }
            default: {
                throw new IllegalArgumentException("Max size policy is not defined [" + (Object)((Object)maxSizePolicy) + "]");
            }
        }
        return evictableSize;
    }

    private long getEvictionCriteriaValue(Record record, MapConfig.EvictionPolicy evictionPolicy) {
        long value;
        switch (evictionPolicy) {
            case LRU: 
            case LFU: {
                value = record.getEvictionCriteriaNumber();
                break;
            }
            default: {
                throw new IllegalArgumentException("Not an appropriate eviction policy [" + (Object)((Object)evictionPolicy) + ']');
            }
        }
        return value;
    }

    private int getRecordStoreSize(String mapName, PartitionContainer partitionContainer) {
        RecordStore existingRecordStore = partitionContainer.getExistingRecordStore(mapName);
        if (existingRecordStore == null) {
            return 0;
        }
        return existingRecordStore.size();
    }

    private long getRecordStoreHeapCost(String mapName, PartitionContainer partitionContainer) {
        RecordStore existingRecordStore = partitionContainer.getExistingRecordStore(mapName);
        if (existingRecordStore == null) {
            return 0L;
        }
        return existingRecordStore.getHeapCost();
    }

    private int getApproximateMaxSize(int maxSizeFromConfig) {
        return maxSizeFromConfig * 95 / 100;
    }

    private boolean isEvictablePerNode(MapContainer mapContainer) {
        int nodeTotalSize = 0;
        MaxSizeConfig maxSizeConfig = mapContainer.getMapConfig().getMaxSizeConfig();
        int maxSize = this.getApproximateMaxSize(maxSizeConfig.getSize());
        String mapName = mapContainer.getName();
        MapServiceContext mapServiceContext = mapContainer.getMapServiceContext();
        List<Integer> partitionIds = this.findPartitionIds();
        for (int partitionId : partitionIds) {
            PartitionContainer container = mapServiceContext.getPartitionContainer(partitionId);
            if (container == null || (nodeTotalSize += this.getRecordStoreSize(mapName, container)) < maxSize) continue;
            return true;
        }
        return false;
    }

    private boolean isEvictablePerPartition(MapContainer mapContainer, int partitionId) {
        MapServiceContext mapServiceContext = mapContainer.getMapServiceContext();
        MaxSizeConfig maxSizeConfig = mapContainer.getMapConfig().getMaxSizeConfig();
        int maxSize = this.getApproximateMaxSize(maxSizeConfig.getSize());
        String mapName = mapContainer.getName();
        PartitionContainer container = mapServiceContext.getPartitionContainer(partitionId);
        if (container == null) {
            return false;
        }
        int size = this.getRecordStoreSize(mapName, container);
        return size >= maxSize;
    }

    private boolean isEvictableHeapSize(MapContainer mapContainer) {
        long usedHeapSize = this.getUsedHeapSize(mapContainer);
        if (usedHeapSize == -1L) {
            return false;
        }
        MaxSizeConfig maxSizeConfig = mapContainer.getMapConfig().getMaxSizeConfig();
        int maxSize = this.getApproximateMaxSize(maxSizeConfig.getSize());
        return (long)maxSize < usedHeapSize / 0x100000L;
    }

    private boolean isEvictableFreeHeapSize(MapContainer mapContainer) {
        long currentFreeHeapSize = this.getAvailableMemory();
        MaxSizeConfig maxSizeConfig = mapContainer.getMapConfig().getMaxSizeConfig();
        int minFreeHeapSize = this.getApproximateMaxSize(maxSizeConfig.getSize());
        return (long)minFreeHeapSize > currentFreeHeapSize / 0x100000L;
    }

    private boolean isEvictableHeapPercentage(MapContainer mapContainer) {
        long total;
        long usedHeapSize = this.getUsedHeapSize(mapContainer);
        if (usedHeapSize == -1L) {
            return false;
        }
        MaxSizeConfig maxSizeConfig = mapContainer.getMapConfig().getMaxSizeConfig();
        int maxSize = this.getApproximateMaxSize(maxSizeConfig.getSize());
        return (double)maxSize < 100.0 * (double)usedHeapSize / (double)(total = this.getTotalMemory());
    }

    private boolean isEvictableFreeHeapPercentage(MapContainer mapContainer) {
        long total;
        long currentFreeHeapSize = this.getAvailableMemory();
        MaxSizeConfig maxSizeConfig = mapContainer.getMapConfig().getMaxSizeConfig();
        int freeHeapPercentage = this.getApproximateMaxSize(maxSizeConfig.getSize());
        return (double)freeHeapPercentage > 100.0 * (double)currentFreeHeapSize / (double)(total = this.getTotalMemory());
    }

    private List<Integer> findPartitionIds() {
        NodeEngine nodeEngine = this.mapServiceContext.getNodeEngine();
        InternalPartitionService partitionService = nodeEngine.getPartitionService();
        int partitionCount = partitionService.getPartitionCount();
        ArrayList<Integer> partitionIds = null;
        for (int partitionId = 0; partitionId < partitionCount; ++partitionId) {
            if (!this.isOwnerOrBackup(partitionId)) continue;
            if (partitionIds == null) {
                partitionIds = new ArrayList<Integer>();
            }
            partitionIds.add(partitionId);
        }
        return partitionIds == null ? Collections.emptyList() : partitionIds;
    }

    private boolean isOwnerOrBackup(int partitionId) {
        NodeEngine nodeEngine = this.mapServiceContext.getNodeEngine();
        InternalPartitionService partitionService = nodeEngine.getPartitionService();
        InternalPartition partition = partitionService.getPartition(partitionId, false);
        Address thisAddress = nodeEngine.getThisAddress();
        return partition.isOwnerOrBackup(thisAddress);
    }

    private long getTotalMemory() {
        return this.memoryInfoAccessor.getTotalMemory();
    }

    private long getFreeMemory() {
        return this.memoryInfoAccessor.getFreeMemory();
    }

    private long getMaxMemory() {
        return this.memoryInfoAccessor.getMaxMemory();
    }

    private long getAvailableMemory() {
        long totalMemory = this.getTotalMemory();
        long freeMemory = this.getFreeMemory();
        long maxMemory = this.getMaxMemory();
        return freeMemory + (maxMemory - totalMemory);
    }

    private long getUsedHeapSize(MapContainer mapContainer) {
        long heapCost = 0L;
        String mapName = mapContainer.getName();
        MapServiceContext mapServiceContext = mapContainer.getMapServiceContext();
        List<Integer> partitionIds = this.findPartitionIds();
        for (int partitionId : partitionIds) {
            PartitionContainer container = mapServiceContext.getPartitionContainer(partitionId);
            if (container == null) continue;
            heapCost += this.getRecordStoreHeapCost(mapName, container);
        }
        return heapCost += mapContainer.getNearCacheSizeEstimator().getSize();
    }
}

