/*
 * Decompiled with CFR 0.152.
 */
package com.hazelcast.cluster;

import com.hazelcast.impl.ClusterOperation;
import com.hazelcast.impl.Constants;
import com.hazelcast.impl.MemberImpl;
import com.hazelcast.impl.Node;
import com.hazelcast.impl.Processable;
import com.hazelcast.impl.ThreadContext;
import com.hazelcast.impl.base.PacketProcessor;
import com.hazelcast.impl.base.SystemLogService;
import com.hazelcast.logging.ILogger;
import com.hazelcast.nio.Packet;
import com.hazelcast.util.Clock;
import com.hazelcast.util.ThreadWatcher;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.LockSupport;
import java.util.logging.Level;

public final class ClusterService
implements Runnable,
Constants {
    private static final int PACKET_BULK_SIZE = 64;
    private static final int PROCESSABLE_BULK_SIZE = 64;
    private final ILogger logger;
    private final long PERIODIC_CHECK_INTERVAL_MILLIS = TimeUnit.SECONDS.toMillis(1L);
    private final long MAX_IDLE_MILLIS;
    private final boolean RESTART_ON_MAX_IDLE;
    private final Queue<Packet> packetQueue = new ConcurrentLinkedQueue<Packet>();
    private final Queue<Processable> processableQueue = new ConcurrentLinkedQueue<Processable>();
    private final PacketProcessor[] packetProcessors = new PacketProcessor[500];
    private final Runnable[] periodicRunnables = new Runnable[5];
    private final Node node;
    private long lastPeriodicCheck = 0L;
    private long lastCheck = 0L;
    private volatile boolean running = true;
    private final ThreadWatcher threadWatcher = new ThreadWatcher();
    private final Thread serviceThread;

    public ClusterService(Node node) {
        this.node = node;
        this.logger = node.getLogger(ClusterService.class.getName());
        this.MAX_IDLE_MILLIS = (long)node.groupProperties.MAX_NO_HEARTBEAT_SECONDS.getInteger() * 1000L;
        this.RESTART_ON_MAX_IDLE = node.groupProperties.RESTART_ON_MAX_IDLE.getBoolean();
        this.serviceThread = new Thread(node.threadGroup, this, node.getThreadNamePrefix("ServiceThread"));
    }

    public Thread getServiceThread() {
        return this.serviceThread;
    }

    public void registerPeriodicRunnable(Runnable runnable) {
        int len = this.periodicRunnables.length;
        for (int i = 0; i < len; ++i) {
            if (this.periodicRunnables[i] != null) continue;
            this.periodicRunnables[i] = runnable;
            return;
        }
        throw new RuntimeException("Not enough space for a runnable " + runnable);
    }

    public void registerPacketProcessor(ClusterOperation operation, PacketProcessor packetProcessor) {
        PacketProcessor processor = this.packetProcessors[operation.getValue()];
        if (processor != null) {
            this.logger.log(Level.SEVERE, (Object)((Object)operation) + " is registered already with " + processor);
        }
        this.packetProcessors[operation.getValue()] = packetProcessor;
    }

    public PacketProcessor getPacketProcessor(ClusterOperation operation) {
        PacketProcessor packetProcessor = this.packetProcessors[operation.getValue()];
        if (packetProcessor == null) {
            this.logger.log(Level.SEVERE, (Object)((Object)operation) + " has no registered processor!");
        }
        return packetProcessor;
    }

    public void enqueuePacket(Packet packet) {
        if (packet.callId != -1L) {
            SystemLogService css = this.node.getSystemLogService();
            packet.callState = css.getOrCreateCallState(packet.callId, packet.lockAddress, packet.threadId);
            if (css.shouldLog(SystemLogService.Level.CS_INFO)) {
                css.info(packet, "Enqueue Packet ", (Object)packet.operation);
            }
        }
        this.packetQueue.offer(packet);
        this.unpark();
    }

    public boolean enqueueAndWait(final Processable processable, int seconds) {
        try {
            final CountDownLatch l = new CountDownLatch(1);
            this.enqueueAndReturn(new Processable(){

                public void process() {
                    processable.process();
                    l.countDown();
                }
            });
            this.node.checkNodeState();
            return l.await(seconds, TimeUnit.SECONDS);
        }
        catch (InterruptedException interruptedException) {
            return false;
        }
    }

    public void enqueueAndWait(final Processable processable) {
        try {
            final CountDownLatch l = new CountDownLatch(1);
            this.enqueueAndReturn(new Processable(){

                public void process() {
                    processable.process();
                    l.countDown();
                }
            });
            this.node.checkNodeState();
            l.await();
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    public void enqueueAndReturn(Processable processable) {
        this.processableQueue.offer(processable);
        this.unpark();
    }

    void unpark() {
        LockSupport.unpark(this.serviceThread);
    }

    private void processPacket(Packet packet) {
        if (!this.running) {
            return;
        }
        MemberImpl memberFrom = this.node.clusterManager.getMember(packet.conn.getEndPoint());
        if (memberFrom != null) {
            memberFrom.didRead();
        }
        if (packet.operation.getValue() < 0 || packet.operation.getValue() > 500) {
            String msg = "Unknown operation " + (Object)((Object)packet.operation);
            this.logger.log(Level.SEVERE, msg);
            throw new RuntimeException(msg);
        }
        PacketProcessor packetProcessor = this.packetProcessors[packet.operation.getValue()];
        if (packetProcessor == null) {
            String msg = "No Packet processor found for operation : " + (Object)((Object)packet.operation) + " from " + packet.conn;
            this.logger.log(Level.SEVERE, msg);
            throw new RuntimeException(msg);
        }
        SystemLogService css = this.node.getSystemLogService();
        if (css.shouldLog(SystemLogService.Level.CS_INFO)) {
            css.logObject(packet, SystemLogService.Level.CS_INFO, "Processing packet");
            css.logObject(packet, SystemLogService.Level.CS_INFO, packetProcessor.getClass());
        }
        packetProcessor.process(packet);
    }

    private void processProcessable(Processable processable) {
        if (!this.running) {
            return;
        }
        processable.process();
    }

    public void run() {
        ThreadContext.get().setCurrentFactory(this.node.factory);
        boolean readPackets = false;
        boolean readProcessables = false;
        while (this.running) {
            try {
                this.threadWatcher.incrementRunCount();
                readPackets = this.dequeuePackets() != 0;
                boolean bl = readProcessables = this.dequeueProcessables() != 0;
                if (readPackets || readProcessables) continue;
                try {
                    long startWait = System.nanoTime();
                    LockSupport.parkNanos(TimeUnit.MILLISECONDS.toNanos(1L));
                    long now = System.nanoTime();
                    this.threadWatcher.addWait(now - startWait, now);
                    this.checkPeriodics();
                }
                catch (Exception e) {
                    this.node.handleInterruptedException(Thread.currentThread(), e);
                }
            }
            catch (OutOfMemoryError e) {
                this.node.onOutOfMemory(e);
            }
            catch (Throwable e) {
                this.logger.log(Level.SEVERE, e.getMessage(), e);
            }
        }
        this.packetQueue.clear();
        this.processableQueue.clear();
    }

    private void publishUtilization() {
        this.node.getCpuUtilization().serviceThread = this.threadWatcher.publish(this.running);
    }

    private int dequeuePackets() throws Throwable {
        Packet packet = null;
        try {
            for (int i = 0; i < 64; ++i) {
                this.checkPeriodics();
                packet = this.packetQueue.poll();
                if (packet == null) {
                    return i;
                }
                this.processPacket(packet);
            }
        }
        catch (OutOfMemoryError e) {
            throw e;
        }
        catch (Throwable e) {
            this.logger.log(Level.SEVERE, "error processing messages  packet=" + packet, e);
            throw e;
        }
        return 64;
    }

    private int dequeueProcessables() throws Throwable {
        Processable processable = null;
        try {
            for (int i = 0; i < 64; ++i) {
                this.checkPeriodics();
                processable = this.processableQueue.poll();
                if (processable == null) {
                    return i;
                }
                this.processProcessable(processable);
            }
        }
        catch (OutOfMemoryError e) {
            throw e;
        }
        catch (Throwable e) {
            this.logger.log(Level.SEVERE, "error processing messages  processable=" + processable, e);
            throw e;
        }
        return 64;
    }

    public void start() {
        this.lastPeriodicCheck = Clock.currentTimeMillis();
        this.lastCheck = Clock.currentTimeMillis();
        this.running = true;
    }

    public void stop() {
        this.packetQueue.clear();
        this.processableQueue.clear();
        try {
            final CountDownLatch stopLatch = new CountDownLatch(1);
            this.processableQueue.offer(new Processable(){

                public void process() {
                    ClusterService.this.node.cleanupServiceThread();
                    ClusterService.this.running = false;
                    stopLatch.countDown();
                }
            });
            stopLatch.await(3L, TimeUnit.SECONDS);
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    public String toString() {
        return "ClusterService packetQueueSize=" + this.packetQueue.size() + "unknownQueueSize=" + this.processableQueue.size() + " isMaster= " + this.node.isMaster() + " isMaster= " + this.node.getMasterAddress();
    }

    private void checkPeriodics() {
        long now = Clock.currentTimeMillis();
        if (this.RESTART_ON_MAX_IDLE && now - this.lastCheck > this.MAX_IDLE_MILLIS) {
            if (this.logger.isLoggable(Level.INFO)) {
                StringBuilder sb = new StringBuilder("Hazelcast ServiceThread is blocked for ");
                sb.append(now - this.lastCheck);
                sb.append(" ms. Restarting Hazelcast!");
                sb.append("\n\tnow:").append(now);
                sb.append("\n\tlastCheck:").append(this.lastCheck);
                sb.append("\n\tmaxIdleMillis:").append(this.MAX_IDLE_MILLIS);
                sb.append("\n\tRESTART_ON_MAX_IDLE:").append(this.RESTART_ON_MAX_IDLE);
                sb.append("\n");
                this.logger.log(Level.INFO, sb.toString());
            }
            new Thread(new Runnable(){

                public void run() {
                    ((ClusterService)ClusterService.this).node.factory.restart();
                }
            }, "hz.RestartThread").start();
        }
        this.lastCheck = now;
        if (now - this.lastPeriodicCheck > this.PERIODIC_CHECK_INTERVAL_MILLIS) {
            this.publishUtilization();
            for (Runnable runnable : this.periodicRunnables) {
                if (runnable == null) continue;
                runnable.run();
            }
            this.lastPeriodicCheck = now;
        }
    }
}

