/*
 * Decompiled with CFR 0.152.
 */
package io.dropwizard.bundles.assets;

import com.google.common.base.CharMatcher;
import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheBuilderSpec;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.cache.Weigher;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Maps;
import com.google.common.hash.Hashing;
import com.google.common.io.Files;
import com.google.common.io.Resources;
import com.google.common.net.MediaType;
import io.dropwizard.bundles.assets.UrlUtil;
import io.dropwizard.servlets.assets.ByteRange;
import io.dropwizard.servlets.assets.ResourceURL;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.List;
import java.util.Map;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.eclipse.jetty.http.MimeTypes;

public class AssetServlet
extends HttpServlet {
    private static final long serialVersionUID = 6393345594784987908L;
    private static final MediaType DEFAULT_MEDIA_TYPE = MediaType.HTML_UTF_8;
    private static final CharMatcher SLASHES = CharMatcher.is('/');
    private final transient CacheBuilderSpec cacheSpec;
    private final transient LoadingCache<String, Asset> cache;
    private final transient MimeTypes mimeTypes;
    private Charset defaultCharset;
    private String cacheControlHeader = null;

    public AssetServlet(Iterable<Map.Entry<String, String>> resourcePathToUriPathMapping, String indexFile, Charset defaultCharset, CacheBuilderSpec spec, Iterable<Map.Entry<String, String>> overrides, Iterable<Map.Entry<String, String>> mimeTypes) {
        this.defaultCharset = defaultCharset;
        AssetLoader loader = new AssetLoader(resourcePathToUriPathMapping, indexFile, overrides);
        CacheBuilder<Object, Object> cacheBuilder = CacheBuilder.from(spec);
        if (spec.toParsableString().contains("maximumWeight=")) {
            cacheBuilder.weigher(new AssetSizeWeigher());
        }
        this.cache = cacheBuilder.build(loader);
        this.cacheSpec = spec;
        this.mimeTypes = new MimeTypes();
        this.setMimeTypes(mimeTypes);
    }

    public void setCacheControlHeader(String cacheControlHeader) {
        this.cacheControlHeader = cacheControlHeader;
    }

    public String getCacheControlHeader() {
        return this.cacheControlHeader;
    }

    public void setMimeTypes(Iterable<Map.Entry<String, String>> mimeTypes) {
        for (Map.Entry<String, String> mime : mimeTypes) {
            this.mimeTypes.addMimeMapping(mime.getKey(), mime.getValue());
        }
    }

    public MimeTypes getMimeTypes() {
        return this.mimeTypes;
    }

    public void setDefaultCharset(Charset defaultCharset) {
        this.defaultCharset = defaultCharset;
    }

    public Charset getDefaultCharset() {
        return this.defaultCharset;
    }

    public CacheBuilderSpec getCacheSpec() {
        return this.cacheSpec;
    }

    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
        try {
            String mimeTypeOfExtension;
            String ifRange;
            Asset cachedAsset;
            StringBuilder builder = new StringBuilder(req.getServletPath());
            if (req.getPathInfo() != null) {
                builder.append(req.getPathInfo());
            }
            if ((cachedAsset = this.cache.getUnchecked(builder.toString())) == null) {
                resp.sendError(404);
                return;
            }
            if (this.isCachedClientSide(req, cachedAsset)) {
                resp.sendError(304);
                return;
            }
            String rangeHeader = req.getHeader("Range");
            int resourceLength = cachedAsset.getResource().length;
            ImmutableList<Object> ranges = ImmutableList.of();
            boolean usingRanges = false;
            if (rangeHeader != null && ((ifRange = req.getHeader("If-Range")) == null || cachedAsset.getETag().equals(ifRange))) {
                try {
                    ranges = this.parseRangeHeader(rangeHeader, resourceLength);
                }
                catch (NumberFormatException e) {
                    resp.sendError(416);
                    return;
                }
                if (ranges.isEmpty()) {
                    resp.sendError(416);
                    return;
                }
                resp.setStatus(206);
                usingRanges = true;
                resp.addHeader("Content-Range", "bytes " + Joiner.on(",").join(ranges) + "/" + resourceLength);
            }
            resp.setDateHeader("Last-Modified", cachedAsset.getLastModifiedTime());
            resp.setHeader("ETag", cachedAsset.getETag());
            if (this.cacheControlHeader != null) {
                resp.setHeader("Cache-Control", this.cacheControlHeader);
            }
            if ((mimeTypeOfExtension = this.mimeTypes.getMimeByExtension(req.getRequestURI())) == null) {
                mimeTypeOfExtension = req.getServletContext().getMimeType(req.getRequestURI());
            }
            MediaType mediaType = DEFAULT_MEDIA_TYPE;
            if (mimeTypeOfExtension != null) {
                try {
                    mediaType = MediaType.parse(mimeTypeOfExtension);
                    if (this.defaultCharset != null && mediaType.is(MediaType.ANY_TEXT_TYPE)) {
                        mediaType = mediaType.withCharset(this.defaultCharset);
                    }
                }
                catch (IllegalArgumentException illegalArgumentException) {
                    // empty catch block
                }
            }
            if (mediaType.is(MediaType.ANY_VIDEO_TYPE) || mediaType.is(MediaType.ANY_AUDIO_TYPE) || usingRanges) {
                resp.addHeader("Accept-Ranges", "bytes");
            }
            resp.setContentType(mediaType.type() + '/' + mediaType.subtype());
            if (mediaType.charset().isPresent()) {
                resp.setCharacterEncoding(mediaType.charset().get().toString());
            }
            try (ServletOutputStream output = resp.getOutputStream();){
                if (usingRanges) {
                    for (ByteRange byteRange : ranges) {
                        output.write(cachedAsset.getResource(), byteRange.getStart(), byteRange.getEnd() - byteRange.getStart() + 1);
                    }
                } else {
                    output.write(cachedAsset.getResource());
                }
            }
        }
        catch (RuntimeException ignored) {
            resp.sendError(404);
        }
    }

    private boolean isCachedClientSide(HttpServletRequest req, Asset cachedAsset) {
        return cachedAsset.getETag().equals(req.getHeader("If-None-Match")) || req.getDateHeader("If-Modified-Since") >= cachedAsset.getLastModifiedTime();
    }

    private ImmutableList<ByteRange> parseRangeHeader(String rangeHeader, int resourceLength) {
        String[] parts;
        ImmutableList.Builder builder = ImmutableList.builder();
        if (rangeHeader.contains("=") && (parts = rangeHeader.split("=")).length > 1) {
            List<String> ranges = Splitter.on(",").trimResults().splitToList(parts[1]);
            for (String range : ranges) {
                builder.add(ByteRange.parse(range, resourceLength));
            }
        }
        return builder.build();
    }

    private static class StaticAsset
    implements Asset {
        private final byte[] resource;
        private final String etag;
        private final long lastModifiedTime;

        private StaticAsset(byte[] resource, long lastModifiedTime) {
            this.resource = resource;
            this.etag = '\"' + Hashing.murmur3_128().hashBytes(resource).toString() + '\"';
            this.lastModifiedTime = lastModifiedTime;
        }

        @Override
        public byte[] getResource() {
            return this.resource;
        }

        @Override
        public String getETag() {
            return this.etag;
        }

        @Override
        public long getLastModifiedTime() {
            return this.lastModifiedTime;
        }
    }

    private static class FileSystemAsset
    implements Asset {
        private final File file;
        private byte[] bytes;
        private String etag;
        private long lastModifiedTime;

        public FileSystemAsset(File file) {
            this.file = file;
            this.refresh();
        }

        @Override
        public byte[] getResource() {
            this.maybeRefresh();
            return this.bytes;
        }

        @Override
        public String getETag() {
            this.maybeRefresh();
            return this.etag;
        }

        @Override
        public long getLastModifiedTime() {
            this.maybeRefresh();
            return this.lastModifiedTime / 1000L * 1000L;
        }

        private synchronized void maybeRefresh() {
            if (this.lastModifiedTime != this.file.lastModified()) {
                this.refresh();
            }
        }

        private synchronized void refresh() {
            try {
                byte[] newBytes = Files.toByteArray(this.file);
                String newETag = Hashing.murmur3_128().hashBytes(newBytes).toString();
                this.bytes = newBytes;
                this.etag = '\"' + newETag + '\"';
                this.lastModifiedTime = this.file.lastModified();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }

    private static final class AssetSizeWeigher
    implements Weigher<String, Asset> {
        private AssetSizeWeigher() {
        }

        @Override
        public int weigh(String key, Asset asset) {
            return asset.getResource().length;
        }
    }

    private static interface Asset {
        public byte[] getResource();

        public String getETag();

        public long getLastModifiedTime();
    }

    private static class AssetLoader
    extends CacheLoader<String, Asset> {
        private final String indexFilename;
        private final Map<String, String> resourcePathToUriMappings = Maps.newHashMap();
        private final Iterable<Map.Entry<String, String>> overrides;

        private AssetLoader(Iterable<Map.Entry<String, String>> resourcePathToUriMappings, String indexFilename, Iterable<Map.Entry<String, String>> overrides) {
            for (Map.Entry<String, String> mapping : resourcePathToUriMappings) {
                String uriPath;
                String trimmedPath = SLASHES.trimFrom(mapping.getKey());
                String resourcePath = trimmedPath.isEmpty() ? trimmedPath : trimmedPath + '/';
                String trimmedUri = SLASHES.trimTrailingFrom(mapping.getValue());
                String string = uriPath = trimmedUri.isEmpty() ? "/" : trimmedUri;
                if (this.resourcePathToUriMappings.containsKey(resourcePath)) {
                    throw new IllegalArgumentException("ResourcePathToUriMappings contains multiple mappings for " + resourcePath);
                }
                this.resourcePathToUriMappings.put(resourcePath, uriPath);
            }
            this.indexFilename = indexFilename;
            this.overrides = overrides;
        }

        @Override
        public Asset load(String key) throws Exception {
            for (Map.Entry<String, String> mapping : this.resourcePathToUriMappings.entrySet()) {
                if (!key.startsWith(mapping.getValue())) continue;
                Asset asset = this.loadOverride(key);
                if (asset != null) {
                    return asset;
                }
                String requestedResourcePath = SLASHES.trimFrom(key.substring(mapping.getValue().length()));
                String absolutePath = SLASHES.trimFrom(mapping.getKey() + requestedResourcePath);
                try {
                    long lastModified;
                    URL requestedResourceUrl = UrlUtil.switchFromZipToJarProtocolIfNeeded(Resources.getResource(absolutePath));
                    if (ResourceURL.isDirectory(requestedResourceUrl)) {
                        if (this.indexFilename == null) continue;
                        requestedResourceUrl = Resources.getResource(absolutePath + '/' + this.indexFilename);
                        requestedResourceUrl = UrlUtil.switchFromZipToJarProtocolIfNeeded(requestedResourceUrl);
                    }
                    if ((lastModified = ResourceURL.getLastModified(requestedResourceUrl)) < 1L) {
                        lastModified = System.currentTimeMillis();
                    }
                    lastModified = lastModified / 1000L * 1000L;
                    return new StaticAsset(Resources.toByteArray(requestedResourceUrl), lastModified);
                }
                catch (IllegalArgumentException illegalArgumentException) {
                }
            }
            return null;
        }

        private Asset loadOverride(String key) throws Exception {
            for (Map.Entry<String, String> override : this.overrides) {
                File file = null;
                if (override.getKey().equals(key)) {
                    file = new File(override.getValue());
                } else if (key.startsWith(override.getKey())) {
                    file = new File(override.getValue(), key.substring(override.getKey().length()));
                }
                if (file == null || !file.exists()) continue;
                if (file.isDirectory()) {
                    file = new File(file, this.indexFilename);
                }
                if (!file.exists()) continue;
                return new FileSystemAsset(file);
            }
            return null;
        }
    }
}

