package com.gradle.publish;

import com.gradle.publish.protocols.v1.models.publish.ArtifactTypeCodec;
import com.gradle.publish.protocols.v1.models.publish.BuildMetadata;
import com.gradle.publish.protocols.v1.models.publish.PublishArtifact;
import com.gradle.publish.protocols.v1.models.publish.PublishMavenCoordinates;
import com.gradle.publish.protocols.v1.models.publish.PublishNewVersionRequest;
import org.gradle.api.DefaultTask;
import org.gradle.api.artifacts.component.ComponentIdentifier;
import org.gradle.api.artifacts.component.ModuleComponentIdentifier;
import org.gradle.api.file.Directory;
import org.gradle.api.file.DirectoryProperty;
import org.gradle.api.internal.notations.ComponentIdentifierParserFactory;
import org.gradle.api.logging.Logger;
import org.gradle.api.logging.Logging;
import org.gradle.api.provider.ListProperty;
import org.gradle.api.provider.Property;
import org.gradle.api.provider.Provider;
import org.gradle.api.tasks.Input;
import org.gradle.api.tasks.InputDirectory;
import org.gradle.api.tasks.PathSensitive;
import org.gradle.api.tasks.PathSensitivity;
import org.gradle.api.tasks.TaskAction;
import org.gradle.internal.typeconversion.NotationParser;
import org.gradle.work.DisableCachingByDefault;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

@DisableCachingByDefault(because = "Not cacheable")
public abstract class PublishExistingTask extends DefaultTask {

    private static final Logger LOGGER = Logging.getLogger(PublishExistingTask.class);

    private final PortalPublisher portalPublisher = new PortalPublisher(getProject());

    @InputDirectory
    @PathSensitive(PathSensitivity.RELATIVE)
    public abstract DirectoryProperty getFileRepositoryRoot();

    @Input
    public abstract Property<String> getPluginId();

    @Input
    public abstract Property<String> getPluginDescription();

    @Input
    public abstract Property<String> getDisplayName();

    @Input
    public abstract Property<String> getPluginVersion();

    @Input
    public abstract Property<String> getWebsite();

    @Input
    public abstract Property<String> getVcsUrl();

    @Input
    public abstract ListProperty<String> getTags();

    @Input
    public abstract Property<String> getPluginCoordinates();

    @TaskAction
    void publish() throws Exception {
        List<PublishNewVersionRequest> requests = buildPublishRequests();

        // Compute files location in repo
        ComponentIdentifierParserFactory componentIdentifierParserFactory = new ComponentIdentifierParserFactory();
        NotationParser<Object, ComponentIdentifier> notationParser = (NotationParser<Object, ComponentIdentifier>) componentIdentifierParserFactory.create();

        ModuleComponentIdentifier componentIdentifier = (ModuleComponentIdentifier) getPluginCoordinates().map(notationParser::parseNotation).get();
        Provider<Directory> repoEntry = getFileRepositoryRoot().dir(componentIdentifier.getGroup().replace('.', '/') + "/" + componentIdentifier.getModule() + "/" + componentIdentifier.getVersion());

        Map<PublishArtifact, File> artifacts = collectArtifacts(componentIdentifier, repoEntry.get());

        validateArtifacts(artifacts, repoEntry);
        portalPublisher.publishToPortal(requests, new PublishMavenCoordinates(componentIdentifier.getGroup(), componentIdentifier.getModule(), componentIdentifier.getVersion()), artifacts);
    }

    private void validateArtifacts(Map<PublishArtifact, File> artifacts, Provider<Directory> repoEntry) {
        if (artifacts.isEmpty()) {
            throw new IllegalStateException("Could not find any artifacts to publish in '" + repoEntry.get().getAsFile() + "'");
        }
        boolean hasJar = false;
        boolean hasMetadata = false;
        for (PublishArtifact publishArtifact : artifacts.keySet()) {
            if (publishArtifact.getType().equals("jar")) {
                hasJar = true;
            } else if (publishArtifact.getType().equals("pom")) {
                hasMetadata = true;
            }
        }

        if (!hasJar) {
            throw new IllegalStateException("Could not find a main jar artifact to publish in '" + repoEntry.get().getAsFile() + "'");
        }
        if (!hasMetadata) {
            throw new IllegalStateException("Could not find a maven pom to publish in '" + repoEntry.get().getAsFile() + "'");
        }
    }

    void addAndHashArtifact(Map<PublishArtifact, File> artifacts, File file, String filePrefix) throws IOException {
        String fileName = file.getName();
        String classifierAndExtension = fileName.replace(filePrefix, "");
        int extensionSeparator = classifierAndExtension.indexOf('.');
        String classifier = null;
        if (extensionSeparator > 0) {
            classifier = classifierAndExtension.substring(1, extensionSeparator);
        }
        String extension = classifierAndExtension.substring(extensionSeparator + 1);
        if (extension.contains("md5") || extension.contains("sha")) {
            return;
        }

        try (FileInputStream fis = new FileInputStream(file)) {
            String hash = Hasher.hash(fis);
            try {
                String artifactType = ArtifactTypeCodec.encode(extension, classifier);
                artifacts.put(new PublishArtifact(artifactType, hash), file);
            } catch (IllegalArgumentException e) {
                LOGGER.warn("Ignoring unknown artifact with type \"{}\" and " +
                        "classifier \"{}\".\nYou can only upload normal jars, " +
                        "sources jars, javadoc jars and groovydoc jars\n" +
                        "with or without signatures to the Plugin Portal at this time.",
                    extension, classifier);
            }
        }
    }

    private Map<PublishArtifact, File> collectArtifacts(ModuleComponentIdentifier identifier, Directory repoEntry) throws IOException {
        String filePrefix = identifier.getModule() + "-" + identifier.getVersion();
        Map<PublishArtifact, File> artifacts = new LinkedHashMap<>();

        for (File file : repoEntry.getAsFileTree().getFiles()) {
            addAndHashArtifact(artifacts, file, filePrefix);
        }
        return artifacts;
    }

    private List<PublishNewVersionRequest> buildPublishRequests() {
        List<PublishNewVersionRequest> reqs = new ArrayList<>();
        reqs.add(buildPublishRequest());
        return reqs;
    }

    private PublishNewVersionRequest buildPublishRequest() {
        PublishNewVersionRequest request = new PublishNewVersionRequest();
        BuildMetadata buildMetadata = new BuildMetadata(getProject().getGradle().getGradleVersion());
        request.setBuildMetadata(buildMetadata);

        request.setPluginId(getPluginId().get());

        String pluginVersion = getPluginVersion().getOrElse(getProject().getVersion().toString());
        request.setPluginVersion(pluginVersion);
        request.setDisplayName(getDisplayName().get());

        request.setDescription(getPluginDescription().get());
        request.setTags(getTags().get());
        request.setWebSite(getWebsite().get());
        request.setVcsUrl(getVcsUrl().get());

        return request;
    }

}
