package com.gradle.publish;

import com.gradle.publish.protocols.v1.models.login.LoginCheckRequest;
import com.gradle.publish.protocols.v1.models.login.LoginCheckResponse;
import com.gradle.publish.protocols.v1.models.login.LoginInitRequest;
import com.gradle.publish.protocols.v1.models.login.LoginInitResponse;
import org.apache.commons.io.IOUtils;
import org.gradle.api.DefaultTask;
import org.gradle.api.Project;
import org.gradle.api.tasks.TaskAction;
import org.gradle.work.DisableCachingByDefault;

import java.io.File;
import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.InetAddress;
import java.net.URL;
import java.util.LinkedHashMap;
import java.util.Map;

@DisableCachingByDefault(because = "Not cacheable")
public class LoginTask extends DefaultTask {
    private Config ghConfig = new Config(getProject());

    private boolean isLoggedIn(LoginCheckRequest loginCheckRequest) throws IOException {
        URL url = new URL(ghConfig.getPortalUrl() + loginCheckRequest.requestProtocolURL());
        HttpURLConnection connection = (HttpURLConnection) url.openConnection();
        connection.setRequestMethod(loginCheckRequest.getRequestMethod());
        String response = IOUtils.toString(connection.getInputStream());
        getLogger().debug("Is logged in response: " + response);
        LoginCheckResponse loginCheckResponse = loginCheckRequest.convertResponse(response);

        // NOTE: This was only added in 0.11.0
        ResponseUtil.assertValidResponse("Failed checking if user is logged in.", loginCheckResponse);

        connection.disconnect();
        if (loginCheckResponse.getCompleted()) {
            String key = loginCheckResponse.getKey();
            String secret = loginCheckResponse.getSecret();
            updateKeySecretProperties(key, secret, loginCheckResponse.getSuccessMessage());
        }
        return loginCheckResponse.getCompleted();
    }

    static File getGradleHomePropertiesFile(Project project) {
        File defaultGradleDir = project.getGradle().getGradleUserHomeDir();
        return new File(defaultGradleDir, Project.GRADLE_PROPERTIES);
    }

    private void updateKeySecretProperties(String key, String secret, String message) throws IOException {
        File propertiesFile = getGradleHomePropertiesFile(getProject());
        getLogger().debug("Updating key and secret in file: " + propertiesFile.getAbsolutePath());
        Map<String, String> props = new LinkedHashMap<String, String>();
        props.put(PublishTask.GRADLE_PUBLISH_KEY, key);
        props.put(PublishTask.GRADLE_PUBLISH_SECRET, secret);
        PropertiesStore.append(propertiesFile, props, "Updated secret and key with server message: " + message);
        getLogger().quiet("Your publishing API keys have been written to: " + propertiesFile.getAbsolutePath());
    }

    @TaskAction
    public void login() throws IOException, InterruptedException {
        String keyName = InetAddress.getLocalHost().getHostName();
        LoginInitRequest request = new LoginInitRequest(keyName);
        URL url = new URL(ghConfig.getPortalUrl() + request.requestProtocolURL());
        HttpURLConnection connection = (HttpURLConnection) url.openConnection();
        connection.setRequestMethod(request.getRequestMethod());

        RequestUtil.addClientVersionAsRequestHeader(connection);

        String stringResponse = IOUtils.toString(connection.getInputStream());
        int status = connection.getResponseCode();
        getLogger().debug("Login response: " + stringResponse);
        LoginInitResponse apiResponse = Util.convertResponse(request, url, stringResponse, status);

        // NOTE: This was only added in 0.11.0
        ResponseUtil.assertValidResponse("Failed login initialization request.", apiResponse);

        File propertiesFile = getGradleHomePropertiesFile(getProject());
        getLogger().quiet("\nTo add your publishing API keys to " + propertiesFile.getAbsolutePath() + ", open the following URL in your browser:\n\n" +
            "   " + apiResponse.getValidateUrl() + "\n" +
            "\n");
        connection.disconnect();
        Long startTime = System.currentTimeMillis();
        while (!isLoggedIn(apiResponse.getNextRequest()) &&
            (System.currentTimeMillis() - startTime) < ghConfig.loginMaxWait()) {
            Thread.sleep(ghConfig.loginSleepInterval());
            getLogger().trace("Trying login again: " + apiResponse.getToken());
        }
    }

}
