package com.gradle.publish;

public class PluginId {

    public static final String ID_SEPARATOR_ON_START_OR_END = "cannot begin or end with '" + PluginId.SEPARATOR + "'";
    public static final String DOUBLE_SEPARATOR = "cannot contain '" + PluginId.SEPARATOR + PluginId.SEPARATOR + "'";

    public static final String PLUGIN_ID_VALID_CHARS_DESCRIPTION = "ASCII alphanumeric characters, '.', '_' and '-'";
    public static final String SEPARATOR = ".";

    private final String value;

    // Only use when id is guaranteed to be valid, prefer of()
    public PluginId(String value) {
        this.value = value;
    }

    public static PluginId of(String value) throws InvalidPluginIdException {
        validate(value);
        return new PluginId(value);
    }

    public static void validate(String value) throws InvalidPluginIdException {
        if (value.startsWith(SEPARATOR) || value.endsWith(SEPARATOR)) {
            throw new InvalidPluginIdException(value, ID_SEPARATOR_ON_START_OR_END);
        } else if (value.contains(PluginId.SEPARATOR + PluginId.SEPARATOR)) {
            throw new InvalidPluginIdException(value, DOUBLE_SEPARATOR);
        } else {
            Character invalidChar = invalidChar(value);
            if (invalidChar != null) {
                throw new InvalidPluginIdException(value, invalidPluginIdCharMessage(invalidChar));
            }
        }
    }

    public static String invalidPluginIdCharMessage(char invalidChar) {
        return "contains invalid char '" + invalidChar + "' (only " + PluginId.PLUGIN_ID_VALID_CHARS_DESCRIPTION + " characters are valid)";
    }

    public boolean isQualified() {
        return value.contains(PluginId.SEPARATOR);
    }

    public PluginId maybeQualify(String qualification) {
        return isQualified() ? this : new PluginId(qualification + PluginId.SEPARATOR + value);
    }

    public String getNamespace() {
        return isQualified() ? value.substring(0, value.lastIndexOf(SEPARATOR)) : null;
    }

    public boolean inNamespace(String namespace) {
        return isQualified() && getNamespace().equals(namespace);
    }

    public String getName() {
        return isQualified() ? value.substring(value.lastIndexOf(PluginId.SEPARATOR) + 1) : value;
    }

    public PluginId getUnqualified() {
        return isQualified() ? new PluginId(getName()) : this;
    }

    @Override
    public String toString() {
        return value;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        PluginId pluginId = (PluginId) o;

        return value.equals(pluginId.value);
    }

    @Override
    public int hashCode() {
        return value.hashCode();
    }

    private static Character invalidChar(String src) {
        for (char c : src.toCharArray()) {
            if (!(inRange('A', 'Z', c) || inRange('a', 'z', c) || inRange('0', '9', c)
                || c == '.' || c == '-' || c == '_')) {
                return c;
            }
        }
        return null;
    }

    private static boolean inRange(char start, char end, char c) {
        return c >= start && c <= end;
    }
}
