/*
 * Decompiled with CFR 0.152.
 */
package com.google.visualization.datasource.util;

import com.google.common.collect.Lists;
import com.google.visualization.datasource.base.DataSourceException;
import com.google.visualization.datasource.base.ReasonType;
import com.google.visualization.datasource.base.TypeMismatchException;
import com.google.visualization.datasource.datatable.ColumnDescription;
import com.google.visualization.datasource.datatable.DataTable;
import com.google.visualization.datasource.datatable.TableCell;
import com.google.visualization.datasource.datatable.TableRow;
import com.google.visualization.datasource.datatable.value.BooleanValue;
import com.google.visualization.datasource.datatable.value.DateTimeValue;
import com.google.visualization.datasource.datatable.value.DateValue;
import com.google.visualization.datasource.datatable.value.NumberValue;
import com.google.visualization.datasource.datatable.value.TextValue;
import com.google.visualization.datasource.datatable.value.TimeOfDayValue;
import com.google.visualization.datasource.datatable.value.Value;
import com.google.visualization.datasource.datatable.value.ValueType;
import com.google.visualization.datasource.query.AbstractColumn;
import com.google.visualization.datasource.query.AggregationColumn;
import com.google.visualization.datasource.query.AggregationType;
import com.google.visualization.datasource.query.ColumnColumnFilter;
import com.google.visualization.datasource.query.ColumnIsNullFilter;
import com.google.visualization.datasource.query.ColumnSort;
import com.google.visualization.datasource.query.ColumnValueFilter;
import com.google.visualization.datasource.query.ComparisonFilter;
import com.google.visualization.datasource.query.CompoundFilter;
import com.google.visualization.datasource.query.NegationFilter;
import com.google.visualization.datasource.query.Query;
import com.google.visualization.datasource.query.QueryFilter;
import com.google.visualization.datasource.query.QueryGroup;
import com.google.visualization.datasource.query.QuerySelection;
import com.google.visualization.datasource.query.QuerySort;
import com.google.visualization.datasource.query.SimpleColumn;
import com.google.visualization.datasource.query.SortOrder;
import com.google.visualization.datasource.util.SqlDatabaseDescription;
import com.ibm.icu.util.GregorianCalendar;
import com.ibm.icu.util.TimeZone;
import java.sql.Connection;
import java.sql.Date;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.text.StrBuilder;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class SqlDataSourceHelper {
    private static final Log log = LogFactory.getLog((String)SqlDataSourceHelper.class.getName());

    private SqlDataSourceHelper() {
    }

    public static DataTable executeQuery(Query query, SqlDatabaseDescription databaseDescription) throws DataSourceException {
        Connection con = SqlDataSourceHelper.getDatabaseConnection(databaseDescription);
        String tableName = databaseDescription.getTableName();
        StrBuilder queryStringBuilder = new StrBuilder();
        SqlDataSourceHelper.buildSqlQuery(query, queryStringBuilder, tableName);
        List<String> columnIdsList = null;
        if (query.hasSelection()) {
            columnIdsList = SqlDataSourceHelper.getColumnIdsList(query.getSelection());
        }
        Statement stmt = null;
        try {
            stmt = con.createStatement();
            ResultSet rs = stmt.executeQuery(queryStringBuilder.toString());
            DataTable table = SqlDataSourceHelper.buildColumns(rs, columnIdsList);
            SqlDataSourceHelper.buildRows(table, rs);
            DataTable dataTable = table;
            return dataTable;
        }
        catch (SQLException e) {
            String messageToUser = "Failed to execute SQL query. mySQL error message: " + e.getMessage();
            throw new DataSourceException(ReasonType.INTERNAL_ERROR, messageToUser);
        }
        finally {
            if (stmt != null) {
                try {
                    stmt.close();
                }
                catch (SQLException e) {}
            }
            if (con != null) {
                try {
                    con.close();
                }
                catch (SQLException e) {}
            }
        }
    }

    private static Connection getDatabaseConnection(SqlDatabaseDescription databaseDescription) throws DataSourceException {
        Connection con;
        String userName = databaseDescription.getUser();
        String password = databaseDescription.getPassword();
        String url = databaseDescription.getUrl();
        try {
            con = DriverManager.getConnection(url, userName, password);
        }
        catch (SQLException e) {
            log.error((Object)"Failed to connect to database server.", (Throwable)e);
            throw new DataSourceException(ReasonType.INTERNAL_ERROR, "Failed to connect to database server.");
        }
        return con;
    }

    private static void buildSqlQuery(Query query, StrBuilder queryStringBuilder, String tableName) throws DataSourceException {
        SqlDataSourceHelper.appendSelectClause(query, queryStringBuilder);
        SqlDataSourceHelper.appendFromClause(query, queryStringBuilder, tableName);
        SqlDataSourceHelper.appendWhereClause(query, queryStringBuilder);
        SqlDataSourceHelper.appendGroupByClause(query, queryStringBuilder);
        SqlDataSourceHelper.appendOrderByClause(query, queryStringBuilder);
        SqlDataSourceHelper.appendLimitAndOffsetClause(query, queryStringBuilder);
    }

    static void appendLimitAndOffsetClause(Query query, StrBuilder queryStringBuilder) {
        if (query.hasRowLimit()) {
            queryStringBuilder.append("LIMIT ");
            queryStringBuilder.append(query.getRowLimit());
        }
        if (query.hasRowOffset()) {
            queryStringBuilder.append(" OFFSET ").append(query.getRowOffset());
        }
    }

    static void appendGroupByClause(Query query, StrBuilder queryStringBuilder) {
        if (!query.hasGroup()) {
            return;
        }
        queryStringBuilder.append("GROUP BY ");
        QueryGroup queryGroup = query.getGroup();
        List<String> groupColumnIds = queryGroup.getColumnIds();
        ArrayList newColumnIds = Lists.newArrayList();
        for (String groupColumnId : groupColumnIds) {
            newColumnIds.add('`' + groupColumnId + '`');
        }
        queryStringBuilder.appendWithSeparators((Collection)newColumnIds, ", ");
        queryStringBuilder.append(" ");
    }

    static void appendOrderByClause(Query query, StrBuilder queryStringBuilder) {
        if (!query.hasSort()) {
            return;
        }
        queryStringBuilder.append("ORDER BY ");
        QuerySort querySort = query.getSort();
        List<ColumnSort> sortColumns = querySort.getSortColumns();
        int numOfSortColumns = sortColumns.size();
        for (int col = 0; col < numOfSortColumns; ++col) {
            ColumnSort columnSort = sortColumns.get(col);
            queryStringBuilder.append(SqlDataSourceHelper.getColumnId(columnSort.getColumn()));
            if (columnSort.getOrder() == SortOrder.DESCENDING) {
                queryStringBuilder.append(" DESC");
            }
            if (col >= numOfSortColumns - 1) continue;
            queryStringBuilder.append(", ");
        }
        queryStringBuilder.append(" ");
    }

    static void appendWhereClause(Query query, StrBuilder queryStringBuilder) {
        if (query.hasFilter()) {
            QueryFilter queryFilter = query.getFilter();
            queryStringBuilder.append("WHERE ").append(SqlDataSourceHelper.buildWhereClauseRecursively(queryFilter)).append(" ");
        }
    }

    private static StrBuilder buildWhereClauseRecursively(QueryFilter queryFilter) {
        StrBuilder whereClause = new StrBuilder();
        if (queryFilter instanceof ColumnIsNullFilter) {
            SqlDataSourceHelper.buildWhereClauseForIsNullFilter(whereClause, queryFilter);
        } else if (queryFilter instanceof ComparisonFilter) {
            SqlDataSourceHelper.buildWhereCluaseForComparisonFilter(whereClause, queryFilter);
        } else if (queryFilter instanceof NegationFilter) {
            whereClause.append("(NOT ");
            whereClause.append(SqlDataSourceHelper.buildWhereClauseRecursively(((NegationFilter)queryFilter).getSubFilter()));
            whereClause.append(")");
        } else {
            CompoundFilter compoundFilter = (CompoundFilter)queryFilter;
            int numberOfSubFilters = compoundFilter.getSubFilters().size();
            if (numberOfSubFilters == 0) {
                if (compoundFilter.getOperator() == CompoundFilter.LogicalOperator.AND) {
                    whereClause.append("true");
                } else {
                    whereClause.append("false");
                }
            } else {
                ArrayList filterComponents = Lists.newArrayList();
                for (QueryFilter filter : compoundFilter.getSubFilters()) {
                    filterComponents.add(SqlDataSourceHelper.buildWhereClauseRecursively(filter).toString());
                }
                String logicalOperator = SqlDataSourceHelper.getSqlLogicalOperator(compoundFilter.getOperator());
                whereClause.append("(").appendWithSeparators((Collection)filterComponents, " " + logicalOperator + " ").append(")");
            }
        }
        return whereClause;
    }

    private static void buildWhereClauseForIsNullFilter(StrBuilder whereClause, QueryFilter queryFilter) {
        ColumnIsNullFilter filter = (ColumnIsNullFilter)queryFilter;
        whereClause.append("(").append(filter.getColumn().getId()).append(" IS NULL)");
    }

    private static void buildWhereCluaseForComparisonFilter(StrBuilder whereClause, QueryFilter queryFilter) {
        StrBuilder first = new StrBuilder();
        StrBuilder second = new StrBuilder();
        if (queryFilter instanceof ColumnColumnFilter) {
            ColumnColumnFilter filter = (ColumnColumnFilter)queryFilter;
            first.append(filter.getFirstColumn().getId());
            second.append(filter.getSecondColumn().getId());
        } else {
            ColumnValueFilter filter = (ColumnValueFilter)queryFilter;
            first.append(filter.getColumn().getId());
            second.append(filter.getValue().toString());
            if (filter.getValue().getType() == ValueType.TEXT || filter.getValue().getType() == ValueType.DATE || filter.getValue().getType() == ValueType.DATETIME || filter.getValue().getType() == ValueType.TIMEOFDAY) {
                second.insert(0, "\"");
                second.insert(second.length(), "\"");
            }
        }
        whereClause.append(SqlDataSourceHelper.buildWhereClauseFromRightAndLeftParts(first, second, ((ComparisonFilter)queryFilter).getOperator()));
    }

    private static String getSqlLogicalOperator(CompoundFilter.LogicalOperator operator) {
        String stringOperator;
        switch (operator) {
            case AND: {
                stringOperator = "AND";
                break;
            }
            case OR: {
                stringOperator = "OR";
                break;
            }
            default: {
                throw new RuntimeException("Logical operator was not found: " + (Object)((Object)operator));
            }
        }
        return stringOperator;
    }

    private static StrBuilder buildWhereClauseFromRightAndLeftParts(StrBuilder value1, StrBuilder value2, ComparisonFilter.Operator operator) {
        StrBuilder clause;
        switch (operator) {
            case EQ: {
                clause = value1.append("=").append(value2);
                break;
            }
            case NE: {
                clause = value1.append("<>").append(value2);
                break;
            }
            case LT: {
                clause = value1.append("<").append(value2);
                break;
            }
            case GT: {
                clause = value1.append(">").append(value2);
                break;
            }
            case LE: {
                clause = value1.append("<=").append(value2);
                break;
            }
            case GE: {
                clause = value1.append(">=").append(value2);
                break;
            }
            case CONTAINS: {
                value2 = new StrBuilder(value2.toString().replace("\"", ""));
                clause = value1.append(" LIKE ").append("\"%").append(value2).append("%\"");
                break;
            }
            case STARTS_WITH: {
                value2 = new StrBuilder(value2.toString().replace("\"", ""));
                clause = value1.append(" LIKE ").append("\"").append(value2).append("%\"");
                break;
            }
            case ENDS_WITH: {
                value2 = new StrBuilder(value2.toString().replace("\"", ""));
                clause = value1.append(" LIKE ").append("\"%").append(value2).append("\"");
                break;
            }
            case MATCHES: {
                throw new RuntimeException("SQL does not support regular expression");
            }
            case LIKE: {
                value2 = new StrBuilder(value2.toString().replace("\"", ""));
                clause = value1.append(" LIKE ").append("\"").append(value2).append("\"");
                break;
            }
            default: {
                throw new RuntimeException("Operator was not found: " + (Object)((Object)operator));
            }
        }
        clause.insert(0, "(").append(")");
        return clause;
    }

    static void appendSelectClause(Query query, StrBuilder queryStringBuilder) {
        queryStringBuilder.append("SELECT ");
        if (!query.hasSelection()) {
            queryStringBuilder.append("* ");
            return;
        }
        List<AbstractColumn> columns = query.getSelection().getColumns();
        int numOfColsInQuery = columns.size();
        for (int col = 0; col < numOfColsInQuery; ++col) {
            queryStringBuilder.append(SqlDataSourceHelper.getColumnId(columns.get(col)));
            if (col >= numOfColsInQuery - 1) continue;
            queryStringBuilder.append(", ");
        }
        queryStringBuilder.append(" ");
    }

    private static StrBuilder getColumnId(AbstractColumn abstractColumn) {
        StrBuilder columnId = new StrBuilder();
        if (abstractColumn instanceof SimpleColumn) {
            columnId.append("`").append(abstractColumn.getId()).append("`");
        } else {
            AggregationColumn aggregationColumn = (AggregationColumn)abstractColumn;
            columnId.append(SqlDataSourceHelper.getAggregationFunction(aggregationColumn.getAggregationType())).append("(`").append((Object)aggregationColumn.getAggregatedColumn()).append("`)");
        }
        return columnId;
    }

    private static List<String> getColumnIdsList(QuerySelection selection) {
        ArrayList columnIds = Lists.newArrayListWithCapacity((int)selection.getColumns().size());
        for (AbstractColumn column : selection.getColumns()) {
            columnIds.add(column.getId());
        }
        return columnIds;
    }

    private static String getAggregationFunction(AggregationType type) {
        return type.getCode();
    }

    static void appendFromClause(Query query, StrBuilder queryStringBuilder, String tableName) throws DataSourceException {
        if (StringUtils.isEmpty((String)tableName)) {
            log.error((Object)"No table name provided.");
            throw new DataSourceException(ReasonType.OTHER, "No table name provided.");
        }
        queryStringBuilder.append("FROM ");
        queryStringBuilder.append(tableName);
        queryStringBuilder.append(" ");
    }

    static DataTable buildColumns(ResultSet rs, List<String> columnIdsList) throws SQLException {
        DataTable result = new DataTable();
        ResultSetMetaData metaData = rs.getMetaData();
        int numOfCols = metaData.getColumnCount();
        for (int i = 1; i <= numOfCols; ++i) {
            String id = columnIdsList == null ? metaData.getColumnLabel(i) : columnIdsList.get(i - 1);
            ColumnDescription columnDescription = new ColumnDescription(id, SqlDataSourceHelper.sqlTypeToValueType(metaData.getColumnType(i)), metaData.getColumnLabel(i));
            result.addColumn(columnDescription);
        }
        return result;
    }

    private static ValueType sqlTypeToValueType(int sqlType) {
        ValueType valueType;
        switch (sqlType) {
            case -7: 
            case 16: {
                valueType = ValueType.BOOLEAN;
                break;
            }
            case 1: 
            case 12: {
                valueType = ValueType.TEXT;
                break;
            }
            case -6: 
            case -5: 
            case 2: 
            case 3: 
            case 4: 
            case 5: 
            case 6: 
            case 7: 
            case 8: {
                valueType = ValueType.NUMBER;
                break;
            }
            case 91: {
                valueType = ValueType.DATE;
                break;
            }
            case 92: {
                valueType = ValueType.TIMEOFDAY;
                break;
            }
            case 93: {
                valueType = ValueType.DATETIME;
                break;
            }
            default: {
                valueType = ValueType.TEXT;
            }
        }
        return valueType;
    }

    static void buildRows(DataTable dataTable, ResultSet rs) throws SQLException {
        List<ColumnDescription> columnsDescriptionList = dataTable.getColumnDescriptions();
        int numOfCols = dataTable.getNumberOfColumns();
        ValueType[] columnsTypeArray = new ValueType[numOfCols];
        for (int c = 0; c < numOfCols; ++c) {
            columnsTypeArray[c] = columnsDescriptionList.get(c).getType();
        }
        while (rs.next()) {
            TableRow tableRow = new TableRow();
            for (int c = 0; c < numOfCols; ++c) {
                tableRow.addCell(SqlDataSourceHelper.buildTableCell(rs, columnsTypeArray[c], c));
            }
            try {
                dataTable.addRow(tableRow);
            }
            catch (TypeMismatchException e) {}
        }
    }

    private static TableCell buildTableCell(ResultSet rs, ValueType valueType, int column) throws SQLException {
        Value value = null;
        ++column;
        switch (valueType) {
            case BOOLEAN: {
                value = BooleanValue.getInstance(rs.getBoolean(column));
                break;
            }
            case NUMBER: {
                value = new NumberValue(rs.getDouble(column));
                break;
            }
            case DATE: {
                Date date = rs.getDate(column);
                if (date == null) break;
                GregorianCalendar gc = new GregorianCalendar(TimeZone.getTimeZone((String)"GMT"));
                gc.setTime((java.util.Date)date);
                value = new DateValue(gc);
                break;
            }
            case DATETIME: {
                Timestamp timestamp = rs.getTimestamp(column);
                if (timestamp == null) break;
                GregorianCalendar gc = new GregorianCalendar(TimeZone.getTimeZone((String)"GMT"));
                gc.setTime((java.util.Date)timestamp);
                gc.set(14, timestamp.getNanos() / 1000000);
                value = new DateTimeValue(gc);
                break;
            }
            case TIMEOFDAY: {
                Time time = rs.getTime(column);
                if (time == null) break;
                GregorianCalendar gc = new GregorianCalendar(TimeZone.getTimeZone((String)"GMT"));
                gc.setTime((java.util.Date)time);
                value = new TimeOfDayValue(gc);
                break;
            }
            default: {
                String colValue = rs.getString(column);
                value = colValue == null ? TextValue.getNullValue() : new TextValue(rs.getString(column));
            }
        }
        if (rs.wasNull()) {
            return new TableCell(Value.getNullValueFromValueType(valueType));
        }
        return new TableCell(value);
    }
}

