/*
 * Copyright 2009 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.google.template.soy.basicfunctions;

import com.google.common.collect.ImmutableSet;
import com.google.template.soy.data.SoyList;
import com.google.template.soy.data.SoyValue;
import com.google.template.soy.data.restricted.IntegerData;
import com.google.template.soy.jssrc.restricted.JsExpr;
import com.google.template.soy.jssrc.restricted.SoyJsSrcFunction;
import com.google.template.soy.pysrc.restricted.PyExpr;
import com.google.template.soy.pysrc.restricted.SoyPySrcFunction;
import com.google.template.soy.shared.restricted.SoyJavaFunction;
import com.google.template.soy.shared.restricted.SoyPureFunction;

import java.util.List;
import java.util.Set;

import javax.inject.Inject;
import javax.inject.Singleton;

/**
 * Soy function that gets the length of a list.
 *
 */
@Singleton
@SoyPureFunction
final class LengthFunction implements SoyJavaFunction, SoyJsSrcFunction, SoyPySrcFunction {


  @Inject
  LengthFunction() {}


  @Override public String getName() {
    return "length";
  }

  @Override public Set<Integer> getValidArgsSizes() {
    return ImmutableSet.of(1);
  }

  @Override public SoyValue computeForJava(List<SoyValue> args) {
    SoyValue arg = args.get(0);

    if (arg == null) {
      throw new IllegalArgumentException("Argument to length() function is null.");
    }

    if (!(arg instanceof SoyList)) {
      throw new IllegalArgumentException("Argument to length() function is not a SoyList " +
          "(found type " + arg.getClass().getName() + ").");
    }
    return IntegerData.forValue(((SoyList) arg).length());
  }

  @Override public JsExpr computeForJsSrc(List<JsExpr> args) {
    JsExpr arg = args.get(0);

    String exprText = arg.getPrecedence() == Integer.MAX_VALUE ?
                      arg.getText() + ".length" : "(" + arg.getText() + ").length";
    return new JsExpr(exprText, Integer.MAX_VALUE);
  }

  @Override public PyExpr computeForPySrc(List<PyExpr> args) {
    PyExpr arg = args.get(0);

    return new PyExpr("len(" + arg.getText() + ")", Integer.MAX_VALUE);
  }
}
