/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.geometry;

import com.google.common.annotations.GwtCompatible;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.geometry.Matrix3x3;
import com.google.common.geometry.R2Vector;
import com.google.common.geometry.S1Angle;
import com.google.common.geometry.S2;
import com.google.common.geometry.S2Loop;
import com.google.common.geometry.S2Point;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;

@GwtCompatible
public class S2FractalBuilder {
    private int maxLevel = -1;
    private int minLevelArg = -1;
    private int minLevel = -1;
    private double dimension = Math.log(4.0) / Math.log(3.0);
    private double edgeFraction = 0.0;
    private double offsetFraction = 0.0;
    private Random rand;

    public S2FractalBuilder(Random rand) {
        this.rand = rand;
        this.computeOffsets();
    }

    public void setMaxLevel(int maxLevel) {
        Preconditions.checkArgument((maxLevel >= 0 ? 1 : 0) != 0);
        this.maxLevel = maxLevel;
        this.computeMinLevel();
    }

    public void setMinLevel(int minLevelArg) {
        Preconditions.checkArgument((minLevelArg >= -1 ? 1 : 0) != 0);
        this.minLevelArg = minLevelArg;
        this.computeMinLevel();
    }

    private void computeMinLevel() {
        this.minLevel = this.minLevelArg >= 0 && this.minLevelArg <= this.maxLevel ? this.minLevelArg : this.maxLevel;
    }

    public void setFractalDimension(double dimension) {
        Preconditions.checkArgument((dimension >= 1.0 ? 1 : 0) != 0);
        Preconditions.checkArgument((dimension <= 2.0 ? 1 : 0) != 0);
        this.dimension = dimension;
        this.computeOffsets();
    }

    private void computeOffsets() {
        this.edgeFraction = Math.pow(4.0, -1.0 / this.dimension);
        this.offsetFraction = Math.sqrt(this.edgeFraction - 0.25);
    }

    public void setLevelForApproxMinEdges(int minEdges) {
        this.setMinLevel(S2FractalBuilder.levelFromEdges(minEdges));
    }

    public void setLevelForApproxMaxEdges(int maxEdges) {
        this.setMaxLevel(S2FractalBuilder.levelFromEdges(maxEdges));
    }

    private static int levelFromEdges(int edges) {
        return (int)Math.ceil(0.5 * Math.log(edges / 3) / Math.log(2.0));
    }

    public double minRadiusFactor() {
        double kMinDimensionForMinRadiusAtLevel1 = 1.0852230903040407;
        if (this.dimension >= kMinDimensionForMinRadiusAtLevel1) {
            return Math.sqrt(1.0 + 3.0 * this.edgeFraction * (this.edgeFraction - 1.0));
        }
        return 0.5;
    }

    public double maxRadiusFactor() {
        return Math.max(1.0, this.offsetFraction * Math.sqrt(3.0) + 0.5);
    }

    private void getR2Vertices(List<R2Vector> vertices) {
        R2Vector v0 = new R2Vector(1.0, 0.0);
        R2Vector v1 = new R2Vector(-0.5, Math.sqrt(3.0) / 2.0);
        R2Vector v2 = new R2Vector(-0.5, -Math.sqrt(3.0) / 2.0);
        this.getR2VerticesHelper(v0, v1, 0, vertices);
        this.getR2VerticesHelper(v1, v2, 0, vertices);
        this.getR2VerticesHelper(v2, v0, 0, vertices);
    }

    private void getR2VerticesHelper(R2Vector v0, R2Vector v4, int level, List<R2Vector> vertices) {
        if (level >= this.minLevel && this.rand.nextInt(this.maxLevel - level + 1) == 0) {
            vertices.add(v0);
            return;
        }
        R2Vector dir = R2Vector.sub(v4, v0);
        R2Vector v1 = R2Vector.add(v0, R2Vector.mul(dir, this.edgeFraction));
        R2Vector v2 = R2Vector.sub(R2Vector.mul(R2Vector.add(v0, v4), 0.5), R2Vector.mul(dir.ortho(), this.offsetFraction));
        R2Vector v3 = R2Vector.sub(v4, R2Vector.mul(dir, this.edgeFraction));
        this.getR2VerticesHelper(v0, v1, level + 1, vertices);
        this.getR2VerticesHelper(v1, v2, level + 1, vertices);
        this.getR2VerticesHelper(v2, v3, level + 1, vertices);
        this.getR2VerticesHelper(v3, v4, level + 1, vertices);
    }

    public S2Loop makeLoop(Matrix3x3 frame, S1Angle nominalRadius) {
        return new S2Loop(this.makeVertices(frame, nominalRadius));
    }

    public List<S2Point> makeVertices(Matrix3x3 frame, S1Angle nominalRadius) {
        ArrayList r2Vertices = Lists.newArrayList();
        this.getR2Vertices(r2Vertices);
        ArrayList vertices = Lists.newArrayList();
        for (int i = 0; i < r2Vertices.size(); ++i) {
            R2Vector v = (R2Vector)r2Vertices.get(i);
            double theta = Math.atan2(v.y(), v.x());
            double radius = nominalRadius.radians() * v.norm();
            double z = Math.cos(radius);
            double r = Math.sin(radius);
            S2Point p = new S2Point(r * Math.cos(theta), r * Math.sin(theta), z);
            vertices.add(S2.rotate(p, frame));
        }
        return vertices;
    }
}

