/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase;

import com.google.api.client.googleapis.util.Utils;
import com.google.api.client.json.JsonFactory;
import com.google.api.client.json.JsonParser;
import com.google.api.core.ApiFuture;
import com.google.auth.oauth2.AccessToken;
import com.google.auth.oauth2.GoogleCredentials;
import com.google.auth.oauth2.OAuth2Credentials;
import com.google.auth.oauth2.ServiceAccountCredentials;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.firebase.FirebaseOptions;
import com.google.firebase.ThreadManager;
import com.google.firebase.internal.FirebaseScheduledExecutor;
import com.google.firebase.internal.FirebaseService;
import com.google.firebase.internal.ListenableFuture2ApiFuture;
import com.google.firebase.internal.NonNull;
import com.google.firebase.internal.Nullable;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FirebaseApp {
    private static final Logger logger = LoggerFactory.getLogger(FirebaseApp.class);
    private static final Map<String, FirebaseApp> instances = new HashMap<String, FirebaseApp>();
    public static final String DEFAULT_APP_NAME = "[DEFAULT]";
    static final String FIREBASE_CONFIG_ENV_VAR = "FIREBASE_CONFIG";
    private static final TokenRefresher.Factory DEFAULT_TOKEN_REFRESHER_FACTORY = new TokenRefresher.Factory();
    private static final Object appsLock = new Object();
    private final String name;
    private final FirebaseOptions options;
    private final TokenRefresher tokenRefresher;
    private final ThreadManager threadManager;
    private final ThreadManager.FirebaseExecutors executors;
    private final AtomicBoolean deleted = new AtomicBoolean();
    private final Map<String, FirebaseService> services = new HashMap<String, FirebaseService>();
    private volatile ScheduledExecutorService scheduledExecutor;
    private final Object lock = new Object();

    private FirebaseApp(String name, FirebaseOptions options, TokenRefresher.Factory factory) {
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)name) ? 1 : 0) != 0);
        this.name = name;
        this.options = (FirebaseOptions)Preconditions.checkNotNull((Object)options);
        this.tokenRefresher = ((TokenRefresher.Factory)Preconditions.checkNotNull((Object)factory)).create(this);
        this.threadManager = options.getThreadManager();
        this.executors = this.threadManager.getFirebaseExecutors(this);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static List<FirebaseApp> getApps() {
        Object object = appsLock;
        synchronized (object) {
            return ImmutableList.copyOf(instances.values());
        }
    }

    public static FirebaseApp getInstance() {
        return FirebaseApp.getInstance(DEFAULT_APP_NAME);
    }

    public static FirebaseApp getInstance(@NonNull String name) {
        Object object = appsLock;
        synchronized (object) {
            FirebaseApp firebaseApp = instances.get(FirebaseApp.normalize(name));
            if (firebaseApp != null) {
                return firebaseApp;
            }
            List<String> availableAppNames = FirebaseApp.getAllAppNames();
            String availableAppNamesMessage = availableAppNames.isEmpty() ? "" : "Available app names: " + Joiner.on((String)", ").join(availableAppNames);
            String errorMessage = String.format("FirebaseApp with name %s doesn't exist. %s", name, availableAppNamesMessage);
            throw new IllegalStateException(errorMessage);
        }
    }

    public static FirebaseApp initializeApp() {
        return FirebaseApp.initializeApp(DEFAULT_APP_NAME);
    }

    public static FirebaseApp initializeApp(String name) {
        try {
            return FirebaseApp.initializeApp(FirebaseApp.getOptionsFromEnvironment(), name);
        }
        catch (IOException e) {
            throw new IllegalArgumentException("Failed to load settings from the system's environment variables", e);
        }
    }

    public static FirebaseApp initializeApp(FirebaseOptions options) {
        return FirebaseApp.initializeApp(options, DEFAULT_APP_NAME);
    }

    public static FirebaseApp initializeApp(FirebaseOptions options, String name) {
        return FirebaseApp.initializeApp(options, name, DEFAULT_TOKEN_REFRESHER_FACTORY);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static FirebaseApp initializeApp(FirebaseOptions options, String name, TokenRefresher.Factory tokenRefresherFactory) {
        String normalizedName = FirebaseApp.normalize(name);
        Object object = appsLock;
        synchronized (object) {
            Preconditions.checkState((!instances.containsKey(normalizedName) ? 1 : 0) != 0, (Object)("FirebaseApp name " + normalizedName + " already exists!"));
            FirebaseApp firebaseApp = new FirebaseApp(normalizedName, options, tokenRefresherFactory);
            instances.put(normalizedName, firebaseApp);
            return firebaseApp;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    static void clearInstancesForTest() {
        Object object = appsLock;
        synchronized (object) {
            for (FirebaseApp app : ImmutableList.copyOf(instances.values())) {
                app.delete();
            }
            instances.clear();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static List<String> getAllAppNames() {
        ArrayList<String> allAppNames;
        Object object = appsLock;
        synchronized (object) {
            allAppNames = new ArrayList<String>(instances.keySet());
        }
        Collections.sort(allAppNames);
        return ImmutableList.copyOf(allAppNames);
    }

    private static String normalize(@NonNull String name) {
        return ((String)Preconditions.checkNotNull((Object)name)).trim();
    }

    @NonNull
    public String getName() {
        return this.name;
    }

    @NonNull
    public FirebaseOptions getOptions() {
        this.checkNotDeleted();
        return this.options;
    }

    @Nullable
    String getProjectId() {
        GoogleCredentials credentials;
        this.checkNotDeleted();
        String projectId = this.options.getProjectId();
        if (Strings.isNullOrEmpty((String)projectId) && (credentials = this.options.getCredentials()) instanceof ServiceAccountCredentials) {
            projectId = ((ServiceAccountCredentials)credentials).getProjectId();
        }
        if (Strings.isNullOrEmpty((String)projectId)) {
            projectId = System.getenv("GOOGLE_CLOUD_PROJECT");
        }
        if (Strings.isNullOrEmpty((String)projectId)) {
            projectId = System.getenv("GCLOUD_PROJECT");
        }
        return projectId;
    }

    public boolean equals(Object o) {
        return o instanceof FirebaseApp && this.name.equals(((FirebaseApp)o).getName());
    }

    public int hashCode() {
        return this.name.hashCode();
    }

    public String toString() {
        return MoreObjects.toStringHelper((Object)this).add("name", (Object)this.name).toString();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void delete() {
        Object object = this.lock;
        synchronized (object) {
            boolean valueChanged = this.deleted.compareAndSet(false, true);
            if (!valueChanged) {
                return;
            }
            for (FirebaseService service : this.services.values()) {
                service.destroy();
            }
            this.services.clear();
            this.tokenRefresher.stop();
            this.threadManager.releaseFirebaseExecutors(this, this.executors);
            if (this.scheduledExecutor != null) {
                this.scheduledExecutor.shutdownNow();
                this.scheduledExecutor = null;
            }
        }
        object = appsLock;
        synchronized (object) {
            instances.remove(this.name);
        }
    }

    private void checkNotDeleted() {
        Preconditions.checkState((!this.deleted.get() ? 1 : 0) != 0, (String)"FirebaseApp '%s' was deleted", (Object[])new Object[]{this.name});
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private ScheduledExecutorService ensureScheduledExecutorService() {
        if (this.scheduledExecutor == null) {
            Object object = this.lock;
            synchronized (object) {
                this.checkNotDeleted();
                if (this.scheduledExecutor == null) {
                    this.scheduledExecutor = new FirebaseScheduledExecutor(this.getThreadFactory(), "firebase-scheduled-worker");
                }
            }
        }
        return this.scheduledExecutor;
    }

    ThreadFactory getThreadFactory() {
        return this.threadManager.getThreadFactory();
    }

    ScheduledExecutorService getScheduledExecutorService() {
        return this.ensureScheduledExecutorService();
    }

    <T> ApiFuture<T> submit(Callable<T> command) {
        Preconditions.checkNotNull(command);
        return new ListenableFuture2ApiFuture(this.executors.getListeningExecutor().submit(command));
    }

    <T> ScheduledFuture<T> schedule(Callable<T> command, long delayMillis) {
        Preconditions.checkNotNull(command);
        try {
            return this.ensureScheduledExecutorService().schedule(command, delayMillis, TimeUnit.MILLISECONDS);
        }
        catch (Exception e) {
            throw new UnsupportedOperationException("Scheduled tasks not supported", e);
        }
    }

    ScheduledFuture<?> schedule(Runnable runnable, long delayMillis) {
        Preconditions.checkNotNull((Object)runnable);
        try {
            return this.ensureScheduledExecutorService().schedule(runnable, delayMillis, TimeUnit.MILLISECONDS);
        }
        catch (Exception e) {
            throw new UnsupportedOperationException("Scheduled tasks not supported", e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void startTokenRefresher() {
        Object object = this.lock;
        synchronized (object) {
            this.checkNotDeleted();
            this.tokenRefresher.start();
        }
    }

    boolean isDefaultApp() {
        return DEFAULT_APP_NAME.equals(this.getName());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void addService(FirebaseService service) {
        Object object = this.lock;
        synchronized (object) {
            this.checkNotDeleted();
            Preconditions.checkArgument((!this.services.containsKey(((FirebaseService)Preconditions.checkNotNull((Object)service)).getId()) ? 1 : 0) != 0);
            this.services.put(service.getId(), service);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    FirebaseService getService(String id) {
        Object object = this.lock;
        synchronized (object) {
            Preconditions.checkArgument((!Strings.isNullOrEmpty((String)id) ? 1 : 0) != 0);
            return this.services.get(id);
        }
    }

    private static FirebaseOptions getOptionsFromEnvironment() throws IOException {
        JsonParser parser;
        String defaultConfig = System.getenv(FIREBASE_CONFIG_ENV_VAR);
        if (Strings.isNullOrEmpty((String)defaultConfig)) {
            return FirebaseOptions.builder().setCredentials(FirebaseOptions.APPLICATION_DEFAULT_CREDENTIALS).build();
        }
        JsonFactory jsonFactory = Utils.getDefaultJsonFactory();
        FirebaseOptions.Builder builder = FirebaseOptions.builder();
        if (defaultConfig.startsWith("{")) {
            parser = jsonFactory.createJsonParser(defaultConfig);
        } else {
            FileReader reader = new FileReader(defaultConfig);
            parser = jsonFactory.createJsonParser((Reader)reader);
        }
        parser.parseAndClose((Object)builder);
        builder.setCredentials(FirebaseOptions.APPLICATION_DEFAULT_CREDENTIALS);
        return builder.build();
    }

    static class TokenRefresher
    implements OAuth2Credentials.CredentialsChangedListener {
        private final FirebaseApp firebaseApp;
        private final GoogleCredentials credentials;
        private final AtomicReference<State> state;
        private Future<Void> future;

        TokenRefresher(FirebaseApp firebaseApp) {
            this.firebaseApp = (FirebaseApp)Preconditions.checkNotNull((Object)firebaseApp);
            this.credentials = firebaseApp.getOptions().getCredentials();
            this.state = new AtomicReference<State>(State.READY);
        }

        public final synchronized void onChanged(OAuth2Credentials credentials) {
            if (this.state.get() != State.STARTED) {
                return;
            }
            AccessToken accessToken = credentials.getAccessToken();
            long refreshDelay = this.getRefreshDelay(accessToken);
            if (refreshDelay > 0L) {
                this.scheduleRefresh(refreshDelay);
            } else {
                logger.warn("Token expiry ({}) is less than 5 minutes in the future. Not scheduling a proactive refresh.", (Object)accessToken.getExpirationTime());
            }
        }

        protected void cancelPrevious() {
            if (this.future != null) {
                this.future.cancel(true);
            }
        }

        protected void scheduleNext(Callable<Void> task, long delayMillis) {
            logger.debug("Scheduling next token refresh in {} milliseconds", (Object)delayMillis);
            try {
                this.future = this.firebaseApp.schedule(task, delayMillis);
            }
            catch (UnsupportedOperationException e) {
                logger.debug("Failed to schedule token refresh event", (Throwable)e);
            }
        }

        final synchronized void start() {
            if (!this.state.compareAndSet(State.READY, State.STARTED)) {
                return;
            }
            logger.debug("Starting the proactive token refresher");
            this.credentials.addChangeListener((OAuth2Credentials.CredentialsChangedListener)this);
            AccessToken accessToken = this.credentials.getAccessToken();
            long refreshDelay = accessToken != null ? Math.max(this.getRefreshDelay(accessToken), 0L) : 0L;
            this.scheduleRefresh(refreshDelay);
        }

        final synchronized void stop() {
            State previous = this.state.getAndSet(State.STOPPED);
            if (previous == State.STARTED) {
                this.cancelPrevious();
                logger.debug("Stopped the proactive token refresher");
            }
        }

        private void scheduleRefresh(long delayMillis) {
            this.cancelPrevious();
            this.scheduleNext(new Callable<Void>(){

                @Override
                public Void call() throws Exception {
                    logger.debug("Refreshing OAuth2 credential");
                    TokenRefresher.this.credentials.refresh();
                    return null;
                }
            }, delayMillis);
        }

        private long getRefreshDelay(AccessToken accessToken) {
            return accessToken.getExpirationTime().getTime() - System.currentTimeMillis() - TimeUnit.MINUTES.toMillis(5L);
        }

        static enum State {
            READY,
            STARTED,
            STOPPED;

        }

        static class Factory {
            Factory() {
            }

            TokenRefresher create(FirebaseApp app) {
                return new TokenRefresher(app);
            }
        }
    }
}

