/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.auth;

import com.google.api.client.auth.openidconnect.IdToken;
import com.google.api.client.auth.openidconnect.IdTokenVerifier;
import com.google.api.client.googleapis.auth.oauth2.GooglePublicKeysManager;
import com.google.api.client.json.JsonFactory;
import com.google.api.client.json.webtoken.JsonWebSignature;
import com.google.api.client.util.ArrayMap;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.firebase.auth.FirebaseAuthException;
import com.google.firebase.auth.FirebaseToken;
import com.google.firebase.auth.FirebaseTokenVerifier;
import java.io.IOException;
import java.math.BigDecimal;
import java.security.GeneralSecurityException;
import java.security.PublicKey;
import java.util.Map;

final class FirebaseTokenVerifierImpl
implements FirebaseTokenVerifier {
    private static final String RS256 = "RS256";
    private static final String FIREBASE_AUDIENCE = "https://identitytoolkit.googleapis.com/google.identity.identitytoolkit.v1.IdentityToolkit";
    private static final String ERROR_INVALID_CREDENTIAL = "ERROR_INVALID_CREDENTIAL";
    private static final String ERROR_RUNTIME_EXCEPTION = "ERROR_RUNTIME_EXCEPTION";
    private final JsonFactory jsonFactory;
    private final GooglePublicKeysManager publicKeysManager;
    private final IdTokenVerifier idTokenVerifier;
    private final String method;
    private final String shortName;
    private final String articledShortName;
    private final String docUrl;

    private FirebaseTokenVerifierImpl(Builder builder) {
        this.jsonFactory = (JsonFactory)Preconditions.checkNotNull((Object)builder.jsonFactory);
        this.publicKeysManager = (GooglePublicKeysManager)Preconditions.checkNotNull((Object)builder.publicKeysManager);
        this.idTokenVerifier = (IdTokenVerifier)Preconditions.checkNotNull((Object)builder.idTokenVerifier);
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)builder.method) ? 1 : 0) != 0, (Object)"method name must be specified");
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)builder.shortName) ? 1 : 0) != 0, (Object)"shortName must be specified");
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)builder.docUrl) ? 1 : 0) != 0, (Object)"docUrl must be specified");
        this.method = builder.method;
        this.shortName = builder.shortName;
        this.articledShortName = this.prefixWithIndefiniteArticle(this.shortName);
        this.docUrl = builder.docUrl;
    }

    @Override
    public FirebaseToken verifyToken(String token) throws FirebaseAuthException {
        IdToken idToken = this.parse(token);
        this.checkContents(idToken);
        this.checkSignature(idToken);
        return new FirebaseToken((Map<String, Object>)idToken.getPayload());
    }

    GooglePublicKeysManager getPublicKeysManager() {
        return this.publicKeysManager;
    }

    IdTokenVerifier getIdTokenVerifier() {
        return this.idTokenVerifier;
    }

    String getMethod() {
        return this.method;
    }

    String getShortName() {
        return this.shortName;
    }

    String getArticledShortName() {
        return this.articledShortName;
    }

    String getDocUrl() {
        return this.docUrl;
    }

    private String prefixWithIndefiniteArticle(String word) {
        if ("aeiouAEIOU".indexOf(word.charAt(0)) < 0) {
            return "a " + word;
        }
        return "an " + word;
    }

    private IdToken parse(String token) throws FirebaseAuthException {
        try {
            return IdToken.parse((JsonFactory)this.jsonFactory, (String)token);
        }
        catch (IOException e) {
            String detailedError = String.format("Failed to parse Firebase %s. Make sure you passed a string that represents a complete and valid JWT. See %s for details on how to retrieve %s.", this.shortName, this.docUrl, this.articledShortName);
            throw new FirebaseAuthException(ERROR_INVALID_CREDENTIAL, detailedError, e);
        }
    }

    private void checkContents(IdToken token) throws FirebaseAuthException {
        String errorMessage = this.getErrorIfContentInvalid(token);
        if (errorMessage != null) {
            String detailedError = String.format("%s %s", errorMessage, this.getVerifyTokenMessage());
            throw new FirebaseAuthException(ERROR_INVALID_CREDENTIAL, detailedError);
        }
    }

    private void checkSignature(IdToken token) throws FirebaseAuthException {
        try {
            if (!this.isSignatureValid(token)) {
                throw new FirebaseAuthException(ERROR_INVALID_CREDENTIAL, String.format("Failed to verify the signature of Firebase %s. %s", this.shortName, this.getVerifyTokenMessage()));
            }
        }
        catch (IOException | GeneralSecurityException e) {
            throw new FirebaseAuthException(ERROR_RUNTIME_EXCEPTION, "Error while verifying signature.", e);
        }
    }

    private String getErrorIfContentInvalid(IdToken idToken) {
        JsonWebSignature.Header header = idToken.getHeader();
        IdToken.Payload payload = idToken.getPayload();
        String errorMessage = null;
        if (header.getKeyId() == null) {
            errorMessage = this.getErrorForTokenWithoutKid(header, payload);
        } else if (!RS256.equals(header.getAlgorithm())) {
            errorMessage = String.format("Firebase %s has incorrect algorithm. Expected \"%s\" but got \"%s\".", this.shortName, RS256, header.getAlgorithm());
        } else if (!idToken.verifyAudience(this.idTokenVerifier.getAudience())) {
            errorMessage = String.format("Firebase %s has incorrect \"aud\" (audience) claim. Expected \"%s\" but got \"%s\". %s", this.shortName, this.joinWithComma(this.idTokenVerifier.getAudience()), this.joinWithComma(payload.getAudienceAsList()), this.getProjectIdMatchMessage());
        } else if (!idToken.verifyIssuer(this.idTokenVerifier.getIssuers())) {
            errorMessage = String.format("Firebase %s has incorrect \"iss\" (issuer) claim. Expected \"%s\" but got \"%s\". %s", this.shortName, this.joinWithComma(this.idTokenVerifier.getIssuers()), payload.getIssuer(), this.getProjectIdMatchMessage());
        } else if (payload.getSubject() == null) {
            errorMessage = String.format("Firebase %s has no \"sub\" (subject) claim.", this.shortName);
        } else if (payload.getSubject().isEmpty()) {
            errorMessage = String.format("Firebase %s has an empty string \"sub\" (subject) claim.", this.shortName);
        } else if (payload.getSubject().length() > 128) {
            errorMessage = String.format("Firebase %s has \"sub\" (subject) claim longer than 128 characters.", this.shortName);
        } else if (!this.verifyTimestamps(idToken)) {
            errorMessage = String.format("Firebase %s has expired or is not yet valid. Get a fresh %s and try again.", this.shortName, this.shortName);
        }
        return errorMessage;
    }

    private String getVerifyTokenMessage() {
        return String.format("See %s for details on how to retrieve %s.", this.docUrl, this.articledShortName);
    }

    private boolean isSignatureValid(IdToken token) throws GeneralSecurityException, IOException {
        for (PublicKey key : this.publicKeysManager.getPublicKeys()) {
            if (!token.verifySignature(key)) continue;
            return true;
        }
        return false;
    }

    private String getErrorForTokenWithoutKid(JsonWebSignature.Header header, IdToken.Payload payload) {
        if (this.isCustomToken(payload)) {
            return String.format("%s expects %s, but was given a custom token.", this.method, this.articledShortName);
        }
        if (this.isLegacyCustomToken(header, payload)) {
            return String.format("%s expects %s, but was given a legacy custom token.", this.method, this.articledShortName);
        }
        return String.format("Firebase %s has no \"kid\" claim.", this.shortName);
    }

    private String joinWithComma(Iterable<String> strings) {
        return Joiner.on((char)',').join(strings);
    }

    private String getProjectIdMatchMessage() {
        return String.format("Make sure the %s comes from the same Firebase project as the service account used to authenticate this SDK.", this.shortName);
    }

    private boolean verifyTimestamps(IdToken token) {
        long currentTimeMillis = this.idTokenVerifier.getClock().currentTimeMillis();
        return token.verifyTime(currentTimeMillis, this.idTokenVerifier.getAcceptableTimeSkewSeconds());
    }

    private boolean isCustomToken(IdToken.Payload payload) {
        return FIREBASE_AUDIENCE.equals(payload.getAudience());
    }

    private boolean isLegacyCustomToken(JsonWebSignature.Header header, IdToken.Payload payload) {
        return "HS256".equals(header.getAlgorithm()) && new BigDecimal(0).equals(payload.get((Object)"v")) && this.containsLegacyUidField(payload);
    }

    private boolean containsLegacyUidField(IdToken.Payload payload) {
        Object dataField = payload.get((Object)"d");
        if (dataField instanceof ArrayMap) {
            return ((ArrayMap)dataField).get((Object)"uid") != null;
        }
        return false;
    }

    static Builder builder() {
        return new Builder();
    }

    static final class Builder {
        private JsonFactory jsonFactory;
        private GooglePublicKeysManager publicKeysManager;
        private String method;
        private String shortName;
        private IdTokenVerifier idTokenVerifier;
        private String docUrl;

        private Builder() {
        }

        Builder setJsonFactory(JsonFactory jsonFactory) {
            this.jsonFactory = jsonFactory;
            return this;
        }

        Builder setPublicKeysManager(GooglePublicKeysManager publicKeysManager) {
            this.publicKeysManager = publicKeysManager;
            return this;
        }

        Builder setMethod(String method) {
            this.method = method;
            return this;
        }

        Builder setShortName(String shortName) {
            this.shortName = shortName;
            return this;
        }

        Builder setIdTokenVerifier(IdTokenVerifier idTokenVerifier) {
            this.idTokenVerifier = idTokenVerifier;
            return this;
        }

        Builder setDocUrl(String docUrl) {
            this.docUrl = docUrl;
            return this;
        }

        FirebaseTokenVerifierImpl build() {
            return new FirebaseTokenVerifierImpl(this);
        }
    }
}

