/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.database.connection;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.firebase.database.connection.ConnectionContext;
import com.google.firebase.database.connection.HostInfo;
import com.google.firebase.database.connection.NettyWebSocketClient;
import com.google.firebase.database.logging.LogWrapper;
import com.google.firebase.database.util.JsonMapper;
import java.io.EOFException;
import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

class WebsocketConnection {
    private static final long KEEP_ALIVE_TIMEOUT_MS = 45000L;
    private static final long CONNECT_TIMEOUT_MS = 30000L;
    private static final int MAX_FRAME_SIZE = 16384;
    private static final AtomicLong CONN_ID = new AtomicLong(0L);
    private final ScheduledExecutorService executorService;
    private final LogWrapper logger;
    private final WSClient conn;
    private final Delegate delegate;
    private StringList buffer;
    private boolean everConnected = false;
    private boolean isClosed = false;
    private ScheduledFuture<?> keepAlive;
    private ScheduledFuture<?> connectTimeout;

    WebsocketConnection(ConnectionContext connectionContext, HostInfo hostInfo, String optCachedHost, Delegate delegate, String optLastSessionId) {
        this(connectionContext, delegate, new DefaultWSClientFactory(connectionContext, hostInfo, optCachedHost, optLastSessionId));
    }

    WebsocketConnection(ConnectionContext connectionContext, Delegate delegate, WSClientFactory clientFactory) {
        this.executorService = connectionContext.getExecutorService();
        this.delegate = delegate;
        this.logger = new LogWrapper(connectionContext.getLogger(), WebsocketConnection.class, "ws_" + CONN_ID.getAndIncrement());
        this.conn = clientFactory.newClient(new WSClientHandlerImpl());
    }

    void open() {
        this.conn.connect();
        this.connectTimeout = this.executorService.schedule(new Runnable(){

            @Override
            public void run() {
                WebsocketConnection.this.closeIfNeverConnected();
            }
        }, 30000L, TimeUnit.MILLISECONDS);
    }

    void start() {
    }

    void close() {
        if (this.logger.logsDebug()) {
            this.logger.debug("websocket is being closed", new Object[0]);
        }
        this.isClosed = true;
        this.conn.close();
        if (this.connectTimeout != null) {
            this.connectTimeout.cancel(true);
            this.connectTimeout = null;
        }
        if (this.keepAlive != null) {
            this.keepAlive.cancel(true);
            this.keepAlive = null;
        }
    }

    void send(Map<String, Object> message) {
        this.resetKeepAlive();
        try {
            String toSend = JsonMapper.serializeJson(message);
            List<String> frames = this.splitIntoFrames(toSend, 16384);
            if (frames.size() > 1) {
                this.conn.send("" + frames.size());
            }
            for (String seg : frames) {
                this.conn.send(seg);
            }
        }
        catch (IOException e) {
            this.logger.error("Failed to serialize message: " + message.toString(), e);
            this.closeAndNotify();
        }
    }

    private List<String> splitIntoFrames(String src, int maxFrameSize) {
        if (src.length() <= maxFrameSize) {
            return ImmutableList.of((Object)src);
        }
        ImmutableList.Builder frames = ImmutableList.builder();
        for (int i = 0; i < src.length(); i += maxFrameSize) {
            int end = Math.min(i + maxFrameSize, src.length());
            String seg = src.substring(i, end);
            frames.add((Object)seg);
        }
        return frames.build();
    }

    private void handleNewFrameCount(int numFrames) {
        if (this.logger.logsDebug()) {
            this.logger.debug("HandleNewFrameCount: " + numFrames, new Object[0]);
        }
        this.buffer = new StringList(numFrames);
    }

    private void appendFrame(String message) {
        int framesRemaining = this.buffer.append(message);
        if (framesRemaining > 0) {
            return;
        }
        String combined = this.buffer.combine();
        try {
            Map<String, Object> decoded = JsonMapper.parseJson(combined);
            if (this.logger.logsDebug()) {
                this.logger.debug("handleIncomingFrame complete frame: " + decoded, new Object[0]);
            }
            this.delegate.onMessage(decoded);
        }
        catch (IOException e) {
            this.logger.error("Error parsing frame: " + combined, e);
            this.closeAndNotify();
        }
        catch (ClassCastException e) {
            this.logger.error("Error parsing frame (cast error): " + combined, e);
            this.closeAndNotify();
        }
    }

    private String extractFrameCount(String message) {
        if (message.length() <= 6) {
            try {
                int frameCount = Integer.parseInt(message);
                if (frameCount > 0) {
                    this.handleNewFrameCount(frameCount);
                }
                return null;
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
        this.handleNewFrameCount(1);
        return message;
    }

    private void handleIncomingFrame(String message) {
        if (this.isClosed) {
            return;
        }
        this.resetKeepAlive();
        if (this.buffer != null && this.buffer.hasRemaining()) {
            this.appendFrame(message);
        } else {
            String remaining = this.extractFrameCount(message);
            if (remaining != null) {
                this.appendFrame(remaining);
            }
        }
    }

    private void resetKeepAlive() {
        if (this.isClosed) {
            return;
        }
        if (this.keepAlive != null) {
            this.keepAlive.cancel(false);
            if (this.logger.logsDebug()) {
                this.logger.debug("Reset keepAlive. Remaining: " + this.keepAlive.getDelay(TimeUnit.MILLISECONDS), new Object[0]);
            }
        } else if (this.logger.logsDebug()) {
            this.logger.debug("Reset keepAlive", new Object[0]);
        }
        this.keepAlive = this.executorService.schedule(this.nop(), 45000L, TimeUnit.MILLISECONDS);
    }

    private Runnable nop() {
        return new Runnable(){

            @Override
            public void run() {
                if (WebsocketConnection.this.conn != null) {
                    WebsocketConnection.this.conn.send("0");
                    WebsocketConnection.this.resetKeepAlive();
                }
            }
        };
    }

    private void closeAndNotify() {
        if (!this.isClosed) {
            this.close();
            this.delegate.onDisconnect(this.everConnected);
        }
    }

    private void onClosed() {
        if (!this.isClosed) {
            if (this.logger.logsDebug()) {
                this.logger.debug("closing itself", new Object[0]);
            }
            this.closeAndNotify();
        }
    }

    private void closeIfNeverConnected() {
        if (!this.everConnected && !this.isClosed) {
            if (this.logger.logsDebug()) {
                this.logger.debug("timed out on connect", new Object[0]);
            }
            this.closeAndNotify();
        }
    }

    static interface WSClientEventHandler {
        public void onOpen();

        public void onMessage(String var1);

        public void onClose();

        public void onError(Throwable var1);
    }

    private static class DefaultWSClientFactory
    implements WSClientFactory {
        final ConnectionContext context;
        final HostInfo hostInfo;
        final String optCachedHost;
        final String optLastSessionId;

        DefaultWSClientFactory(ConnectionContext context, HostInfo hostInfo, String optCachedHost, String optLastSessionId) {
            this.context = context;
            this.hostInfo = hostInfo;
            this.optCachedHost = optCachedHost;
            this.optLastSessionId = optLastSessionId;
        }

        @Override
        public WSClient newClient(WSClientEventHandler delegate) {
            String host = this.optCachedHost != null ? this.optCachedHost : this.hostInfo.getHost();
            URI uri = HostInfo.getConnectionUrl(host, this.hostInfo.isSecure(), this.hostInfo.getNamespace(), this.optLastSessionId);
            return new NettyWebSocketClient(uri, this.context.getUserAgent(), this.context.getThreadFactory(), delegate);
        }
    }

    static interface WSClientFactory {
        public WSClient newClient(WSClientEventHandler var1);
    }

    static interface WSClient {
        public void connect();

        public void close();

        public void send(String var1);
    }

    static interface Delegate {
        public void onMessage(Map<String, Object> var1);

        public void onDisconnect(boolean var1);
    }

    private static class StringList {
        private final List<String> buffer;
        private int remaining;

        StringList(int capacity) {
            Preconditions.checkArgument((capacity > 0 ? 1 : 0) != 0);
            this.buffer = new ArrayList<String>(capacity);
            this.remaining = capacity;
        }

        int append(String frame) {
            Preconditions.checkState((boolean)this.hasRemaining());
            this.buffer.add(frame);
            return --this.remaining;
        }

        boolean hasRemaining() {
            return this.remaining > 0;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        String combine() {
            Preconditions.checkState((!this.hasRemaining() ? 1 : 0) != 0);
            try {
                StringBuilder sb = new StringBuilder();
                for (String frame : this.buffer) {
                    sb.append(frame);
                }
                String string = sb.toString();
                return string;
            }
            finally {
                this.buffer.clear();
            }
        }
    }

    private class WSClientHandlerImpl
    implements WSClientEventHandler {
        private WSClientHandlerImpl() {
        }

        @Override
        public void onOpen() {
            if (WebsocketConnection.this.logger.logsDebug()) {
                WebsocketConnection.this.logger.debug("websocket opened", new Object[0]);
            }
            WebsocketConnection.this.executorService.execute(new Runnable(){

                @Override
                public void run() {
                    WebsocketConnection.this.connectTimeout.cancel(false);
                    WebsocketConnection.this.everConnected = true;
                    WebsocketConnection.this.resetKeepAlive();
                }
            });
        }

        @Override
        public void onMessage(final String message) {
            if (WebsocketConnection.this.logger.logsDebug()) {
                WebsocketConnection.this.logger.debug("ws message: " + message, new Object[0]);
            }
            WebsocketConnection.this.executorService.execute(new Runnable(){

                @Override
                public void run() {
                    WebsocketConnection.this.handleIncomingFrame(message);
                }
            });
        }

        @Override
        public void onClose() {
            if (WebsocketConnection.this.logger.logsDebug()) {
                WebsocketConnection.this.logger.debug("closed", new Object[0]);
            }
            if (!WebsocketConnection.this.isClosed) {
                WebsocketConnection.this.executorService.execute(new Runnable(){

                    @Override
                    public void run() {
                        WebsocketConnection.this.onClosed();
                    }
                });
            }
        }

        @Override
        public void onError(Throwable e) {
            if (e.getCause() != null && e.getCause() instanceof EOFException) {
                WebsocketConnection.this.logger.error("WebSocket reached EOF", e);
            } else {
                WebsocketConnection.this.logger.error("WebSocket error", e);
            }
            WebsocketConnection.this.executorService.execute(new Runnable(){

                @Override
                public void run() {
                    WebsocketConnection.this.onClosed();
                }
            });
        }
    }
}

