/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.ir.inline

import ksp.org.jetbrains.kotlin.backend.common.LoweringContext
import ksp.org.jetbrains.kotlin.backend.common.phaser.PhaseDescription
import ksp.org.jetbrains.kotlin.ir.IrElement
import ksp.org.jetbrains.kotlin.ir.IrStatement
import ksp.org.jetbrains.kotlin.ir.declarations.IrDeclarationParent
import ksp.org.jetbrains.kotlin.ir.declarations.IrFunction
import ksp.org.jetbrains.kotlin.ir.expressions.IrFunctionReference
import ksp.org.jetbrains.kotlin.ir.util.isInlineArrayConstructor
import ksp.org.jetbrains.kotlin.ir.util.isInlineParameter

/**
 * Transforms all callable references (including defaults) to inline lambdas, marks inline lambdas for later passes.
 */
@PhaseDescription("CommonInlineCallableReferenceToLambdaPhase")
open class CommonInlineCallableReferenceToLambdaPhase(
    context: LoweringContext,
    inlineFunctionResolver: InlineFunctionResolver
) : InlineCallableReferenceToLambdaPhase(context, inlineFunctionResolver) {
    fun lower(function: IrFunction) = function.accept(this, function.parent)

    override fun visitFunction(declaration: IrFunction, data: IrDeclarationParent?): IrStatement {
        super.visitFunction(declaration, data)
        if (inlineFunctionResolver.needsInlining(declaration)) {
            for (parameter in declaration.parameters) {
                if (parameter.isInlineParameter()) {
                    val defaultExpression = parameter.defaultValue?.expression ?: continue
                    parameter.defaultValue?.expression = defaultExpression.transformToLambda(declaration)
                }
            }
        }

        return declaration
    }

    override fun visitFunctionReference(expression: IrFunctionReference, data: IrDeclarationParent?): IrElement {
        super.visitFunctionReference(expression, data)

        val owner = expression.symbol.owner
        if (!owner.isInlineArrayConstructor()) return expression

        return expression.transformToLambda(data)
    }
}
