/*
 * Copyright 2010-2022 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.ir.backend.js

import ksp.org.jetbrains.kotlin.config.phaser.PhaserState
import ksp.org.jetbrains.kotlin.ir.backend.js.dce.DceDumpNameCache
import ksp.org.jetbrains.kotlin.ir.backend.js.dce.eliminateDeadDeclarations
import ksp.org.jetbrains.kotlin.ir.backend.js.transformers.irToJs.JsIrProgramFragment
import ksp.org.jetbrains.kotlin.ir.backend.js.utils.JsStaticContext
import ksp.org.jetbrains.kotlin.ir.declarations.IrModuleFragment
import ksp.org.jetbrains.kotlin.js.backend.ast.JsClass
import ksp.org.jetbrains.kotlin.js.backend.ast.JsFunction
import ksp.org.jetbrains.kotlin.js.backend.ast.RecursiveJsVisitor
import ksp.org.jetbrains.kotlin.js.inline.clean.ClassPostProcessor
import ksp.org.jetbrains.kotlin.js.inline.clean.FunctionPostProcessor
import ksp.org.jetbrains.kotlin.serialization.js.ModuleKind

fun optimizeProgramByIr(
    modules: Iterable<IrModuleFragment>,
    context: JsIrBackendContext,
    moduleKind: ModuleKind,
    removeUnusedAssociatedObjects: Boolean
) {
    val dceDumpNameCache = DceDumpNameCache() // in JS mode only DCE Graph could be dumped
    eliminateDeadDeclarations(modules, context, moduleKind, removeUnusedAssociatedObjects, dceDumpNameCache)

    val phaserState = PhaserState()
    optimizationLoweringList.forEachIndexed { _, lowering ->
        modules.forEach { module ->
            lowering.invoke(context.phaseConfig, phaserState, context, module)
        }
    }
}

fun optimizeFragmentByJsAst(fragment: JsIrProgramFragment, context: JsStaticContext) {
    val voidName = context.backendContext.intrinsics.void.owner.backingField?.let(context::getNameForField)

    val optimizer = object : RecursiveJsVisitor() {
        override fun visitFunction(x: JsFunction) {
            super.visitFunction(x)
            FunctionPostProcessor(x, voidName).apply()
        }

        override fun visitClass(x: JsClass) {
            super.visitClass(x)
            ClassPostProcessor(x).apply()
        }
    }

    fragment.declarations.statements.forEach { it.accept(optimizer) }
    fragment.classes.values.forEach { klass ->
        klass.postDeclarationBlock.statements.forEach { it.accept(optimizer)}
        klass.preDeclarationBlock.statements.forEach { it.accept(optimizer)}
    }
}
