/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.scopes.impl

import ksp.org.jetbrains.kotlin.fir.FirSession
import ksp.org.jetbrains.kotlin.fir.canHaveDeferredReturnTypeCalculation
import ksp.org.jetbrains.kotlin.fir.declarations.FirCallableDeclaration
import ksp.org.jetbrains.kotlin.fir.resolve.ScopeSession
import ksp.org.jetbrains.kotlin.fir.scopes.CallableCopyTypeCalculator
import ksp.org.jetbrains.kotlin.fir.scopes.DelicateScopeAPI
import ksp.org.jetbrains.kotlin.fir.scopes.FirDelegatingTypeScope
import ksp.org.jetbrains.kotlin.fir.scopes.FirTypeScope
import ksp.org.jetbrains.kotlin.fir.scopes.ProcessorAction
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirNamedFunctionSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirPropertySymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirVariableSymbol
import ksp.org.jetbrains.kotlin.name.Name

/**
 * This scope is a wrapper which is intended to use with scopes that can create callable copies.
 *
 * The main purpose of this scope is to update dispatched callables return types
 * in case it is not yet calculated due to implicit body resolve logic.
 */
class FirScopeWithCallableCopyReturnTypeUpdater(
    private val delegate: FirTypeScope,
    private val callableCopyTypeCalculator: CallableCopyTypeCalculator
) : FirDelegatingTypeScope(delegate) {
    override fun processFunctionsByName(name: Name, processor: (FirNamedFunctionSymbol) -> Unit) {
        delegate.processFunctionsByName(name) {
            updateReturnType(it.fir)
            processor(it)
        }
    }

    override fun processPropertiesByName(name: Name, processor: (FirVariableSymbol<*>) -> Unit) {
        delegate.processPropertiesByName(name) {
            updateReturnType(it.fir)
            processor(it)
        }
    }

    override fun processDirectOverriddenFunctionsWithBaseScope(
        functionSymbol: FirNamedFunctionSymbol,
        processor: (FirNamedFunctionSymbol, FirTypeScope) -> ProcessorAction
    ): ProcessorAction {
        return delegate.processDirectOverriddenFunctionsWithBaseScope(functionSymbol) { symbol, scope ->
            updateReturnType(symbol.fir)
            processor(symbol, scope)
        }
    }

    override fun processDirectOverriddenPropertiesWithBaseScope(
        propertySymbol: FirPropertySymbol,
        processor: (FirPropertySymbol, FirTypeScope) -> ProcessorAction
    ): ProcessorAction {
        return delegate.processDirectOverriddenPropertiesWithBaseScope(propertySymbol) { symbol, scope ->
            updateReturnType(symbol.fir)
            processor(symbol, scope)
        }
    }

    private fun updateReturnType(declaration: FirCallableDeclaration) {
        if (declaration.canHaveDeferredReturnTypeCalculation) {
            callableCopyTypeCalculator.computeReturnType(declaration)
        }
    }

    override fun toString(): String {
        return delegate.toString()
    }

    @DelicateScopeAPI
    override fun withReplacedSessionOrNull(
        newSession: FirSession,
        newScopeSession: ScopeSession
    ): FirScopeWithCallableCopyReturnTypeUpdater? {
        return delegate.withReplacedSessionOrNull(newSession, newScopeSession)?.let {
            FirScopeWithCallableCopyReturnTypeUpdater(delegate, callableCopyTypeCalculator)
        }
    }
}
