/*
 * Copyright 2010-2023 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.js.checkers.declaration

import ksp.org.jetbrains.kotlin.KtNodeTypes
import ksp.org.jetbrains.kotlin.config.LanguageFeature
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.declaration.FirFileChecker
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.fir.analysis.forEachChildOfType
import ksp.org.jetbrains.kotlin.fir.analysis.js.checkers.sanitizeName
import ksp.org.jetbrains.kotlin.fir.declarations.FirFile
import ksp.org.jetbrains.kotlin.fir.packageFqName
import ksp.org.jetbrains.kotlin.psi.KtPsiUtil
import ksp.org.jetbrains.kotlin.text

object FirJsPackageDirectiveChecker: FirFileChecker(MppCheckerKind.Common) {
    // inspired by FirJsNameCharsChecker.check()
    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(declaration: FirFile) {
        if (declaration.packageFqName.isRoot) return
        if (context.languageVersionSettings.supportsFeature(LanguageFeature.JsAllowInvalidCharsIdentifiersEscaping)) return

        declaration.packageDirective.source?.forEachChildOfType(setOf(KtNodeTypes.REFERENCE_EXPRESSION), depth = -1) {
            val name = KtPsiUtil.unquoteIdentifier(it.text.toString())
            if (sanitizeName(name) != name) {
                reporter.reportOn(
                    it,
                    FirErrors.INVALID_CHARACTERS,
                    "$name contains illegal characters"
                )
            }
        }
    }
}
