/*
 * Copyright 2010-2023 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.js.checkers.declaration

import ksp.org.jetbrains.kotlin.builtins.StandardNames
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.declaration.FirClassChecker
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.overriddenFunctions
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.unsubstitutedScope
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.js.FirJsErrors
import ksp.org.jetbrains.kotlin.fir.declarations.FirClass
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isExpect
import ksp.org.jetbrains.kotlin.fir.resolve.defaultType
import ksp.org.jetbrains.kotlin.fir.scopes.getFunctions
import ksp.org.jetbrains.kotlin.fir.types.ConeKotlinType
import ksp.org.jetbrains.kotlin.fir.types.classId
import ksp.org.jetbrains.kotlin.fir.types.isSubtypeOf
import ksp.org.jetbrains.kotlin.name.Name
import ksp.org.jetbrains.kotlin.util.OperatorNameConventions

sealed class FirJsMultipleInheritanceChecker(mppKind: MppCheckerKind) : FirClassChecker(mppKind) {
    object Regular : FirJsMultipleInheritanceChecker(MppCheckerKind.Platform) {
        context(context: CheckerContext, reporter: DiagnosticReporter)
        override fun check(declaration: FirClass) {
            if (declaration.isExpect) return
            super.check(declaration)
        }
    }

    object ForExpectClass : FirJsMultipleInheritanceChecker(MppCheckerKind.Common) {
        context(context: CheckerContext, reporter: DiagnosticReporter)
        override fun check(declaration: FirClass) {
            if (!declaration.isExpect) return
            super.check(declaration)
        }
    }

    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(declaration: FirClass) {
        declaration.checkFunctionIfSubtypeOf(
            functionToCheck = OperatorNameConventions.GET,
            supertype = context.session.builtinTypes.charSequenceType.coneType,
            context, reporter,
        )

        declaration.checkFunctionIfSubtypeOf(
            functionToCheck = StandardNames.NEXT_CHAR,
            supertype = context.session.builtinTypes.charIteratorType.coneType,
            context, reporter,
        )
    }

    private fun FirClass.checkFunctionIfSubtypeOf(
        functionToCheck: Name,
        supertype: ConeKotlinType,
        context: CheckerContext,
        reporter: DiagnosticReporter,
    ) {
        if (!defaultType().isSubtypeOf(supertype, context.session)) {
            return
        }

        val scope = unsubstitutedScope(context)
        val overridesWithSameName = scope.getFunctions(functionToCheck)

        for (function in overridesWithSameName) {
            val overridden = function.overriddenFunctions(symbol, context)
            if (
                overridden.size > 1 &&
                overridden.any { it.callableId.classId == supertype.classId }
            ) {
                reporter.reportOn(source, FirJsErrors.WRONG_MULTIPLE_INHERITANCE, function, context)
            }
        }
    }
}
