/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.syntax

import ksp.com.intellij.psi.PsiElement
import ksp.org.jetbrains.kotlin.KtLightSourceElement
import ksp.org.jetbrains.kotlin.KtPsiSourceElement
import ksp.org.jetbrains.kotlin.KtSourceElement
import ksp.org.jetbrains.kotlin.fir.FirElement
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.declaration.FirDeclarationChecker
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression.FirExpressionChecker
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.declarations.FirDeclaration
import ksp.org.jetbrains.kotlin.fir.expressions.FirStatement

/**
 * Checker that heavily relies on source tree. So it may have different implementation for each tree variant. Subclass should either
 *
 * - implement `checkPsiOrLightTree` in an AST-agnostic manner, or
 * - implement both `checkPsi` and `checkLightTree` if it's difficult to handle PSI and LT tree in a unified way.
 */
interface FirSyntaxChecker<D : FirElement, P : PsiElement> {

    fun checkSyntax(element: D, context: CheckerContext, reporter: DiagnosticReporter) {
        val source = element.source ?: return
        if (!isApplicable(element, source)) return
        @Suppress("UNCHECKED_CAST")
        when (source) {
            is KtPsiSourceElement -> checkPsi(element, source, source.psi as P, context, reporter)
            is KtLightSourceElement -> checkLightTree(element, source, context, reporter)
        }
    }

    fun isApplicable(element: D, source: KtSourceElement): Boolean = true

    fun checkPsi(element: D, source: KtPsiSourceElement, psi: P, context: CheckerContext, reporter: DiagnosticReporter) {
        checkPsiOrLightTree(element, source, context, reporter)
    }

    fun checkLightTree(element: D, source: KtLightSourceElement, context: CheckerContext, reporter: DiagnosticReporter) {
        checkPsiOrLightTree(element, source, context, reporter)
    }

    /**
     *  By default psi tree should be equivalent to light tree and can be processed the same way.
     */
    fun checkPsiOrLightTree(element: D, source: KtSourceElement, context: CheckerContext, reporter: DiagnosticReporter) {}
}

abstract class FirDeclarationSyntaxChecker<D : FirDeclaration, P : PsiElement> :
    FirDeclarationChecker<D>(MppCheckerKind.Common),
    FirSyntaxChecker<D, P> {

    context(context: CheckerContext, reporter: DiagnosticReporter)
    final override fun check(declaration: D) {
        checkSyntax(declaration, context, reporter)
    }
}

abstract class FirExpressionSyntaxChecker<E : FirStatement, P : PsiElement> :
    FirExpressionChecker<E>(MppCheckerKind.Common),
    FirSyntaxChecker<E, P> {
    context(context: CheckerContext, reporter: DiagnosticReporter)
    final override fun check(expression: E) {
        checkSyntax(expression, context, reporter)
    }
}
