/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.experimental

import ksp.org.jetbrains.kotlin.KtFakeSourceElementKind
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression.FirFunctionCallChecker
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors
import ksp.org.jetbrains.kotlin.fir.expressions.FirLiteralExpression
import ksp.org.jetbrains.kotlin.fir.expressions.FirFunctionCall

object EmptyRangeChecker : FirFunctionCallChecker(MppCheckerKind.Common) {
    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(expression: FirFunctionCall) {
        if (expression.source?.kind is KtFakeSourceElementKind) return
        val left = expression.rangeLeft ?: return
        val right = expression.rangeRight ?: return

        val needReport = when (expression.calleeReference.name.asString()) {
            "rangeTo" -> {
                left > right
            }
            "downTo" -> {
                right > left
            }
            "until" -> {
                left >= right
            }
            else -> false
        }

        if (needReport) {
            reporter.reportOn(expression.source, FirErrors.EMPTY_RANGE)
        }
    }

    private val FirFunctionCall.rangeLeft: Long?
        get() {
            return (explicitReceiver as? FirLiteralExpression)?.value as? Long
        }

    private val FirFunctionCall.rangeRight: Long?
        get() {
            val arg = argumentList.arguments.getOrNull(0) as? FirLiteralExpression
            return arg?.value as? Long
        }
}
