/*
 * Copyright 2010-2023 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.declaration

import ksp.org.jetbrains.kotlin.KtRealSourceElementKind
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.checkMissingDependencySuperTypes
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression.FirMissingDependencyClassProxy
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression.FirMissingDependencyClassProxy.MissingTypeOrigin
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.getContainingClassSymbol
import ksp.org.jetbrains.kotlin.fir.correspondingProperty
import ksp.org.jetbrains.kotlin.fir.declarations.FirValueParameter
import ksp.org.jetbrains.kotlin.fir.declarations.utils.isData
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirAnonymousFunctionSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirConstructorSymbol
import ksp.org.jetbrains.kotlin.fir.symbols.impl.FirRegularClassSymbol
import ksp.org.jetbrains.kotlin.fir.types.ConeClassLikeType
import ksp.org.jetbrains.kotlin.fir.types.coneType
import ksp.org.jetbrains.kotlin.name.Name

/**
 * In this checker, we check for two cases of missing dependency type:
 * - implicit type of lambda parameter
 * ```
 * foo { x -> ...}
 * ```
 * - explicit type of data class constructor parameter
 * ```
 * data class Foo(val x: Some)
 * ```
 * The second check is required, as we might call `x.toString/hashCode/equals` in the implictly generated functions of data class,
 * which wouldn't be checked by use-site checkers
 */
object FirMissingDependencyClassForParameterChecker : FirValueParameterChecker(MppCheckerKind.Common), FirMissingDependencyClassProxy {
    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(declaration: FirValueParameter) {
        val containingDeclaration = declaration.containingDeclarationSymbol

        when {
            containingDeclaration is FirAnonymousFunctionSymbol -> {
                checkLambdaParameter(declaration)
            }
            declaration.correspondingProperty != null && containingDeclaration.getContainingClassSymbol()?.isData == true -> {
                checkDataClassParameter(declaration)
            }
        }
    }

    context(context: CheckerContext, reporter: DiagnosticReporter)
    private fun checkLambdaParameter(parameter: FirValueParameter) {
        if (parameter.returnTypeRef.source?.kind is KtRealSourceElementKind) return

        val missingTypes = mutableSetOf<ConeClassLikeType>()
        considerType(parameter.returnTypeRef.coneType, missingTypes, context)
        reportMissingTypes(
            parameter.source, missingTypes, context, reporter,
            missingTypeOrigin = MissingTypeOrigin.LambdaParameter(
                parameter.name.takeIf { !it.isSpecial } ?: Name.identifier("_")
            )
        )
    }

    context(context: CheckerContext, reporter: DiagnosticReporter)
    private fun checkDataClassParameter(parameter: FirValueParameter) {
        checkMissingDependencySuperTypes(parameter.returnTypeRef.coneType, parameter.source)
    }
}
