/*
 * Copyright 2010-2023 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.analysis.decompiler.psi

import ksp.com.intellij.openapi.application.ApplicationManager
import ksp.com.intellij.openapi.diagnostic.logger
import ksp.com.intellij.openapi.project.Project
import ksp.com.intellij.openapi.vfs.StandardFileSystems
import ksp.com.intellij.openapi.vfs.VirtualFile
import ksp.com.intellij.openapi.vfs.VirtualFileManager
import ksp.com.intellij.psi.search.GlobalSearchScope
import ksp.com.intellij.util.io.URLUtil
import ksp.org.jetbrains.kotlin.name.StandardClassIds
import ksp.org.jetbrains.kotlin.serialization.deserialization.builtins.BuiltInSerializerProtocol
import ksp.org.jetbrains.kotlin.utils.exceptions.errorWithAttachment
import java.net.URL

abstract class BuiltinsVirtualFileProvider {
    abstract fun getBuiltinVirtualFiles(): Set<VirtualFile>

    abstract fun createBuiltinsScope(project: Project): GlobalSearchScope

    companion object {
        fun getInstance(): BuiltinsVirtualFileProvider =
            ApplicationManager.getApplication().getService(BuiltinsVirtualFileProvider::class.java)
    }
}

abstract class BuiltinsVirtualFileProviderBaseImpl : BuiltinsVirtualFileProvider() {
    private val builtInUrls: Set<URL> by lazy {
        val classLoader = this::class.java.classLoader
        StandardClassIds.builtInsPackages.mapNotNullTo(mutableSetOf()) { builtInPackageFqName ->
            val resourcePath = BuiltInSerializerProtocol.getBuiltInsFilePath(builtInPackageFqName)
            classLoader.getResource(resourcePath)
        }
    }

    override fun createBuiltinsScope(project: Project): GlobalSearchScope {
        return GlobalSearchScope.filesScope(project, builtinVirtualFilesCached)
    }

    protected abstract fun findVirtualFile(url: URL): VirtualFile?

    private val builtinVirtualFilesCached: Set<VirtualFile> by lazy(LazyThreadSafetyMode.PUBLICATION) {
        builtInUrls.mapNotNullTo(mutableSetOf()) { url ->
            val file = findVirtualFile(url)
            if (file == null) {
                logger<BuiltinsVirtualFileProvider>().warn("VirtualFile not found for builtin $url")
            }
            file
        }
    }

    override fun getBuiltinVirtualFiles(): Set<VirtualFile> = builtinVirtualFilesCached
}

class BuiltinsVirtualFileProviderCliImpl : BuiltinsVirtualFileProviderBaseImpl() {
    override fun findVirtualFile(url: URL): VirtualFile? {
        val split = URLUtil.splitJarUrl(url.path)
            ?: errorWithAttachment("URL for builtins does not contain jar separator") {
                withEntry("url", url) { url.toString() }
            }
        val jarPath = split.first
        val builtinFile = split.second
        val pathToQuery = jarPath + URLUtil.JAR_SEPARATOR + builtinFile
        val jarFileSystem = VirtualFileManager.getInstance().getFileSystem(StandardFileSystems.JAR_PROTOCOL)
        return jarFileSystem.findFileByPath(pathToQuery)
    }
}