/*
 * Copyright 2010-2023 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.analysis.decompiler.js

import ksp.com.intellij.openapi.vfs.VirtualFile
import ksp.org.jetbrains.kotlin.analysis.decompiler.psi.KotlinMetadataDecompiler
import ksp.org.jetbrains.kotlin.analysis.decompiler.stub.file.KotlinMetadataStubBuilder
import ksp.org.jetbrains.kotlin.metadata.ProtoBuf
import ksp.org.jetbrains.kotlin.metadata.js.JsProtoBuf
import ksp.org.jetbrains.kotlin.psi.stubs.KotlinStubVersions
import ksp.org.jetbrains.kotlin.serialization.js.DynamicTypeDeserializer
import ksp.org.jetbrains.kotlin.serialization.js.JsSerializerProtocol
import ksp.org.jetbrains.kotlin.utils.JsMetadataVersion
import java.io.ByteArrayInputStream

class KotlinJavaScriptMetaFileDecompiler : KotlinMetadataDecompiler<JsMetadataVersion>(
    fileType = KotlinJavaScriptMetaFileType,
    serializerProtocol = { JsSerializerProtocol },
    flexibleTypeDeserializer = DynamicTypeDeserializer,
    expectedBinaryVersion = { JsMetadataVersion.INSTANCE },
    invalidBinaryVersion = { JsMetadataVersion.INVALID_VERSION },
    stubVersion = KotlinStubVersions.JS_STUB_VERSION
) {
    override fun readFile(bytes: ByteArray, file: VirtualFile): KotlinMetadataStubBuilder.FileWithMetadata {
        val stream = ByteArrayInputStream(bytes)

        val version = JsMetadataVersion.readFrom(stream)
        if (!version.isCompatibleWithCurrentCompilerVersion()) {
            return KotlinMetadataStubBuilder.FileWithMetadata.Incompatible(version)
        }

        JsProtoBuf.Header.parseDelimitedFrom(stream)

        val proto = ProtoBuf.PackageFragment.parseFrom(stream, JsSerializerProtocol.extensionRegistry)
        return KotlinMetadataStubBuilder.FileWithMetadata.Compatible(proto, version, JsSerializerProtocol)
    }
}
